#ifndef TEXTRECX_H_SENTRY
#define TEXTRECX_H_SENTRY

/*
    Keep in mind that both textrec.* and textrecx.* modules only
    implement FSMs and other info transformations in memory.
    They do NOT perform any input/output!  Hence they don't suck
    any I/O-related stuff into the linking.
 */

#include "textrec.h"

enum field_extras {
    ftypex_typemask = 0x1f,
    ftypex_multiple = 0x20,
    ftypex_mandatory = 0x40
};

enum { max_field_name_len = 28 };   /* should be 2^n - 4 */


/* WARNING the schema, which consists of textrec_field_descriptor
   elements, is supposed to be a global constant, but if it is not,
   then it is the user's duty to make sure it exists longer than
   any of the data items read using the schema.

   The len field is ignored for all types but the blob.

   At the end of the scheme, there must be a termination record,
   which has an empty name field.

   Placing at any position within the scheme (but usually right at the
   first position, just to be able to easily skip it) a record with 
   the name field consisting of exactly one space (" ") means unknown
   fields shall be silently ignored; if the scheme contains no such
   record, unknown field names will cause an error.
   
 */
struct textrecx_field_descriptor {
    unsigned char ftype;
    char name[max_field_name_len + 1];
    unsigned short len;
};


struct textrecx_data_item {
    const struct textrecx_field_descriptor *descr;
    long long i;   /* the value for bools and ints, the buffer's length
                      for blobs and strings (strlen(s) + 1) */
    uchar *buf;
    struct textrecx_data_item *next;
};


struct textrecx_context {
    const struct textrecx_field_descriptor *scheme;
    struct textrec_parser parser;
    struct textrecx_data_item *first, *last;
};


enum {
    textrecx_res_eof = textrec_res_eof,
    textrecx_res_ok = textrec_res_ok,
    textrecx_res_unknown_field = 1,
    textrecx_res_duplicate_field = 2,
    textrecx_res_mandatory_missing = 3,
    textrecx_res_incorrect_blob_size = 4
};

const char *textrecx_ctx_error_message(int code);

/* -- reading -- */

void textrecx_init_context(struct textrecx_context *ctx,
                           const struct textrecx_field_descriptor *scm);

    /* returns values of textrec_res_* and textrecx_res_* */
int textrecx_feedchar(struct textrecx_context *ctx, int c);

    /* returns boolean, data must be disposed only if true is returned */
int textrecx_get_data(struct textrecx_context *ctx,
                      struct textrecx_data_item **dataptr);

void textrecx_dispose_data(struct textrecx_data_item *p);

void textrecx_cleanup_context(struct textrecx_context *ctx);



/* -- examination -- */

const struct textrecx_field_descriptor *
textrecx_descr_by_name(const struct textrecx_field_descriptor *scm,
                       const char *name);

struct textrecx_data_item *
textrecx_data_by_name(struct textrecx_data_item *p, const char *name);

struct textrecx_data_item *
textrecx_data_by_descr(struct textrecx_data_item *p,
                       const struct textrecx_field_descriptor *descr);

const unsigned char *
textrecx_blob_by_name(struct textrecx_data_item *p, const char *name);

const char *
textrecx_string_by_name(struct textrecx_data_item *p, const char *name);

    /* returns boolean */
int textrecx_integer_by_name(struct textrecx_data_item *p, const char *name,
                             long long *n);



/* -- manipulation -- */

    /* please note *source is assigned NULL,
       *target becomes the owner for both lists
     */
void textrecx_merge_data(struct textrecx_data_item **target,
                         struct textrecx_data_item **source);


/* -- serialization -- */

    /* args: the_char, user_data; returns 0 on success, nonzero means stop */
typedef int (*textrecx_putchar_cb)(int, void *);


int textrecx_serialize_integer(long long value,
                               const char *name, int name_width,
                               textrecx_putchar_cb cb, void *ud);
int textrecx_serialize_boolean(int value,
                               const char *name, int name_width,
                               textrecx_putchar_cb cb, void *ud);
int textrecx_serialize_blob(const unsigned char *data, int len,
                            const char *name, int name_width,
                            textrecx_putchar_cb cb, void *ud);
int textrecx_serialize_string(const char *str,
                              const char *name, int name_width,
                              textrecx_putchar_cb cb, void *ud);


    /* output_name is optional; NULL and "" mean use the name from
       the descriptor;
       returns 0 on success, non-zero is the value from callback
     */
int textrecx_serialize_data_item(const struct textrecx_data_item *item,
                                 int name_width, const char *output_name,
                                 textrecx_putchar_cb cbfunc, void *ud);

    /* returns 0 on success, non-zero is the value from callback */
int textrecx_serialize_data(const struct textrecx_data_item *first,
                      int name_width, textrecx_putchar_cb cbfunc, void *ud);


#endif
