#include <stdlib.h>
#include <string.h>

#include "hexdata.h"
#include "textrec.h"

#include "textrecx.h"



enum { text_width = 79 };
    /* in the present version, only affects blob serialization */


/* --------- reading: implementation ------------ */

static int recx_type_cb(const char *field_name, void *user_data)
{
    struct textrecx_context *ctx = user_data;
    int i, tolerate;

    tolerate = 0;
    for(i = 0;; i++) {
        const struct textrecx_field_descriptor *fld = ctx->scheme + i;
        if(!fld->name[0])
            return tolerate ? ftype_ignored : ftype_unknown;
        if(fld->name[0] == ' ' && !fld->name[1]) {
            tolerate = 1;
            continue;
        }
        if(0 == strcmp(field_name, fld->name))
            return fld->ftype & ftypex_typemask;
    }
    /* we never reach here */
}

static int have_field(struct textrecx_context *ctx, int idx)
{
    struct textrecx_data_item *tmp;
    const struct textrecx_field_descriptor *fld;
    if(idx < 0)
        return 0;
    fld = ctx->scheme + idx;
    for(tmp = ctx->first; tmp; tmp = tmp->next)
        if(tmp->descr == fld)
            return 1;
    return 0;
}

static void field_lookup(struct textrecx_context *ctx, const char *name,
                         int *scheme_idx, int *ret_res)
{
    int i, have, idx, tolerate;
    const struct textrecx_field_descriptor *fld;

    tolerate = 0;
    for(i = 0;; i++) {
        fld = ctx->scheme + i;
        if(!fld->name[0]) {
            *scheme_idx = -1;
            *ret_res = tolerate ?
                textrecx_res_ok : textrecx_res_unknown_field;
            return;
        }
        if(fld->name[0] == ' ' && !fld->name[1]) {
            tolerate = 1;
            continue;
        }
        if(0 == strcmp(name, fld->name)) {
            idx = i;
            break;
        }
    }

    have = have_field(ctx, idx);
    if(have && !(ctx->scheme[idx].ftype & ftypex_multiple)) {
        *scheme_idx = -1;
        *ret_res = textrecx_res_duplicate_field;
        return;
    }

    *scheme_idx = idx;
    ret_res = textrec_res_ok;
}

static void make_data_record(struct textrecx_context *ctx, int scm_idx)
{
    struct textrecx_data_item *tmp;

    tmp = malloc(sizeof(*tmp));
    memset(tmp, 0, sizeof(*tmp));
    tmp->descr = ctx->scheme + scm_idx;

    if(ctx->first)
        ctx->last->next = tmp;
    else
        ctx->first = tmp;
    ctx->last = tmp;
}


static int
recx_blob_cb(const char *fnm, const uchar *buf, int len, void *ud)
{
    struct textrecx_context *ctx = ud;
    int idx, retres;

    field_lookup(ctx, fnm, &idx, &retres);
    if(idx == -1)
        return retres;

    if(len != ctx->scheme[idx].len)
        return textrecx_res_incorrect_blob_size;

    make_data_record(ctx, idx);
    ctx->last->i = len;
    ctx->last->buf = malloc(len);
    memcpy(ctx->last->buf, buf, len);
    return textrecx_res_ok;
}

static int recx_integer_cb(const char *fnm, long long num, void *ud)
{
    struct textrecx_context *ctx = ud;
    int idx, retres;

    field_lookup(ctx, fnm, &idx, &retres);
    if(idx == -1)
        return retres;

    make_data_record(ctx, idx);
    ctx->last->i = num;
    ctx->last->buf = NULL;
    return textrecx_res_ok;
}

static int recx_boolean_cb(const char *fnm, int b, void *ud)
{
    return recx_integer_cb(fnm, b, ud);
}

static int recx_string_cb(const char *fnm, const uchar *s, int len, void *ud)
{
    struct textrecx_context *ctx = ud;
    int idx, retres;

    field_lookup(ctx, fnm, &idx, &retres);
    if(idx == -1)
        return retres;

    make_data_record(ctx, idx);
    ctx->last->i = len + 1;
    ctx->last->buf = malloc(len + 1);
    memcpy(ctx->last->buf, s, len + 1);
    return textrecx_res_ok;
}


static void prepare_parser(struct textrecx_context *ctx)
{
    textrec_init(&ctx->parser);
    textrec_set_type_cb(&ctx->parser, recx_type_cb);
    textrec_set_blob_cb(&ctx->parser, recx_blob_cb);
    textrec_set_integer_cb(&ctx->parser, recx_integer_cb);
    textrec_set_boolean_cb(&ctx->parser, recx_boolean_cb);
    textrec_set_string_cb(&ctx->parser, recx_string_cb);
    textrec_set_user_data(&ctx->parser, ctx);
}

static int check_mandatory_fields(struct textrecx_context *ctx)
{
    int i;
    for(i = 0;; i++) {
        const struct textrecx_field_descriptor *fld = ctx->scheme + i;
        int have;
        if(!fld->name[0])
            break;
        if(!(fld->ftype & ftypex_mandatory))
            continue;
        have = have_field(ctx, i);
        if(!have)
            return 0;
    }
    return 1;
}


/* --------- reading: public functions ------------ */


void textrecx_init_context(struct textrecx_context *ctx,
                           const struct textrecx_field_descriptor *scm)
{
    ctx->scheme = scm;
    ctx->first = NULL;
    ctx->last = NULL;
    prepare_parser(ctx);
}


int textrecx_feedchar(struct textrecx_context *ctx, int c)
{
    int res;
    res = textrec_feedchar(&ctx->parser, c);
    if(res == textrec_res_eof) {
        int ok = check_mandatory_fields(ctx);
        if(!ok)
            return textrecx_res_mandatory_missing;
    }
    return res;
}


int textrecx_get_data(struct textrecx_context *ctx,
                      struct textrecx_data_item **dataptr)
{
    if(!ctx->first)
        return 0;
    *dataptr = ctx->first;
    ctx->first = NULL;
    ctx->last = NULL;
    return 1;
}


void textrecx_dispose_data(struct textrecx_data_item *p)
{
    struct textrecx_data_item *first, *tmp;
    first = p;
    while(first) {
        tmp = first;
        first = first->next;
        if(tmp->buf)
            free(tmp->buf);
        free(tmp);
    }
}


void textrecx_cleanup_context(struct textrecx_context *ctx)
{
    if(ctx->first)
        textrecx_dispose_data(ctx->first);
    textrec_cleanup(&ctx->parser);
}

/* 
enum {
    textrecx_res_eof = textrec_res_eof,
    textrecx_res_ok = textrec_res_ok,
    textrecx_res_unknown_field = 1,
    textrecx_res_duplicate_field = 2,
    textrecx_res_mandatory_missing = 3,
    textrecx_res_incorrect_blob_size = 4
};
*/

const char *textrecx_ctx_error_message(int code)
{
    switch(code) {
    case textrecx_res_unknown_field:        return "unknown field name";
    case textrecx_res_duplicate_field:      return "duplicate field";
    case textrecx_res_mandatory_missing:    return "mandatory field missing";
    case textrecx_res_incorrect_blob_size:  return "wrong blob size";
    default:
        return textrec_error_message(code);
    }
}





/* ------------- examination -------------- */

const struct textrecx_field_descriptor *
textrecx_descr_by_name(const struct textrecx_field_descriptor *scm,
                       const char *name)
{
    int i;
    for(i = 0; scm[i].name[0]; i++)
        if(0 == strcmp(scm[i].name, name))
            return scm + i;
    return NULL;
}

struct textrecx_data_item *
textrecx_data_by_name(struct textrecx_data_item *p, const char *name)
{
    struct textrecx_data_item *tmp;
    for(tmp = p; tmp; tmp = tmp->next)
        if(0 == strcmp(tmp->descr->name, name))
            return tmp;
    return NULL;
}

struct textrecx_data_item *
textrecx_data_by_descr(struct textrecx_data_item *p,
                       const struct textrecx_field_descriptor *descr)
{
    struct textrecx_data_item *tmp;
    for(tmp = p; tmp; tmp = tmp->next)
        if(tmp->descr == descr)
            return tmp;
    return NULL;
}

const unsigned char *
textrecx_blob_by_name(struct textrecx_data_item *p, const char *name)
{
    struct textrecx_data_item *q = textrecx_data_by_name(p, name);
    if(!q || ((q->descr->ftype & ftypex_typemask) != ftype_blob))
        return NULL;
    return q->buf;
}


const char *
textrecx_string_by_name(struct textrecx_data_item *p, const char *name)
{
    struct textrecx_data_item *q = textrecx_data_by_name(p, name);
    if(!q || ((q->descr->ftype & ftypex_typemask) != ftype_string))
        return NULL;
    return (const char*) q->buf;
}


int textrecx_integer_by_name(struct textrecx_data_item *p, const char *name,
                             long long *n)
{
    struct textrecx_data_item *q = textrecx_data_by_name(p, name);
    if(!q || ((q->descr->ftype & ftypex_typemask) != ftype_integer))
        return 0;
    *n = q->i;
    return 1;
}


/* ------------- manipulation -------------- */

void textrecx_merge_data(struct textrecx_data_item **target,
                         struct textrecx_data_item **source)
{
    if(*target) {
        textrecx_merge_data(&((*target)->next), source);
    } else {
        *target = *source;
        *source = NULL;
    }
}

/* ------------- serialization -------------- */


static int
serialize_puts(const char *str, textrecx_putchar_cb cb, void *ud)
{
    const char *p;
    for(p = str; *p; p++) {
        int res;
        res = cb(*p, ud);
        if(res != 0)
            return res;
    }
    return 0;
}


static int serialize_put_spaces(int n, textrecx_putchar_cb cb, void *ud)
{
    int i;
    for(i = 0; i < n; i++) {
        int res;
        res = cb(' ', ud);
        if(res != 0)
            return res;
    }
    return 0;
}

static int
serialize_fldname(const char *nm, int nw, textrecx_putchar_cb cb, void *ud)
{
    int len, spc, res;
    len = strlen(nm);
    spc = len < nw ? nw - len : 1;
    res = serialize_puts(nm, cb, ud);
    if(res != 0)
        return res;
    res = serialize_put_spaces(spc, cb, ud);
    if(res != 0)
        return res;

    return 0;
}

static int
serialize_string_field(const char *str, textrecx_putchar_cb cb, void *ud)
{
    const char *p;
    for(p = str; *p; p++) {
        int res;
        if(*p == '\n') {
            res = serialize_puts("\n+", cb, ud);
            if(res != 0)
                return res;
            continue;
        }
        res = cb(*p, ud);
        if(res != 0)
            return res;
    }
    return 0;
}


static const char *longlong2str(long long n)
{
    static char buf[24];
    char *p = buf + (sizeof(buf)-1);
    int neg;

    if(n == 0)          /* yes, that's special case */
        return "0";

    neg = n < 0;
    if(neg)
        n = -n;
    *p = 0;
    while(n > 0) {
        p--;
        *p = n % 10 + '0';
        n /= 10;
    }
    if(neg) {
        p--;
        *p = '-';
    }
    return p;
}


static int serialize_blob(const unsigned char *s, int len, int name_width,
                          textrecx_putchar_cb cb, void *ud)
{
    int res, i, bytes_per_line;

    bytes_per_line = (text_width - name_width) / 2;

    for(i = 0; i < len; i++) {
        char bbf[3];

        hexbyte2str(bbf, s[i]);
        res = serialize_puts(bbf, cb, ud);
        if(res != 0)
            return res;

        if(!((i+1) % bytes_per_line) && i < len-1) {
            res = cb('\n', ud);
            if(res != 0)
                return res;
            res = serialize_put_spaces(name_width, cb, ud);
            if(res != 0)
                return res;
        }
    }
#if 0
    res = cb('\n', ud);
    if(res != 0)
        return res;
#endif

    return 0;
}

int textrecx_serialize_integer(long long value,
                               const char *name, int name_width,
                               textrecx_putchar_cb cb, void *ud)
{
    int res;
    res = serialize_fldname(name, name_width, cb, ud);
    if(res != 0)
        return res;
    res = serialize_puts(longlong2str(value), cb, ud);
    if(res != 0)
        return res;
    res = cb('\n', ud);
    return res;
}

int textrecx_serialize_boolean(int value,
                               const char *name, int name_width,
                               textrecx_putchar_cb cb, void *ud)
{
    int res;
    res = serialize_fldname(name, name_width, cb, ud);
    if(res != 0)
        return res;
    res = serialize_puts(value ? "yes\n" : "no\n", cb, ud);
    return res;
}

int textrecx_serialize_blob(const unsigned char *data, int len,
                            const char *name, int name_width,
                            textrecx_putchar_cb cb, void *ud)
{
    int res;
    res = serialize_fldname(name, name_width, cb, ud);
    if(res != 0)
        return res;
    res = serialize_blob(data, len, name_width, cb, ud);
    if(res != 0)
        return res;
    res = cb('\n', ud);
    return res;
}

int textrecx_serialize_string(const char *str,
                              const char *name, int name_width,
                              textrecx_putchar_cb cb, void *ud)
{
    int res;
    res = serialize_fldname(name, name_width, cb, ud);
    if(res != 0)
        return res;
    res = serialize_string_field(str, cb, ud);
    if(res != 0)
        return res;
    res = cb('\n', ud);
    return res;
}


int textrecx_serialize_data_item(const struct textrecx_data_item *item,
                                 int name_width, const char *output_name,
                                 textrecx_putchar_cb cb, void *ud)
{
    int res;
    const char *visname = output_name ? output_name : item->descr->name;
    res = serialize_fldname(visname, name_width, cb, ud);
    if(res != 0)
        return res;
    switch(item->descr->ftype & ftypex_typemask) {
    case ftype_string:
        res = serialize_string_field((const char *)item->buf, cb, ud);
        break;
    case ftype_blob:
        res = serialize_blob(item->buf, item->i, name_width, cb, ud);
        break;
    case ftype_integer:
        res = serialize_puts(longlong2str(item->i), cb, ud);
        break;
    case ftype_boolean:
        res = serialize_puts(item->i ? "yes" : "no", cb, ud);
        break;
    case ftype_unknown:
    case ftype_ignored:
    default:
        res = serialize_puts("*ERROR*UNKNOWN*TYPE*", cb, ud);
    } 
    res = cb('\n', ud);
    if(res != 0)
        return res;
    return 0;
}

int textrecx_serialize_data(const struct textrecx_data_item *first,
                            int name_width,
                            textrecx_putchar_cb cb, void *ud)
{
    const struct textrecx_data_item *tmp;
    for(tmp = first; tmp; tmp = tmp->next) {
        int res = textrecx_serialize_data_item(tmp, name_width, NULL, cb, ud);
        if(res != 0)
            return res;
    }
    return 0;
}







/* --------------------------------------------------------------- */
/* ------ TESTS -------------------------------------------------- */
/* --------------------------------------------------------------- */

#ifdef TEXTRECX_TEST

#include <stdio.h>

#include "cryptodf.h"
#include "fk_crypt.h"

const char teststr[] =
"node_id     72d5c132de46e81193bf\n"
"point       77\n"
"minpref     9\n"
"signed_by   -1\n"
"master_pub  09812c702f7c99bad1b19e3ca507ea372b77be3e69c372d5c132de46e81193bf\n"
"master_hash fef9e76d5e0df05b8fba491502afe19148a68998486f2ee5d6d04d6cf570c981\n"
"certbody    fedace8772d5c132de46e81193bf4dff01b7980ef2f8f64ec3c48b7ed4169856\n"
"            afac6d4859ed0bffda41e079c4bdbb45a1070452\n"
"signature   fb78cdd5ff7db75af6f09454e30ac6109c115758914e3f54ad19a3ee4c17555e\n"
"            762fc97e45bd9c772272456bff8bda6714ef617f1611e9d8eaf40ebd15919305\n"
"public      f2f8f64ec3c48b7ed4169856afac6d4859ed0bffda41e079c4bdbb45a1070452\n"
"kex_public  3fe975d9ac5e4b804c4c6dd2a1747250a9dc9dc61e6629bf4aa2ac1b88652543\n"
"kex_cert    01ba46d53fe975d9ac5e4b804c4c6dd2a1747250a9dc9dc61e6629bf4aa2ac1b\n"
"            88652543\n"
"kex_signature "
            "e2758195a17c71150e69abf49f3aa01568b48d5480506055cf42ed2e74047626\n"
"            b63a749c0e0b6c9051b1083946c88f7d915c0738e9ab15f828d7c72f9b03c80c\n"
"kex_timestamp 28985045\n"
"long_comment well this is a comment and it is intentionally made long so\n"
"     we can demonstrate multiline comments you know that\n"
"+       and even with leading space\n";




static struct textrecx_field_descriptor the_scheme[] = {
    { ftype_string,  "node_id"                           },
    { ftype_integer, "point"                             },
    { ftype_integer, "minpref"                           },
    { ftype_integer, "signed_by"                         },
    { ftype_blob,    "master_pub",    public_key_size    },
    { ftype_blob,    "master_hash",   yespower_hash_size },
    { ftype_blob,    "certbody",      feda_cert_size     },
    { ftype_blob,    "signature",     signature_size     },
    { ftype_blob,    "public",        public_key_size    },
    { ftype_blob,    "kex_public",    kex_public_size    },
    { ftype_blob,    "kex_cert",      feda_kex_info_size },
    { ftype_blob,    "kex_signature", signature_size     },
    { ftype_integer, "kex_timestamp"                     },
    { ftype_string,  "long_comment"                      },
    { ftype_unknown, "" }
};

static int putchar_cb(int c, void *data)
{
    putchar(c);
    return 0;
}


int main()
{
    int res;
    const char *p;
    struct textrecx_context ctx;
#if 1
    struct textrecx_data_item *tmp;
#endif

    textrecx_init_context(&ctx, the_scheme);
    for(p = teststr; *p; p++) {
        res = textrecx_feedchar(&ctx, *p);
        if(res != textrecx_res_ok) {
            fprintf(stderr, "early quit (%d)\n", res);
            break;
        }
    }
    if(res == textrecx_res_ok) {
        res = textrecx_feedchar(&ctx, EOF);
        if(res != textrecx_res_eof)
            fprintf(stderr, "feeding EOF didn't lead to eof (%d)\n", res);
    }

#if 1
    for(tmp = ctx.first; tmp; tmp = tmp->next) {
        printf("%s: %lld %s\n", tmp->descr->name, tmp->i, tmp->buf ? "*" : "");
    }
    printf("\n\n");
#endif
    textrecx_serialize_data(ctx.first, 15, putchar_cb, NULL);

    return 0;
}


#if 0
int main()
{
    int i;
    printf("%d\n", (int) sizeof(struct textrecx_field_descriptor));
    for(i = 0; the_scheme[i].name[0]; i++)
        printf("%s\n", the_scheme[i].name);

    return 0;
}
#endif

#endif
