#ifndef TEXTREC_H_SENTRY
#define TEXTREC_H_SENTRY

/*
    It is very important to remember: this module only implements FSMs.
    It does NOT do any input/output!
 */


enum field_types {
    ftype_unknown = 0,
    ftype_ignored,   /* field accepted but not passed anywhere */
    ftype_string,
    ftype_blob,
    ftype_integer,
    ftype_boolean
        /* WARNING it is assumed in the textrecx module that
           these never exceed 0x1f (a.k.a. 31)
         */
};

typedef unsigned char uchar;

/*
    Function that gets a field name as a string and returns field type.
    Profile like this:
        int type_func(const char *field_name, void *user_data);
 */
typedef int (*field_type_cb)(const char *, void *);

/*
    Functions to call for the respective field type.
    Zero return value means the parser should continue.  On non-zero
    values, parser changes to the stop state, using the value as its
    result code.  This value should be positive, as negatives are used
    for errors detected by the parser itself.
 */

    /* args:   field_name, string, string_length, user_data */
typedef int (*field_string_cb)(const char *, const uchar *, int, void*);

    /* args:   field_name, blob, blob_size, user_data */
typedef int (*field_blob_cb)(const char *, const uchar *, int, void*);

    /* args:   field_name, value, user_data */
typedef int (*field_integer_cb)(const char *, long long, void*);

    /* args:   field_name, truth_value, user_data */
typedef int (*field_boolean_cb)(const char *, int, void*);

enum { textrec_state_finish = 0 };

enum {
    textrec_res_ok = 0,
    textrec_res_eof = -1, /* this is okay, too, but you shouldn't continue */
    textrec_res_no_type_cb = -2,
    textrec_res_unknown_type = -3,
    textrec_res_no_string_cb = -4,
    textrec_res_no_blob_cb = -5,
    textrec_res_no_integer_cb = -6,
    textrec_res_no_boolean_cb = -7,
    textrec_res_unfinished_line = -10,
    textrec_res_no_line_to_continue = -11,
    textrec_res_incorrect_blob_data = -12,
    textrec_res_incorrect_integer = -13,
    textrec_res_incorrect_boolean = -14,
    textrec_res_no_memory = -20,
    textrec_res_internal_error = -30
};

struct textrec_parser {
    int state, res_code, line, curtype;
    int empty_line_code;
    field_type_cb    type_cb;
    field_string_cb  string_cb;
    field_blob_cb    blob_cb;
    field_integer_cb integer_cb;
    field_boolean_cb boolean_cb;
    void *user_data;

    char *namebuf;
    int namebufsize, namecurlen;
    uchar *databuf;
    int databufsize, datacurlen;
};

void textrec_init(struct textrec_parser *ps);
void textrec_cleanup(struct textrec_parser *ps);
void textrec_set_type_cb(struct textrec_parser *ps, field_type_cb cb);
void textrec_set_string_cb(struct textrec_parser *ps, field_string_cb cb);
void textrec_set_blob_cb(struct textrec_parser *ps, field_blob_cb cb);
void textrec_set_integer_cb(struct textrec_parser *ps, field_integer_cb cb);
void textrec_set_boolean_cb(struct textrec_parser *ps, field_boolean_cb cb);
void textrec_set_user_data(struct textrec_parser *ps, void *ud);


/*
    0 a.k.a. textrec_res_ok means the caller should continue,
    -1 a.k.a. textrec_res_eof means EOF was encountered after a line end
      (trailing space is tolerated), other negatives correspond to errors
      (there must be a textrec_res_* constant for it),
    positives are for the user errors (returned from callbacks)
 */
int textrec_feedchar(struct textrec_parser *ps, int c);

const char *textrec_error_message(int code);

#endif
