#ifndef SERVLOG_H_SENTRY
#define SERVLOG_H_SENTRY

#include <stdarg.h>

/*
    Three channels are supported, configured independently:

    1) logging to stderr
    2) logging to a given file, in append mode, fflush'ed after each msg
    3) logging through the system logger

    Each channel is configured separately.  The functions performing
    actual logging use all configured channels, so it is possible, e.g.,
    to send all the information to all the three channels at once.

    By default, only the stderr channel is enabled, with the normal
    verbosity level and private information logging allowed.  The other
    two channels are disabled.
 */

enum servlog_levels {
    srvl_disable =  0,    /* setup funcs only: disable the channel */

    srvl_alert   =  1,    /* show even in the "quiet" mode         */
    srvl_normal  =  2,    /* normal messages                       */
    srvl_info    =  3,    /* show in the "verbose" mode only       */
    srvl_debug   =  4,    /* show only for -vv and higher          */
    srvl_debug2  =  5,    /* show only for -vvv and higher,
                                never send to syslog               */

    srvl_private = 256    /* contains privacy-sensitive info, only
                          log if sensitive info logging is allowed */
};

enum servlog_facilities {
    srvl_facil_default = 0,
    srvl_facil_user    = 0,
    srvl_facil_daemon  = 1,
    srvl_facil_local0  = 2,
    srvl_facil_local1  = 3,
    srvl_facil_local2  = 4,
    srvl_facil_local3  = 5,
    srvl_facil_local4  = 6,
    srvl_facil_local5  = 7,
    srvl_facil_local6  = 8,
    srvl_facil_local7  = 9
};

void servlog_message_vl(int level, const char *fmt, va_list args);
void servlog_message(int level, const char *fmt, ...);

    /* s1 (sic!) may be NULL or empty */
void servlog_perror(int level, const char *s1, const char *s2);

void setup_stderr_log(int mode);

    /* returns boolean */
int setup_file_log(int mode, const char *filename);

void setup_syslog(int mode, const char *ident, int facility);


    /* function should look like

          void func(void *userdata, const char *message) { ... }

       extra_log_id should be considered opaque, but it is safe to assume
       it is a pointer so it can be used as a truth value and also can
       be assigned the NULL value
     */

typedef void (*extra_log_callback)(void *, const char *);
typedef void *extra_log_id;

extra_log_id add_extra_log(extra_log_callback f, void *usrdata, int level);
int change_extra_log(extra_log_id id, int level);
int remove_extra_log(extra_log_id id);


#endif
