#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <monocypher/monocypher.h>


#include "cryptodf.h"
/* #include "xyespwr.h" -- DON'T include this again here! */
#include "fedapref.h"
#include "fileutil.h"
#include "kfiles.h"
#include "kfschems.h"
#include "keyutils.h"
#include "mastrkey.h"

#include "knowndb.h"


/* *******************************************************************

For, e.g., node 72d5c132de46e81193bf:

($WORKDIR stands for $HOME/.fedanet/keys or whatever is set with -c)


$WORKDIR/72d/5c1/72d5c132de46e81193bf/              is the node directory


$WORKDIR/72d/5c1/72d5c132de46e81193bf/node          "immutable" node info
          * node_id
          * rank
          * master_pub
          * master_hash
          * master_hsign
          -- * minpref --


$WORKDIR/72d/5c1/72d5c132de46e81193bf/info          "changeable" node info
          * minpref





$WORKDIR/72d/5c1/72d5c132de46e81193bf/0             actual point information
$WORKDIR/72d/5c1/72d5c132de46e81193bf/1
$WORKDIR/72d/5c1/72d5c132de46e81193bf/47
$WORKDIR/72d/5c1/72d5c132de46e81193bf/253
          * node_id
          * point
          * timestamp
          * signed_by
          * public
          * certbody
          * signature


$WORKDIR/72d/5c1/72d5c132de46e81193bf/47.29115286   old point information


$WORKDIR/72d/5c1/72d5c132de46e81193bf/47.kex        point key exchange pubkey
          * kex_public
          * kex_cert
          * kex_signature
          * kex_timestamp
          * ptk_timestamp


NB: old key exchange keys are not stored in any way, it is senseless as
they are only needed for encrypting messages/files intended for the
particular party, and for this purpose, only the actual (latest known) key
must be used.

   ******************************************************************* */




struct known_nodes_db {
    const char *dir;     /* not owned! we assume it exists elsewhere */
    int minpref;
};


struct known_nodes_db *make_kndb(const char *dir, int minpref)
{
    struct known_nodes_db *p;

    p = malloc(sizeof(*p));
    p->dir = dir;
    p->minpref = minpref;
    return p;
}


void dispose_kndb(struct known_nodes_db *p)
{
    free(p);
}


static int kndb_error_by_kfiles_error(int code)
{  
    switch(code) {
    case kfiles_res_ok:               return kndb_res_success;
    case kfiles_res_file_not_found:   return kndb_res_file_not_found;
    case kfiles_res_cant_open_file:   return kndb_res_file_error;
    case kfiles_res_unexpected_thing: return kndb_res_unexpected_thing;
    default:                          return kndb_res_format_error;
    }
}

int kndb_get_node(struct known_nodes_db *p, const unsigned char *node_id,
                  int *rank, unsigned char *pubkey, unsigned char *hash,
                  unsigned char *hsign)
{
    int r, errl;
    struct kc_path_set ps;
    struct textrecx_data_item *data = NULL;
    long long lln;
    const unsigned char *blob_pub;
    const unsigned char *blob_hash;
    const unsigned char *blob_hsign;

    make_path_set(p->dir, node_id, -1, &ps);

    r = read_textrecx_file(ps.node_file, schema_known_node, &data, &errl);
    if(r == kfiles_res_file_not_found) {
        r = kndb_res_node_unknown;
        goto quit;
    }
    if(r != kfiles_res_ok) {
        r = kndb_error_by_kfiles_error(r);
        goto quit;
    }
    r = textrecx_integer_by_name(data, "rank", &lln);
    blob_pub =   textrecx_blob_by_name(data, "master_pub");
    blob_hash =  textrecx_blob_by_name(data, "master_hash");
    blob_hsign = textrecx_blob_by_name(data, "master_hsign");
    if(!r || !blob_pub || !blob_hash || !blob_hsign) {
        r = kndb_res_unexpected_thing;
        goto quit;
    }
    if(rank)
        *rank = lln;
    if(pubkey)
        memcpy(pubkey, blob_pub, public_key_size);
    if(hash)
        memcpy(pubkey, blob_hash, public_key_size);
    if(hsign)
        memcpy(pubkey, blob_hsign, public_key_size);
    r = kndb_res_success;

quit:
    if(data)
        textrecx_dispose_data(data);
    dispose_path_set(&ps);
    return r;
}


int kndb_save_node(struct known_nodes_db *p, const unsigned char *node_id,
                   const unsigned char *master_pub, int rank,
                   const unsigned char *master_hash,
                   const unsigned char *master_hsign)
{
    int res;
    struct kc_path_set ps;
    FILE *f;

    make_path_set(p->dir, node_id, -1, &ps);
    res = make_directory_path(ps.node_dir, 0);
    if(res == -1) {
        res = kndb_res_file_error;
        goto quit;
    }
    f = fopen(ps.node_file, "w");
    if(!f) {
        res = kndb_res_file_error;
        goto quit;
    }

    textrecx_serialize_blob(node_id, node_id_size, "node_id",
                            fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_integer(rank, "rank",
                            fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(master_pub, public_key_size, "master_pub",
                            fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(master_hash, yespower_hash_size, "master_hash",
                            fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(master_hsign, signature_size, "master_hsign",
                            fieldname_width, putchar_cb_fputc, f);

    /* we don't write minpref here -- it might change, so it doesn't belong
       to the node file (which is, once saved, should never ever change */

    fclose(f);

    res = kndb_res_success;
quit:
    dispose_path_set(&ps);
    return res;
}

int kndb_check_save_node(struct known_nodes_db *kndb,
                         const unsigned char *node_id,
                         const unsigned char *master_pub,
                         const unsigned char *master_hash,
                         const unsigned char *master_hsign,
                         const unsigned char *computed_hash)
{
    int rank;

    rank = check_master_pub(node_id, master_pub, master_hash, master_hsign, 0);
    switch(rank) {
    case checkmasterpub_hash_fail:       return kndb_res_unexpected_thing;
    case checkmasterpub_wrong_id:        return kndb_res_wrong_node_key;
    case checkmasterpub_wrong_hash:      return kndb_res_node_key_differs;
    case checkmasterpub_wrong_hash_sign: return kndb_res_signature_fail;
    }

    if(0 != memcmp(master_hash, computed_hash, yespower_hash_size))
        return kndb_res_node_key_differs;

    if(rank < kndb->minpref)
        return kndb_res_node_rank_low;

    return kndb_save_node(kndb, node_id, master_pub, rank,
                          master_hash, master_hsign);
}

int kndb_get_minpref(struct known_nodes_db *p, const unsigned char *node_id,
                     int *minpref)
{
    int r, erl;
    struct kc_path_set ps;
    struct textrecx_data_item *data = NULL;
    long long lln;

    make_path_set(p->dir, node_id, -1, &ps);

    r = read_textrecx_file(ps.info_file, schema_known_node_info, &data, &erl);
    if(r == kfiles_res_file_not_found) {
        r = kndb_res_node_unknown;
        goto quit;
    }
    if(r != kfiles_res_ok) {
        r = kndb_error_by_kfiles_error(r);
        goto quit;
    }

    r = textrecx_integer_by_name(data, "minpref", &lln);
    if(!r) {
        r = kndb_res_format_error;
        goto quit;
    }

    *minpref = lln;
    r = kndb_res_success;

quit:
    if(data)
        textrecx_dispose_data(data);
    dispose_path_set(&ps);
    return r;
}

void kndb_set_minpref(struct known_nodes_db *p, const unsigned char *node_id,
                      int minpref)
{
    static const struct textrecx_field_descriptor minpref_descr =
        { ftype_integer, "minpref" };
    int r, erl;
    struct kc_path_set ps;
    struct textrecx_data_item *data = NULL;
    struct textrecx_data_item *the_item = NULL;

    make_path_set(p->dir, node_id, -1, &ps);

    r = read_textrecx_file(ps.info_file, schema_known_node_info, &data, &erl);
    if(r == kfiles_res_ok)
        the_item = textrecx_data_by_name(data, "minpref");

    if(the_item) {
        the_item->i = minpref;
    } else {
        the_item = malloc(sizeof(*the_item));
        the_item->descr = &minpref_descr;
        the_item->i = minpref;
        the_item->buf = NULL;
        the_item->next = data;
        data = the_item;
    }

    write_textrecx_file(ps.info_file, data);
    textrecx_dispose_data(data);
    dispose_path_set(&ps);
}

int kndb_get_point_pubkey(struct known_nodes_db *p,
                          const unsigned char *node_id, int point,
                          unsigned char *pubkey, int *timestamp)
{
    int r, errl;
    struct kc_path_set ps;
    struct textrecx_data_item *data = NULL;
    const unsigned char *blob;
    long long lln;

    make_path_set(p->dir, node_id, point, &ps);

    r = read_textrecx_file(ps.point_file, schema_known_point, &data, &errl);
    if(r == kfiles_res_file_not_found) {
        r = kndb_res_point_unknown;
        goto quit;
    }
    if(r != kfiles_res_ok) {
        r = kndb_error_by_kfiles_error(r);
        goto quit;
    }
    blob = textrecx_blob_by_name(data, "public");
    r = textrecx_integer_by_name(data, "timestamp", &lln);
    if(!r || !blob) {
        r = kndb_res_unexpected_thing;
        goto quit;
    }
    memcpy(pubkey, blob, public_key_size);
    if(timestamp)
        *timestamp = lln;
    r = kndb_res_success;

quit:
    if(data)
        textrecx_dispose_data(data);
    dispose_path_set(&ps);
    return r;
}




int kndb_save_point(struct known_nodes_db *p,
                    const struct feda_cert_info *cert,
                    const unsigned char *signature)
{
    int r;
    struct kc_path_set ps;
    FILE *f;

    make_path_set(p->dir, cert->node_id, cert->point_id, &ps);

    r = make_directory_path(ps.node_dir, 0);
    if(r == -1) {
        r = kndb_res_file_error;
        goto quit;
    }

    f = fopen(ps.point_file, "w");
    if(!f) {
        r = kndb_res_file_error;
        goto quit;
    }
    textrecx_serialize_blob(cert->node_id, node_id_size, "node_id",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_integer(cert->point_id, "point",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_integer(get_cert_timestamp(cert), "timestamp",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_integer(cert->signer_id, "signed_by",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(cert->key, public_key_size, "public",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob((const unsigned char *)cert,
                               feda_cert_size, "certbody",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(signature, signature_size, "signature",
                               fieldname_width, putchar_cb_fputc, f);
    fclose(f);
    r = kndb_res_success;
quit:
    dispose_path_set(&ps);
    return r;
}



int kndb_did_we_know(struct known_nodes_db *p,
                     const unsigned char *node_id, int point,
                     const unsigned char *zp_certbody,
                     const unsigned char *zp_signature,
                     const unsigned char *point_certbody,
                     const unsigned char *point_signature,
                     unsigned char *pubkey)
{
    int r;
    unsigned char data_master_pub[public_key_size];
    unsigned char data_public[public_key_size];
    const unsigned char *check_key;
    const struct feda_cert_info *zpcert = NULL, *ptcert;
    int signed_by;
    int data_rank, known_pt_tstamp, new_pt_tstamp;
    int update_zp = 0;

    r = kndb_get_node(p, node_id, &data_rank, data_master_pub, NULL, NULL);
    if(r != kndb_res_success)
        return r;

        /* having master_pub we can check, but maybe we know it already? */

    ptcert = (const struct feda_cert_info *)point_certbody;

    r = kndb_get_point_pubkey(p, node_id, point,
                              data_public, &known_pt_tstamp);
    if(r == kndb_res_success &&
        0 == memcmp(data_public, ptcert->key, public_key_size)
    ) {         /* we know exactly the same key for the point in question */
        if(pubkey)
            memcpy(pubkey, data_public, public_key_size);
        return kndb_res_success;
    }

        /* okay, let's go down the long road */

    signed_by = ptcert->signer_id == 0 ? 0 : -1;
    if(point != 0 && signed_by == 0) {
        if(!zp_certbody || !zp_signature)  /* a bit of paranoia */
            return kndb_res_signature_fail;
        zpcert = (const struct feda_cert_info *)zp_certbody;
        if(!compare_feda_cert(zpcert, node_id, 0, -1, -1, NULL))
            return kndb_res_signature_fail;
        r = crypto_eddsa_check(zp_signature, data_master_pub,
                               zp_certbody, feda_cert_size);
        if(r != 0 /* sic */)
            return kndb_res_signature_fail;
        check_key = zpcert->key;  /* this is inside zp_certbody */
    } else {
        check_key = data_master_pub;
    }

    if(!compare_feda_cert(ptcert, node_id, point, signed_by, -1, NULL))
        return kndb_res_signature_fail;

    r = crypto_eddsa_check(point_signature, check_key,
                           point_certbody, feda_cert_size);
    if(r != 0 /* sic */)
        return kndb_res_signature_fail;

        /* all signatures look good, but isn't anything outdated? */

    if(point != 0 && signed_by == 0) {  /* first, check the zero point key */
        int known_zp_tstamp;
        unsigned char known_zp_key[public_key_size];

        r = kndb_get_point_pubkey(p, node_id, 0,
                                  known_zp_key, &known_zp_tstamp);
        if(r == kndb_res_success) {
            if(0 != memcmp(known_zp_key, zpcert->key, public_key_size)) {
                int new_zp_timestamp = get_cert_timestamp(zpcert);
                if(new_zp_timestamp < known_zp_tstamp)
                    return kndb_res_zpcert_outdated;
                update_zp = new_zp_timestamp > known_zp_tstamp;
            }
        } else {
            /* if we couldn't get the ZP cert, update as well */
            update_zp = 1;
        }
    }
        /* now check the point key */
    new_pt_tstamp = get_cert_timestamp(ptcert);
        /* note we already know the keys are different */
    if(new_pt_tstamp < known_pt_tstamp)
        return kndb_res_zpcert_outdated;

    if(update_zp) {
        r = kndb_save_point(p, zpcert, zp_signature);
        if(r != kndb_res_success)
            return r;
    }
    r = kndb_save_point(p, ptcert, point_signature);
    if(r != kndb_res_success)
        return r;

    if(pubkey)
        memcpy(pubkey, ptcert->key, public_key_size);
    return kndb_res_success;
}


int kndb_consider_node(struct known_nodes_db *p,
                       const unsigned char *node_id,
                       const unsigned char *node_master_pub)
{
    int r, rank;
    unsigned char known_master_pub[public_key_size];

    if(0 != memcmp(node_id, node_master_pub + (public_key_size-node_id_size),
                   node_id_size)
    ) {
        return kndb_res_wrong_node_key;
    }

        /* do we have it? */
    r = kndb_get_node(p, node_id, &rank, known_master_pub, NULL, NULL);
    if(r == kndb_res_success) { /* yes we do */
        if(0 == memcmp(known_master_pub, node_master_pub, public_key_size))
            return kndb_res_success;
        else
            return kndb_res_node_key_differs;
    }

    /* if we're here, it means get_known_cert didn't return anything
       useful, so there's nothing to dispose yet; the fact is that
       we don't know that node (yet) */

    return kndb_res_node_unknown; /* we're not allowed to take the hash */
}


const char *kndb_result_message(int code)
{
    switch(code) {
    case kndb_res_success:          return "success";
    case kndb_res_node_unknown:     return "node unknown";
    case kndb_res_node_key_differs: return "node key differs";
    case kndb_res_wrong_node_key:   return "wrong node key";
    case kndb_res_node_rank_low:    return "node rank low";
    case kndb_res_point_unknown:    return "point unknown";
    case kndb_res_cert_outdated:    return "cert outdated";
    case kndb_res_zpcert_outdated:  return "ZP cert outdated";
    case kndb_res_signature_fail:   return "signature check failed";
    case kndb_res_file_not_found:   return "file not found";
    case kndb_res_file_error:       return "file error";
    case kndb_res_format_error:     return "format error";
    case kndb_res_unexpected_thing: return "unexpected thing (bug?)";
    case kndb_res_not_implemented:  return "not implemented";
    default:
        return "error code unknown";
    }
}


