#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

/* #include "message.h"  // no way! */
#include "cryptodf.h"
#include "hexdata.h"
#include "fileutil.h"
#include "kfiles.h"


/* file names within the keys directory (e.g., ~/.fedanet/keys) */

#define FILENAME_MASTERKEY "master.key"
#define FILENAME_SECRETKEY "secret.key"
#define FILENAME_ZEROPOINTKEY "zeropoint.key"
#define FILENAME_CONFIG "feda.conf"
#define FILENAME_MSGKEY_DEFAULT "kex"
#define FILENAME_MSGKEY_PREFIX "kex_"
#define FILENAME_MSGKEY_SUFFIX ".key"

char *make_masterkey_fname(const char *wdir)
{
    return concat_path(wdir, FILENAME_MASTERKEY);
}

char *make_secretkey_fname(const char *wdir)
{
    return concat_path(wdir, FILENAME_SECRETKEY);
}

char *make_zeropointkey_fname(const char *wdir)
{
    return concat_path(wdir, FILENAME_ZEROPOINTKEY);
}

char *make_config_fname(const char *wdir)
{
    return concat_path(wdir, FILENAME_CONFIG);
}

char *make_msgkey_fname(const char *wdir, const char *key_id)
{
    static char buf[32]; /* 4 for the prefix, 10 for key_id, 4 for the
                            extenstion, the rest is, well, just in case */
    const char *fn;
    if(key_id && *key_id) {
        sprintf(buf, "%s%.10s%s", FILENAME_MSGKEY_PREFIX, key_id,
                                  FILENAME_MSGKEY_SUFFIX);
        fn = buf;
    } else {
        fn = FILENAME_MSGKEY_DEFAULT FILENAME_MSGKEY_SUFFIX;
    }
    return concat_path(wdir, fn);
}

void dispose_fname(char *p)
{
    dispose_path(p);
}

void make_path_set(const char *wdir, const unsigned char node_id[], int point,
                   struct kc_path_set *s)
{
    char nodename[node_id_size * 2 + 1];
    char nodepath[8 + node_id_size * 2 + 1];  /* 8 is for abc/def/  */

    hexdata2str(nodename, node_id, node_id_size);

    nodepath[0] = nodename[0];     /* yes, I know :-) */
    nodepath[1] = nodename[1];
    nodepath[2] = nodename[2];
    nodepath[3] = '/';
    nodepath[4] = nodename[3];
    nodepath[5] = nodename[4];
    nodepath[6] = nodename[5];
    nodepath[7] = '/';
    strcpy(nodepath + 8, nodename);

    s->node_dir = concat_path(wdir, nodepath);
    s->node_file = concat_path(s->node_dir, "node");
    s->info_file = concat_path(s->node_dir, "info");
    s->zero_point_file = point == 0 ? NULL : concat_path(s->node_dir, "0");
    if(point >= 0 && point <= max_point_number) {
        char pointname[16];
        sprintf(pointname, "%d", point);
        s->point_file = concat_path(s->node_dir, pointname);
        strcat(pointname, ".kex");
        s->point_kex_file = concat_path(s->node_dir, pointname);
    } else {
        s->point_file = NULL;
        s->point_kex_file = NULL;
    }
}


    /* the result is malloc'ed and it is the caller's duty to free it */
char *make_old_point_fname(const char *fname, int timestamp)
{
    int len, sz;
    char *res;
    len = strlen(fname);
    sz = len + 14; /* 12 for the number, one for '.' and one for term.zero */

    res = malloc(sz);  /* term.zero accounted already */
    strcpy(res, fname);
    sprintf(res + len, ".%d", timestamp);
    return res;
}


void dispose_path_set(struct kc_path_set *s)
{
    free(s->node_dir);
    free(s->node_file);
    free(s->info_file);
    if(s->zero_point_file)
        free(s->zero_point_file);
    if(s->point_file)
        free(s->point_file);
    if(s->point_kex_file)
        free(s->point_kex_file);
}




#if 0
static void message_parsctx_abort(const char *fname, int line, int code)
{
    message(mlv_normal, "%s:%d: parse error [%s]\n", fname, line,
                        textrecx_ctx_error_message(code));
}
#endif


const char *kfiles_ctx_error_message(int code)
{
    switch(code) {
    case kfiles_res_file_not_found: return "file not found";
    case kfiles_res_cant_open_file: return "can't open file";
    case kfiles_res_unexpected_thing: return "unexpected situation (bug)";
    default:
        return textrecx_ctx_error_message(code);
    }
}



int read_textrecx_file(const char *filename,
                       const struct textrecx_field_descriptor *schema,
                       struct textrecx_data_item **data,
                       int *errline)
{
    FILE *f;
    int c, res;
    struct textrecx_context ctx;

    if(errline)
        *errline = -1;

    *data = NULL;

    f = fopen(filename, "r");
    if(!f)
        return errno == ENOENT ?
            kfiles_res_file_not_found : kfiles_res_cant_open_file;

    textrecx_init_context(&ctx, schema);
    while((c = fgetc(f)) != EOF) {
        res = textrecx_feedchar(&ctx, c);
        if(res != textrecx_res_ok)
            break;
    }
    fclose(f);
    if(c == EOF)
        res = textrecx_feedchar(&ctx, EOF);
    if(res != textrecx_res_ok && res != textrecx_res_eof) {
        if(errline)
            *errline = ctx.parser.line;
        goto quit;
    }

    res = textrecx_get_data(&ctx, data);
    if(!res) {
        res = kfiles_res_unexpected_thing;
        goto quit;
    }

    res = kfiles_res_ok;
quit:
    textrecx_cleanup_context(&ctx);
    return res;
}


/* ------- read_single_blob --------------------------------------- */

int read_single_blob(const char *filename, const char *blobname,
                     unsigned char *buf, int buflen, int *errline)
{
    static struct textrecx_field_descriptor scm[] = {
        { ftype_unknown, " ", 0 },
        { ftype_blob,    "",  0 },
        { ftype_unknown, "",  0 }
    };
    static const int nmsize = sizeof(scm[1].name);
    int res;
    struct textrecx_data_item *data;
    const unsigned char *bp;

    strncpy(scm[1].name, blobname, nmsize - 1);
    scm[1].name[nmsize-1] = 0;
    scm[1].len = buflen;

    res = read_textrecx_file(filename, scm, &data, errline);
    if(res != kfiles_res_ok)
        return res;  /* no need to dispose the data, see the
                        read_textrecx_file function */

    /* okay, now we own some data */

    bp = textrecx_blob_by_name(data, blobname);
    if(!bp) {
        textrecx_dispose_data(data);
        return kfiles_res_unexpected_thing;
    }
    memcpy(buf, bp, buflen);
    textrecx_dispose_data(data);
    return kfiles_res_ok;
}

/* ------------------------------------------------------------ */



int putchar_cb_fputc(int c, void *fstream)
{
    FILE *f = fstream;
    fputc(c, f);
    return textrecx_res_ok;
}



int write_textrecx_file(const char *fname, struct textrecx_data_item *data)
{
    int res;
    FILE *f;

    f = fopen(fname, "w");
    if(!f)
        return kfiles_res_cant_open_file;

    res = textrecx_serialize_data(data, fieldname_width,
                                  putchar_cb_fputc, f);
    fclose(f);
    return res;
}

