#include <time.h>
#include <sys/types.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>

#include "keyutils.h"


int get_random(void *buf, int len)
{
    int fd, rc, cnt;
    fd = open("/dev/urandom", O_RDONLY);
    if(fd == -1)
        return 0;
    cnt = 0;
    while(cnt < len) {
        rc = read(fd, ((char*)buf) + cnt, len - cnt);
        if(rc < 1) {
                /* -1 is error, but, well, EOF on /dev/urandom is even worse */
            close(fd);
            return 0;
        }
        cnt += rc;
    }
    close(fd);
    return 1;
}


int all_zeroes(const unsigned char *buf, int len)
{
    int i;
    int z = 0, nz = 0;
    for(i = 0; i < len; i++)
        if(buf[i])
            nz++;
        else
            z++;

    /* please note we intentionally don't break the loop; otherwise
       it would make a timing attack, well, not really possible but
       possible to the extent cryptography folks will cry loudly;
       the ``nz < len'' below doesn't change anything but the compiler
       will issue a warning without it
     */
    return z == len && nz < len;
}

void place_timemark(int tm, unsigned char p[4])
{
    p[0] = (tm >> 24) & 0xff;
    p[1] = (tm >> 16) & 0xff;
    p[2] = (tm >> 8) & 0xff;
    p[3] = tm & 0xff;
}

void increment_buf(unsigned char *buf, int len)
{
    int i = 0;
    (*buf)++;
    while(buf[i] == 0 && i < len-1) {
        i++;
        buf[i]++;
    }
}


/* The function get_rand_longlong is based on the code of the library
   named ``randq'' by Alessandro Cudazzor.  The library comes under the
   MIT license, which is permissive.  I'm not sure if the code of my
   function may qualify as containing ``substantial portion'' of the
   original library, as actually it even doesn't look like the original
   code; only the variable names u, v, w and x, together with the magic
   numbers (which the library's author claims to come from the well-known
   "Numerical recipes" book) and the sequence of operations is still
   the same as there, but all the text is completely different.

   As it is the particular text, not an algorithm, not numbers, but the
   text is THE thing subject to copyright, I don't include the original
   copyright notice here.  If you're in doubt, please take a look at the
   original library and see if my text has much in common with that.
 */

#define RAND_STEP \
    u = u * 2862933555777941757LL + 7046029254386353087LL; \
    v ^= v >> 17; \
    v ^= v << 31; \
    v ^= v >> 8;  \
    w = 4294957665U*(w & 0xffffffff) + (w >> 32);

static unsigned long long get_rand_longlong()
{
    static int seeded = 0;
    static unsigned long long u, v, w;
    unsigned long long x;

    if(!seeded) {
        unsigned long long start;
        int res;
        res = get_random(&start, sizeof(start));
        if(!res)  /* fallback for paranoia; we don't want to fail here */
            start = time(NULL) ^ (getpid() << 16);

        v = 4101842887655102017LL;
        w = 1;
        u = start ^ v; 
        RAND_STEP
        v = u; 
        RAND_STEP
        w = v; 
        /* RAND_STEP */
    }

    RAND_STEP

    x = u ^ (u << 21);                   
    x ^= x >> 35;
    x ^= x << 4;
    return (x + v) ^ w;
}

#undef RAND_STEP


void fill_noise(unsigned char *mem, int len)
{
    /* please note mem donesn't have to be aligned anyhow, so we can't
       write 64-bit numbers to it directly
     */

    enum { buf_items = 256 };
    static unsigned long long buf[buf_items];

    int items_count, partlen, i;

    items_count = (len + sizeof(*buf) - 1) / sizeof(*buf);
    partlen = len;
    if(items_count > buf_items) {
        items_count = buf_items;
        partlen = sizeof(buf);
    }
    for(i = 0; i < items_count; i++)
         buf[i] = get_rand_longlong();
    memcpy(mem, buf, partlen);

    if(partlen < len)
        fill_noise(mem + partlen, len - partlen);
}


int rand_from_range(int first, int last)
{
    unsigned long n;
    n = get_rand_longlong();
    return first + (n % (last - first + 1));
}



const unsigned char fedacert_magic[4] = { 0xfe, 0xda, 0xce, 0x87 };

void compose_feda_cert(struct feda_cert_info *cert,
    const unsigned char node_id[], int point_id, int signer_id, int timestamp,
    const unsigned char key[])
{
    memcpy(cert->feda, fedacert_magic, sizeof(cert->feda));
    memcpy(cert->node_id, node_id, sizeof(cert->node_id));
    cert->point_id = point_id;
    cert->signer_id = signer_id;
    cert->timestamp[0] = (timestamp >> 24) & 0xff;
    cert->timestamp[1] = (timestamp >> 16) & 0xff;
    cert->timestamp[2] = (timestamp >>  8) & 0xff;
    cert->timestamp[3] = timestamp         & 0xff;
    memcpy(cert->key, key, sizeof(cert->key));
}

int get_cert_timestamp(const struct feda_cert_info *cert)
{
    return
        (((unsigned int)cert->timestamp[0]) << 24) |
        (((unsigned int)cert->timestamp[1]) << 16) |
        (((unsigned int)cert->timestamp[2]) <<  8) |
        (unsigned int)cert->timestamp[3];
}

const unsigned char *get_cert_pubkey(const unsigned char *cert)
{
    return ((struct feda_cert_info *)cert)->key;
}

int compare_feda_cert(const struct feda_cert_info *cert,
    const unsigned char node_id[], int point_id, int signer_id, int timestamp,
    const unsigned char *key)
{
    return
        0 == memcmp(cert->feda, fedacert_magic, sizeof(cert->feda)) &&
        0 == memcmp(cert->node_id, node_id, sizeof(cert->node_id)) &&
        cert->point_id == point_id &&
        cert->signer_id == (signer_id == -1 ? 0xff : signer_id) &&
        (timestamp == -1 || timestamp == get_cert_timestamp(cert)) &&
        (!key || 0 == memcmp(cert->key, key, sizeof(cert->key)));
}


void compose_kex_info(struct feda_kex_info *kexinfo,
                      int timestamp, const unsigned char kex_pub[])
{
    kexinfo->timestamp[0] = (timestamp >> 24) & 0xff;
    kexinfo->timestamp[1] = (timestamp >> 16) & 0xff;
    kexinfo->timestamp[2] = (timestamp >>  8) & 0xff;
    kexinfo->timestamp[3] = timestamp         & 0xff;
    memcpy(kexinfo->kex_public, kex_pub, sizeof(kexinfo->kex_public));
}

int compare_kex_info(const struct feda_kex_info *kexinfo,
                     int timestamp, const unsigned char kex_pub[])
{
    struct feda_kex_info fki;
    compose_kex_info(&fki, timestamp, kex_pub);
    return 0 == memcmp(kexinfo, &fki, sizeof(fki));
}


