#include <string.h>
#include <linux/if_tun.h>

#include "ip6struc.h"

static void react_on_nexth(int nexth, int *res)
{
    switch(nexth) {
    case 6:    /* TCP */
    case 17:   /* UDP */
    case 58:   /* ICMPv6 */
            /* nothing unusual/unexpected, no reaction */
        return;

    case 0:    /* Hop-by-Hop Options Header */
    case 41:   /* Encapsulated IPv6 Header */
    case 43:   /* Routing Header */
    case 44:   /* Fragment Header */
    case 46:   /* Resource ReSerVation Protocol */
    case 50:   /* Encapsulating Security Payload */
    case 51:   /* Authentication Header */
    case 59:   /* No next header */
    case 60:   /* Destination Options Header */
            /* we are not good with IPv6 internals, heh */
        *res |= parse6_nexth_cant;
        return;

    default:   /* this may be perfectly okay, but we don't know */
        *res |= parse6_nexth_unknown;
        return;
    }
}

int parse_ipv6_packet(struct ipv6_data *data, const void *p, int sz, int pi)
{
    const unsigned char *q = p;
    int res = 0;
    int flb;

    memset(data, 0, sizeof(*data));

    if(sz < (pi ? 44 : 40))
        return parse6_too_short | parse6_no_info;

    if(pi) {
        data->pi_flags = ((int)q[0] << 8) | q[1];
        q += 2;
        data->pi_proto = ((int)q[0] << 8) | q[1];
        q += 2;

        if(data->pi_flags & TUN_PKT_STRIP)
            res |= parse6_buf_small;
        if(data->pi_flags & ~TUN_PKT_STRIP)
            res |= parse6_inv_flags;

        if(data->pi_proto == ETH_P_IP)
            res |= parse6_pi_ipv4;
        else
        if(data->pi_proto != ETH_P_IPV6)
            res |= parse6_pi_unknown_prot;
    } else {
        data->pi_flags = 0xffff;
        data->pi_proto = 0xffff;
    }

    data->version = (q[0] >> 4) & 0x0f;
    if(data->version == 4)
        res |= parse6_vers_ipv4;
    else
    if(data->version != 6)
        res |= parse6_vers_unknown;

    data->traffic_class = ((q[0] & 0x0f) << 4) | ((q[1] >> 4) & 0x0f);

    flb = q[1] & 0x0f;
    q += 2;
    flb <<= 8; flb |= *q; q++;
    flb <<= 8; flb |= *q; q++;
    data->flow_label = flb;

    data->payload_length = ((int)q[0] << 8) | q[1];
    q += 2;
    if(data->payload_length != sz - (pi ? 44 : 40))
        res |= parse6_inv_payload_len;

    data->next_header = *q;
    q++;
    react_on_nexth(data->next_header, &res);

    data->hop_limit = *q;
    q++;

    data->source_addr = q;
    q += 16;
    data->dest_addr = q;
    q += 16;
    data->payload = q;
    data->payload_size = sz - (q - (const unsigned char*)p);

    return res;
}



const char * const *parse6_get_diags(int code)
{
    static const char * strings[12];
    int idx;

    memset(strings, 0, sizeof(strings));

    if(code == 0) {
        strings[0] = "Success";
        return strings;
    }

    idx = 0;

#define CHECK_CODE(the_bit, msg) do { \
    if(code & the_bit) { strings[idx] = msg; idx++; } } while(0)

    CHECK_CODE(parse6_too_short,       "read size shorter than the header");
    CHECK_CODE(parse6_buf_small,       "PI flags indicate short buf");
    CHECK_CODE(parse6_inv_flags,       "PI flags contain smth. strange");
    CHECK_CODE(parse6_pi_ipv4,         "PI proto indicates it's IPv4");
    CHECK_CODE(parse6_pi_unknown_prot, "PI proto contains smth. strange");
    CHECK_CODE(parse6_vers_ipv4,       "hdr version field is 4");
    CHECK_CODE(parse6_vers_unknown,    "hdr version field not 4 nor 6");
    CHECK_CODE(parse6_inv_payload_len, "hdr payld len mismatches sz");
    CHECK_CODE(parse6_nexth_unknown,   "hdr next_hdr value unknown");
    CHECK_CODE(parse6_nexth_cant,      "next_hdr present, and it may be ok");

#undef CHECK_CODE

    return strings;
}


int ipv6_decrement_hop_limit(void *p)
{
    unsigned char *hl = (unsigned char *)p + 7;
    if(*hl < 1)
        return 0;
    (*hl)--;
    /* return *hl > 0; / no this is bad idea */
    return 1;  /* even if it has just became zero, we should send it
                  in the hope the next hop will be the final destination */
}
