#ifndef INADRCOL_H_SENTRY
#define INADRCOL_H_SENTRY

/*****************************************************************

The structure inaddr_item represents an IP/PORT pair which is
of any interest for the running server.  It is only intended to represent a
member of the collection implemented by the inaddr_collection
structure.

Every item in the collection simultaneously acts as a member of a search
tree (implemented with the treebyte module) and a bidirectional list sorted
by the time of the last communication (with the exception for explicilty
configured peers, which are permanent and are only inserted in the tree,
not in the list).

*****************************************************************/


#include "treebyte.h"


struct inaddr_item;

struct inaddr_collection {
    struct treebyte tree;
    struct inaddr_item *first, *last;
    long long starttime;
    long curtime, timeout;
};

enum { total_bytes_in_addr = 6 };   /* 4 for the IP, 2 for the port */

struct inaddr_item {
    struct inaddr_collection *the_collection;
    struct inaddr_item *prev, *next;
    unsigned char key[total_bytes_in_addr];
                          /* 0..3 hold the IP, 4,5 hold the port, MSBF */
    long timemark;          /* relative to the collection's start time */
    void *userdata;
    void (*timeout_hook)(struct inaddr_item *the_item);
    void (*destruction_hook)(struct inaddr_item *the_item);
};


void inaddrcoll_init(struct inaddr_collection *coll,
                     long long start_time, int timeout);

    /* returns boolean (whether the time has really been changed) */
int inaddrcoll_update_time(struct inaddr_collection *coll,
                           long long current_time);


    /* if add is false, the func returns NULL if nothing found;
       if add is true, it always returns a valid pointer; the
       situation of a newly-added item may be detected by checking
       if the coll->last field has just changed; note that adding is
       always performed at the _end_ of the list
     */
struct inaddr_item *inaddrcoll_find(struct inaddr_collection *coll,
                                    unsigned int ip, unsigned short port,
                                    int add);

#if 0
/* don't do this again. this is impossible to detect in case we didn't save
   (and hence now we don't know) the coll->last value before the addition
   attempt)
 */
int inaddrcoll_isnew(const struct inaddr_collection *coll,
                     const struct inaddr_item *item);
#endif

    /* adds to the tree, but not to the list; removes the item from the
       list if it is already there */
struct inaddr_item *inaddrcoll_permadd(struct inaddr_collection *coll,
                                       unsigned int ip, unsigned short port);


void inaddritem_getaddr(struct inaddr_item *item,
                        unsigned int *ip, unsigned short *port);


    /* reset the timemark, place to the head of the list */
void inaddritem_reset(struct inaddr_item *item);

    /* remove from the collection, call destruction_hook, free the memory */
void inaddritem_remove(struct inaddr_item *item);

    /* look up the list from its head for timed out items, call
       timeout_hook for the timed out, or simply remove them in case
       timeout_hook is NULL; the lookup stops on the first non-timedout
       item; therefore, the timeout_hook, if it is set, _must_ either
       remove the item, or reset it
     */
void inaddrcoll_process(struct inaddr_collection *coll);





#endif
