#include <stdlib.h>
#include <string.h>

#include "addrport.h"

#include "inadrcol.h"



void inaddrcoll_init(struct inaddr_collection *coll, long long start, int tmo)
{
    treebyte_init(&coll->tree, total_bytes_in_addr);
    coll->first = NULL;
    coll->last = NULL;
    coll->starttime = start;
    coll->curtime = 0;
    coll->timeout = tmo;
}

int inaddrcoll_update_time(struct inaddr_collection *coll, long long ct)
{
    int newtm = ct - coll->starttime;
    if(coll->curtime == newtm)
        return 0;
    coll->curtime = newtm;
    return 1; 
}


    /* if add is false, the func returns NULL if nothing found;
       if add is true, it always returns a valid pointer; the
       situation of a newly-added item may be detected by checking
       if the coll->last field has just changed; note that adding is
       always performed at the _end_ of the list
     */
struct inaddr_item *inaddrcoll_find(struct inaddr_collection *coll,
                                    unsigned int ip, unsigned short port,
                                    int add)
{
    unsigned char key[total_bytes_in_addr];
    ipport2mem(key, ip, port);
    
    if(add) {
        void **p = treebyte_provide(&coll->tree, key);
        if(*p) {
            return *p;
        } else {
            struct inaddr_item *item = malloc(sizeof(*item));
            item->the_collection = coll;
            item->next = NULL;
            item->prev = coll->last;
            if(coll->last)
                coll->last->next = item;
            else
                coll->first = item;
            coll->last = item;
            memcpy(item->key, key, sizeof(key));
            item->timemark = coll->curtime;
            item->userdata = NULL;
            item->timeout_hook = NULL;
            item->destruction_hook = NULL;
            *p = item;
            return item;
        }
    } else {
        void *p = treebyte_get(&coll->tree, key);
        return p;
    }
}


static void do_remove_from_list(struct inaddr_collection *coll,
                                struct inaddr_item *item)
{
    if(!item->prev && !item->next && coll->first != item)
        return;
    if(item->prev)
        item->prev->next = item->next;
    else
        coll->first = item->next;
    if(item->next)
        item->next->prev = item->prev;
    else
        coll->last = item->prev;
    item->next = NULL;
    item->prev = NULL;
}


    /* adds to the tree, but not to the list; removes the item from the
       list if it is already there */
struct inaddr_item *inaddrcoll_permadd(struct inaddr_collection *coll,
                                       unsigned int ip, unsigned short port)
{
    unsigned char key[total_bytes_in_addr];
    void **p;
    struct inaddr_item *item;

    ipport2mem(key, ip, port);

    p = treebyte_provide(&coll->tree, key);
    if(*p) {
        item = *p;
        do_remove_from_list(coll, item);
    } else {
        item = malloc(sizeof(*item));
        item->the_collection = coll;
        item->next = NULL;
        item->prev = NULL;
        memcpy(item->key, key, sizeof(key));
        item->timemark = coll->curtime;
        item->userdata = NULL;
        item->timeout_hook = NULL;
        item->destruction_hook = NULL;
        *p = item;
    }
    return item;
}


void inaddritem_getaddr(struct inaddr_item *item,
                        unsigned int *ip, unsigned short *port)
{
    mem2ipport(item->key, ip, port);
}

    /* reset the timemark, place to the head of the list */
void inaddritem_reset(struct inaddr_item *item)
{
    struct inaddr_collection *coll = item->the_collection;

    do_remove_from_list(coll, item);
    item->timemark = coll->curtime;
    item->prev = coll->last;
    if(coll->last)
        coll->last->next = item;
    else
        coll->first = item;
    coll->last = item;
}

    /* remove from the collection, call destruction_hook, free the memory */
void inaddritem_remove(struct inaddr_item *item)
{
    struct inaddr_collection *coll = item->the_collection;

    do_remove_from_list(coll, item);
    treebyte_delete(&coll->tree, item->key);

    if(item->destruction_hook)
        (*item->destruction_hook)(item);

    free(item);
}


    /* look up the list from its head for timed out items, call
       timeout_hook for the timed out, or simply remove them in case
       timeout_hook is NULL; the lookup stops on the first non-timedout
       item; therefore, the timeout_hook, if it is set, _must_ either
       remove the item, or reset it
     */
void inaddrcoll_process(struct inaddr_collection *coll)
{
    long curtime, min2keep;

    curtime = coll->curtime;
    min2keep = curtime - coll->timeout;

    while(coll->first && coll->first->timemark < min2keep) {
        if(coll->first->timeout_hook)
            (*coll->first->timeout_hook)(coll->first);
        else
            inaddritem_remove(coll->first);
    }
}

