#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <linux/if.h>
#include <linux/if_tun.h>
#include <arpa/inet.h>

#include <sue/sue_base.h>

#include "servlog.h"
#include "hexdata.h"
#include "ip6struc.h"
#include "keyutils.h"
#include "fsrv_rx.h"
#include "fsrv_dst.h"

#include "fsrv_tun.h"


#define TUN_READ_BUF_SIZE 2048   /* must not be less than 1500 + 4 */


struct feda_tunnel_gate {
    struct sue_fd_handler fdh;
    struct sue_event_selector *the_selector;
    struct feda_udp_receiver *the_feda_rx;
    struct feda_destination_map *the_destmap;
};

static int get_interface(const char *ifname)
{
    static const char clonedev[] = "/dev/net/tun";
    struct ifreq ifr;
    int fd, err;

    if(strlen(ifname) >= IFNAMSIZ) {
        servlog_message(srvl_alert, "tun dev name too long (%s)", ifname);
        return -1;
    }

    memset(&ifr, 0, sizeof(ifr));
    ifr.ifr_flags = IFF_TUN /* | IFF_NO_PI */;
    strcpy(ifr.ifr_name, ifname);

    fd = open(clonedev, O_RDWR);
    if(fd == -1) {
        servlog_perror(srvl_alert, "can't open clone dev", clonedev);
        return -1;
    }

    err = ioctl(fd, TUNSETIFF, &ifr);
    if(err == -1) {
        servlog_perror(srvl_alert, "ioctl", "TUNSETIFF");
        close(fd);
        return -1;
    }

    return fd;
}

static void deliver_the_packet(struct feda_tunnel_gate *gw,
                               unsigned int packid,
                               const unsigned char *buf, int size,
                               const unsigned char *node_id, int point)
{
    struct feda_peer *fp;

    fp = find_dest_peer(gw->the_destmap, node_id, point);
    if(fp == FEDA_DEST_NO_NODE || fp == FEDA_DEST_NO_NODE_CERT) {
        servlog_message(srvl_debug,
                        "[%08x] dest %s:%02x unreachable, packet dropped",
                        packid, hexdata2a(node_id, node_id_size), point);
        return;
    }
    if(fp == FEDA_DEST_NO_POINT || !fp) {
        servlog_message(srvl_debug,
                        "[%08x] no peer for dest %s:%02x, packet dropped",
                        packid, hexdata2a(node_id, node_id_size), point);
        return;
    }
    if(fp == FEDA_DEST_IS_LOCAL) {
        servlog_message(srvl_debug,
                        "[%08x] (%s:%02x) won't send back locally; dropped",
                        packid, hexdata2a(node_id, node_id_size), point);
        return;
    }

    send_packet_to_peer(gw->the_feda_rx, fp, packid, buf, size);
}


static int ipv6_in_feda(const unsigned char *addr)
{
    return addr[0] == 0xFE && addr[1] == 0xDA;
}

    /* note the buf is non-const to allow decrementing the hop limit */
static void handle_outgoing_ip_packet(struct feda_tunnel_gate *gw,
                                      unsigned char *buf, int size)
{
    int code, r;
    unsigned int packid;
    char addrbuf[48];
    struct ipv6_data data;

    fill_noise((unsigned char*)&packid, sizeof(packid));

    servlog_message(srvl_debug2, "[%08x] packet to send, %d bytes",
                                 packid, size - 4);

    code = parse_ipv6_packet(&data, buf, size, 1);
    code &= ~(parse6_inv_flags | parse6_nexth_unknown | parse6_nexth_cant);
        /* we ignore these three, as it's not our business, heh */
    if(code != 0) {
        const char * const * diags;
        int i;

        if(code & (parse6_no_info | parse6_vers_ipv4 | parse6_vers_unknown)) {
            servlog_message(srvl_alert, "[%08x] pkt unparseable or not IPv6",
                                        packid);
            if((code & parse6_vers_ipv4) && (code & parse6_pi_ipv4)) {
                servlog_message(srvl_alert,
                    "[%08x] IPv4; check routes, don't send IPv4 to FEDA",
                    packid);
            } else {
                servlog_message(srvl_alert,
                    "[%08x] your kernel passed us something strange!",
                    packid);
            }
            return;
        }

        inet_ntop(AF_INET6, data.source_addr, addrbuf, sizeof(addrbuf));
        servlog_message(srvl_debug, "[%08x] src %s", packid, addrbuf);
        inet_ntop(AF_INET6, data.dest_addr, addrbuf, sizeof(addrbuf));
        servlog_message(srvl_debug, "[%08x] dst %s", packid, addrbuf);

        diags = parse6_get_diags(code);
        for(i = 0; diags[i]; i++)
            servlog_message(srvl_info, "[%08x] %s", packid, diags[i]);

        return;
    }

    r = 0;
    if(!ipv6_in_feda(data.source_addr)) {
        inet_ntop(AF_INET6, data.source_addr, addrbuf, sizeof(addrbuf));
        servlog_message(srvl_debug, "[%08x] src %s outside FEDAnet",
                                    packid, addrbuf);
        r = 1;
    }
    if(!ipv6_in_feda(data.dest_addr)) {
        inet_ntop(AF_INET6, data.dest_addr, addrbuf, sizeof(addrbuf));
        servlog_message(srvl_debug, "[%08x] dst %s outside FEDAnet",
                                    packid, addrbuf);
        r = 1;
    }
    if(r)
        return;

    /* the packet looks good, we should try delivering it */
    r = ipv6_decrement_hop_limit(buf + 4);
    if(!r)
        return;
    deliver_the_packet(gw, packid, buf + 4, size - 4,
                       data.dest_addr + 2, data.dest_addr[12]);
}



static void the_fd_handler(struct sue_fd_handler *h, int r, int w, int x)
{
    static unsigned char buf[TUN_READ_BUF_SIZE];
    int rc;
    struct feda_tunnel_gate *gw = h->userdata;

    if(!r || w || x)
        servlog_message(srvl_debug, "tun: the_fd_handler: strange situation");

    rc = read(h->fd, buf, sizeof(buf));
    if(rc == -1)
        servlog_perror(srvl_alert, "tun", "reading error");
    else
    if(rc == 0)
        servlog_message(srvl_alert, "tun: EOF on the tun iface fd");
    else
        handle_outgoing_ip_packet(gw, buf, rc);

    h->want_read = 1;
    h->want_write = 0;
    h->want_except = 0;
}






struct feda_tunnel_gate *make_tunnel_gate(struct sue_event_selector *s,
                                          struct feda_udp_receiver *feda_rx,
                                          struct feda_destination_map *dm,
                                          const char *iface_name)
{
    struct feda_tunnel_gate *p;
    int fd;

    fd = get_interface(iface_name);
    if(fd == -1)
        return NULL;

    p = malloc(sizeof(*p));
    p->fdh.fd = fd;
    p->fdh.want_read = 1;
    p->fdh.want_write = 0;
    p->fdh.want_except = 0;
    p->fdh.userdata = p;
    p->fdh.handle_fd_event = &the_fd_handler;
    p->the_selector = s;
    p->the_feda_rx = feda_rx;
    p->the_destmap = dm;

    /* udp_receiver_set_tun(feda_rx, p, dn);  */

    sue_sel_register_fd(s, &p->fdh);

    return p;
}


void push_packet_locally(struct feda_tunnel_gate *tun,
                         const unsigned char *buf, int size)
{
    int r;
    unsigned char packinf[4] = { 0, 0, ETH_P_IPV6 >> 8, ETH_P_IPV6 & 0xff };
    struct iovec iov[2];

    iov[0].iov_base = packinf;
    iov[0].iov_len = 4;
    iov[1].iov_base = (void*) buf;  /* have to drop the 'const' */
    iov[1].iov_len = size;

    r = writev(tun->fdh.fd, iov, 2);
    if(r == -1) {
        servlog_perror(srvl_alert, "tun", "writev");
        return;
    }
    servlog_message(srvl_debug2, "push_packet_locally: wrote %d of %d+4 bytes",
                                 r, size);
}
