#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <monocypher/monocypher.h>

#include <sue/sue_base.h>

#include "hexdata.h"
#include "cryptodf.h"
#include "keyutils.h"
#include "nmobfusc.h"
#include "servlog.h"
#include "addrport.h"
#include "comcrypt.h"
#include "pointcfg.h"
#include "fsrv_cfg.h"
#include "fsrv_pir.h"
#include "fsrv_dst.h"
#include "fsrv_txq.h"
#include "fsrv_tun.h"
#include "fsrv_pxy.h"
#include "knowndb.h"
/*#include "ip6struc.h"*/

#include "fsrv_rx.h"

/*
   Folks, this module contains a lot of ``magic numbers'' that effectively
   are offsets within dgrams and dgram lengths.  I'm really sorry for that.
   I even hope to live long enough to fix all this mess.  Before sentencing
   me to death penalty, please consider two things.  First, this is very
   common for binary protocol implementations.  E.g., have you ever seen
   DNS from inside?  And second... errr.. I couldn't (at least in any
   reasonable time) make up my mind to realize how all the named constants
   should be organized, and so I decided just to go on, down this dirty way,
   which is still better than nothing.  Having cancer, even despite it's
   curable, makes me value the precious time I still have.

   And one more thing: the whole protocol is implemented here, in this
   single module.  You don't (normally) have to search through the whole
   codebase in case something is to be fixed in these numbers.

   For the protocol specification, please take a look at the
   ../docs/protocol.txt file.
*/




/* ---------------------------------------------------------------
   POTENTIALLY TUNABLE BUT STILL HARDCODED PARAMETERS
   --------------------------------------------------------------- */


/* this is how often the hook for housekeeping will be invoked
   (once per 10 seconds); presently this is hardcoded, but we
   might want to make it configurable one day; the start_delay
   constant is the delay for the housekeeping since the daemon start
 */
enum {
    fsrv_housekeeping_start_delay = 1,
    fsrv_housekeeping_interval    = 10
};


/* The max_unsent_per_peer is the max number of packets still not sent
   for a given peer; once this limit is reached, protocol dgrams remain
   being sent, but dgrams that enclose IPv6 packets get dropped silently
   until the situation improves
 */
enum { max_unsent_per_peer = 1000 };




/* ---------------------------------------------------------------
   DATA STRUCTURES AND THEIR METHODS
   --------------------------------------------------------------- */




struct feda_udp_receiver {
    struct sue_fd_handler fdh;
    struct sue_timeout_handler tmoh;
    struct sue_event_selector *the_selector;
    struct server_conf_info *the_config;
    struct feda_tunnel_gate *the_tun;
    struct feda_destination_map *the_map;
    struct feda_proxy_set *the_proxyset;
    struct crypto_comm_context comctx;
    struct known_nodes_db *kndb;
    struct feda_pir_collection *pir;
    struct feda_transmit_queue *txq;

    struct feda_peer *the_proxy_peer;
    struct feda_peer *the_nodenets_peer;
};


void feda_rx_set_nodenets_peer(struct feda_udp_receiver *feda_rx,
                               struct feda_peer *fp)
{
    if(feda_rx->the_nodenets_peer == fp)
        return;
    servlog_message(srvl_info, "setting peer %s as nodenets target",
                               feda_peer_description(fp));
    feda_rx->the_nodenets_peer = fp;
    destmap_set_nodenets_peer(feda_rx->the_map, fp,
                              feda_rx->the_tun == NULL);
}

void feda_rx_set_proxy_peer(struct feda_udp_receiver *feda_rx,
                            struct feda_peer *fp)
{
    if(feda_rx->the_proxy_peer == fp)
        return;
    if(feda_rx->the_proxy_peer)
        servlog_message(srvl_normal, "warning: CHANGING proxy (old is %s)",
                        feda_peer_description(feda_rx->the_proxy_peer));
    servlog_message(srvl_info, "setting %s as the proxy",
                               feda_peer_description(fp));
    feda_rx->the_proxy_peer = fp;
}

void feda_rx_peer_gone(struct feda_udp_receiver *feda_rx,
                       struct feda_peer *fp)
{
    feda_txq_peer_gone(feda_rx->txq, fp);
    /* handle proxies that may refer to the disappearing object?
       okay, no need: the PIR module does it for us on its own
     */
    if(feda_rx->the_proxy_peer == fp) {
        servlog_message(srvl_alert, "proxy object lost (MUST NEVER HAPPEN)");
        feda_rx->the_proxy_peer = NULL;
    }
    if(feda_rx->the_nodenets_peer == fp) {
        servlog_message(srvl_info, "we've lost the nodenets target peer");
        feda_rx->the_nodenets_peer = NULL;
        destmap_set_nodenets_peer(feda_rx->the_map, NULL, 0);
    }
}

static int can_send_to(const struct feda_udp_receiver *feda_rx,
                       const struct feda_peer *fp)
{
    if(!feda_rx->the_proxy_peer)
        return 1;
    if(feda_peer_is_direct(fp))
        return 1;
    return
        feda_peer_assoc_status(feda_rx->the_proxy_peer) == fpas_established;
}



/* ---------------------------------------------------------------
   SENDING PROTOCOL DATAGRAMS
   --------------------------------------------------------------- */

    /* actual sending is always done through this function, even by proxies */
void perform_send_to(int fd, unsigned int ip, unsigned short port,
                     const void *buf, int len)
{
    struct sockaddr_in saddr;
    int r;

    saddr.sin_family = AF_INET;
    saddr.sin_addr.s_addr = htonl(ip);
    saddr.sin_port = htons(port);

    r = sendto(fd, buf, len, 0, (struct sockaddr*)&saddr, sizeof(saddr));
    if(r < 1) {
        servlog_perror(srvl_alert, "ALERT", "sendto");
        servlog_message(srvl_alert, "error sending %d bytes to %s",
                                    len, ipport2a(ip, port));
    } else
    if(r != len) {
        servlog_message(srvl_alert,
                        "dgram len mismatch: %d to send, %d sent", len, r);
    } else {
        servlog_message(srvl_debug2, "[fd==%d] sent %d bytes to %s",
                                     fd, len, ipport2a(ip, port));
    }
}

static void enqueue_datagram(struct feda_udp_receiver *feda_rx,
                             struct feda_transmit_item *msg)
{
    if(feda_rx->the_proxy_peer) {
        /* once we have a configured proxy, we must take care that
           messages to THE proxy are sent directly, without trying
           to send them THROUGH the proxy
         */
        if(msg->the_peer) {
            msg->direct =
                feda_peer_is_direct(msg->the_peer) ||
                msg->the_peer == feda_rx->the_proxy_peer;
        } else {
            unsigned int ip;
            unsigned short port;
            feda_peer_getaddr(feda_rx->the_proxy_peer, &ip, &port);
            msg->direct = msg->ip == ip && msg->port == port;
        }
    }
    /* this must be the only call to feda_txq_enqueue in this module */
    feda_txq_enqueue(msg);
}

/* generic datagram structures ----------------------------------- */

static void send_plaintext_128(struct feda_udp_receiver *feda_rx,
                               unsigned int ip, unsigned short port,
                               int cmd,
                               unsigned char *payload,
                               int payload_len)
{
    struct feda_transmit_item *msg;

    msg = make_txitem_4ip(feda_rx->txq, 128, 0, ip, port);

    set_plain_dgram_head(msg->buf, cmd);
    if(payload_len > 0)
        memcpy(msg->buf + 2, payload, payload_len);
    fill_noise(msg->buf + 2 + payload_len, msg->len - 2 - payload_len);
    servlog_message(srvl_debug2, "sending plaintext dgram (cmd=%02x) to %s",
                    cmd, ipport2a(msg->ip, msg->port));
    enqueue_datagram(feda_rx, msg);
}

static void send_plaintext_with_pub(struct feda_udp_receiver *feda_rx,
                                    unsigned int ip, unsigned short port,
                                    int cmd,
                                    unsigned char *payload, int payload_len)
{
    unsigned char fullload[126];

    if(payload_len > 94) {
        servlog_message(srvl_alert,
                        "send_plaintext_with_pub: payload too long (%d)",
                        payload_len);
        return;
    }

    memcpy(fullload, feda_rx->comctx.kex_public, kex_public_size);
    memcpy(fullload + kex_public_size, payload, payload_len);
    send_plaintext_128(feda_rx, ip, port, cmd, fullload,
                       kex_public_size + payload_len);
}


static void send_semiencrypted_with_key(struct feda_udp_receiver *feda_rx,
                                        struct feda_peer *fp,
                                        const unsigned char *encrypt_key,
                                        int cmd,
                                        unsigned char *payload,
                                        int payload_len)
{
    struct feda_transmit_item *msg;
    unsigned char nonce[cipher_nonce_total];
    unsigned char *bufp, *macpos, *ct;
    int ctpos, ctsize, totalsize;

    totalsize = 58 + payload_len;

    msg = make_txitem_4peer(feda_rx->txq,
                            totalsize > 128 ? totalsize : 128, 0, fp);

    bufp = msg->buf;
    set_plain_dgram_head(bufp, cmd);
    bufp += 2;
    memcpy(bufp, feda_rx->comctx.kex_public, kex_public_size);
    bufp += kex_public_size;

    memset(nonce, 0, cipher_nonce_offset);
    comctx_fill_nonce(&feda_rx->comctx, nonce + cipher_nonce_offset);
    memcpy(bufp, nonce + cipher_nonce_offset, cipher_nonce_used);
    obfuscate_buffer(bufp, cipher_nonce_used);
    bufp += cipher_nonce_used;

    macpos = bufp;
    bufp += cipher_mac_size;
    ct = bufp;

    ctpos = ct - msg->buf;   /* must always be 58 */
    ctsize = msg->len - ctpos;

    memcpy(ct, payload, payload_len);
    bufp += payload_len;

    if(msg->len - ctpos - payload_len > 0) 
        fill_noise(bufp, msg->len - ctpos - payload_len);

    crypto_aead_lock(ct, macpos, encrypt_key, nonce, NULL, 0, ct, ctsize);

    servlog_message(srvl_debug2,
                    "sending semiencrypted dgram (cmd=%02x, size=%d) to %s",
                    cmd, msg->len, ipport2a(msg->ip, msg->port));

    enqueue_datagram(feda_rx, msg);
}

static void send_semiencrypted_128(struct feda_udp_receiver *feda_rx,
                                   struct feda_peer *fp,
                                   int cmd,
                                   unsigned char *payload,
                                   int payload_len)
{
    send_semiencrypted_with_key(feda_rx, fp, feda_peer_encrypt_key(fp),
                                cmd, payload, payload_len);
}

static void send_encrypted(struct feda_udp_receiver *feda_rx,
                           struct feda_peer *fp,
                           const unsigned char *payload, int payload_len,
                           int direct)
{
    struct feda_transmit_item *msg;
    unsigned char *macpos, *ct;
    int dgram_size, padding, totalsize;

    dgram_size = cipher_nonce_used + cipher_mac_size + payload_len;
    if(fedaprot_min_dgram > dgram_size) {
        padding = fedaprot_min_dgram - dgram_size;
        dgram_size = fedaprot_min_dgram;
    } else {
        padding = 0;
    }
    totalsize = cipher_nonce_total + cipher_mac_size + payload_len + padding;

    msg = make_txitem_4peer(direct ? NULL : feda_rx->txq,
                            totalsize, cipher_nonce_offset, fp);

    memset(msg->buf, 0, cipher_nonce_offset);
    comctx_fill_nonce(&feda_rx->comctx, msg->buf + cipher_nonce_offset);

    macpos = msg->buf + cipher_nonce_total;
    ct = macpos + cipher_mac_size;
    memcpy(ct, payload, payload_len);

    if(padding > 0) 
        fill_noise(ct + payload_len, padding);

    crypto_aead_lock(ct, macpos, feda_peer_encrypt_key(fp), msg->buf, NULL, 0,
                     ct, payload_len + padding);

    obfuscate_buffer(msg->buf + cipher_nonce_offset, cipher_nonce_used);

    if(msg->buf[msg->offset] > fedaprot_zb_maxverb) {
        msg->offset--;
        msg->buf[msg->offset] =
            rand_from_range(fedaprot_zb_minenc, fedaprot_zb_maxenc);
    }

    servlog_message(srvl_debug2,
                    "sending encrypted dgram%s (cmd=%02x, size=%d/%d) to %s",
                    direct ? " DIRECTLY" : "",
                    payload[0], msg->len - msg->offset, payload_len,
                    ipport2a(msg->ip, msg->port));
    if(direct) {
        perform_send_to(feda_rx->fdh.fd, msg->ip, msg->port,
                        msg->buf + msg->offset, msg->len - msg->offset);
        feda_txitem_sent(msg);
    } else {
        enqueue_datagram(feda_rx, msg);
    }
}


/* protocol messages of particular types ------------------------- */

    /* these two don't fall into any of the categories */

static void send_introduction_request(struct feda_udp_receiver *feda_rx,
                                      struct feda_peer *fp)
{
    struct feda_transmit_item *msg;

    msg = make_txitem_4peer(feda_rx->txq, 364, 0, fp);

    set_plain_dgram_head(msg->buf, fedaprot_intro_req);
    fill_noise(msg->buf + 2, msg->len - 2);
    servlog_message(srvl_debug, "sending introduction request to %s",
                    feda_peer_description(fp));
    enqueue_datagram(feda_rx, msg);
}

static void send_self_introduction(struct feda_udp_receiver *feda_rx,
                                   struct feda_peer *fp)
{
    struct feda_transmit_item *msg;
    unsigned char *p;

    msg = make_txitem_4peer(feda_rx->txq, 364, 0, fp);

    set_plain_dgram_head(msg->buf, fedaprot_i_am);
    p = msg->buf + 2;

    memcpy(p, feda_rx->comctx.point->master_pub, public_key_size);
    p += public_key_size;
    memcpy(p, feda_rx->comctx.point->master_hash, yespower_hash_size);
    p += yespower_hash_size;
    memcpy(p, feda_rx->comctx.point->master_hsign, signature_size);
    p += signature_size;
    *p = feda_rx->comctx.point->point;
    p++;
    *p = feda_rx->comctx.point->signed_by;
    p++;
    if(feda_rx->comctx.point->signed_by == 0) {
        memcpy(p, feda_rx->comctx.point->zp_certbody, feda_cert_size);
        p += feda_cert_size;
        memcpy(p, feda_rx->comctx.point->zp_signature, signature_size);
        p += signature_size;
    } else {
        fill_noise(p, feda_cert_size + signature_size);
        p += feda_cert_size + signature_size;
    }
    memcpy(p, feda_rx->comctx.point->certbody, feda_cert_size);
    p += feda_cert_size;
    memcpy(p, feda_rx->comctx.point->signature, signature_size);

    servlog_message(srvl_debug, "sending introduction (certificates) to %s",
                                feda_peer_description(fp));
    enqueue_datagram(feda_rx, msg);
}

    /* the rest is done via category functions */

static void send_exchange_error(struct feda_udp_receiver *feda_rx,
                                  unsigned int ip, unsigned short port,
                                  int error_code)
{
    unsigned char payload = error_code;
    send_plaintext_128(feda_rx, ip, port, fedaprot_error, &payload, 1);
}

    /* please note remote_nonce is passed still obfuscated */
static void send_change_key(struct feda_udp_receiver *feda_rx,
                            unsigned int ip, unsigned short port,
                            const unsigned char *remote_nonce)
{
    enum { signed_part_len =
        kex_public_size + 2*cipher_nonce_used + 4 /* timemart */ };
    unsigned char payload[/* 116 */ signed_part_len + signature_size];
    unsigned char *pp = payload;

    memcpy(pp, feda_rx->comctx.kex_public, kex_public_size);
    pp += kex_public_size;
    memcpy(pp, remote_nonce, cipher_nonce_used);
    pp += cipher_nonce_used;
    place_timemark(timemark_minutes(feda_rx->pir), pp);
    pp += 4;
    comctx_fill_nonce(&feda_rx->comctx, pp);
    obfuscate_buffer(pp, cipher_nonce_used);
    pp += cipher_nonce_used;
    crypto_eddsa_sign(pp, feda_rx->comctx.point_secret,
                      payload, signed_part_len);

    send_plaintext_128(feda_rx, ip, port, fedaprot_change_key,
                       payload, sizeof(payload));
}


static void send_cr_error(struct feda_udp_receiver *feda_rx,
                          struct feda_peer *fp, int code)
{
    unsigned char payload[2];
    payload[0] = fedaprot_cr_error;
    payload[1] = code;
    send_encrypted(feda_rx, fp, payload, 2, 0);
}


static void send_keepalive_dgram(struct feda_udp_receiver *feda_rx,
                                 struct feda_peer *fp)
{
    unsigned char payload = fedaprot_cr_keepalive;
    send_encrypted(feda_rx, fp, &payload, 1, 0);
}


    /* encrypt_key may be NULL, which means to use the peer's one */
static void send_echo_response(struct feda_udp_receiver *feda_rx,
                               struct feda_peer *fp,
                               const unsigned char *tmark_and_nonce,
                               const unsigned char *encrypt_key)
{
    unsigned char buf[102];
    unsigned int ip;
    unsigned short port;

    /*memcpy(buf, peer->key, sizeof(peer->key));*/
    feda_peer_getaddr(fp, &ip, &port);
    ipport2mem(buf, ip, port);
    memcpy(buf+6, tmark_and_nonce, 12);
    memcpy(buf+18, feda_rx->comctx.point->node_id, node_id_size);
    buf[28] = feda_rx->comctx.point->point;
    buf[29] = feda_rx->comctx.point->minpref;
    comctx_fill_nonce(&feda_rx->comctx, buf+30);
    crypto_eddsa_sign(buf+38, feda_rx->comctx.point_secret, 
                      feda_rx->comctx.kex_public, kex_public_size);

    if(encrypt_key) {
        send_semiencrypted_with_key(feda_rx, fp, encrypt_key,
                               fedaprot_echo_reply, buf, sizeof(buf));
    } else {
        send_semiencrypted_128(feda_rx, fp, fedaprot_echo_reply,
                               buf, sizeof(buf));
    }
}

static void send_natcheck_response(struct feda_udp_receiver *feda_rx,
                                   struct feda_peer *fp,
                                   struct feda_peer *siblings[10], int scnt)
{
    int i;
    unsigned char buf[68];
    unsigned int ip;
    unsigned short port;

    /*memcpy(buf, peer->key, sizeof(peer->key));*/  /* 0..5 */
    feda_peer_getaddr(fp, &ip, &port);
    ipport2mem(buf, ip, port);

    buf[6] = feda_peer_stub_credit(fp);
    buf[7] = 0;      /* was rand_from_range(0, 255); it's not a good idea */

    memset(buf + 8, 0, 60);   /* 10 records, 6 bytes each */
    for(i = 0; i < scnt; i++) {
        /*memcpy(buf + 8 + i*6, siblings[i]->key, 6);*/
        feda_peer_getaddr(siblings[i], &ip, &port);
        ipport2mem(buf + 8 + i*6, ip, port);
    }

    send_semiencrypted_128(feda_rx, fp, fedaprot_test_reply, buf, sizeof(buf));
}

void send_encrypted_please_contact(struct feda_udp_receiver *feda_rx,
                                   struct feda_peer *requester,
                                   struct feda_peer *target)
{
    unsigned char payload[8];
    char reqstr[addrport_max_string];
    char tgtstr[addrport_max_string];
    unsigned int ip;
    unsigned short port;

/*
    mem2str(reqstr, requester->key);
    mem2str(tgtstr, target->key);
 */

    feda_peer_getaddr(target, &ip, &port);
    ipport2str(tgtstr, ip, port);
    feda_peer_getaddr(requester, &ip, &port);
        /* note requester's ip/port is left in the variables */
    ipport2str(reqstr, ip, port);

    servlog_message(srvl_debug, "sending to %s (encr) 'please contact' %s",
                                tgtstr, reqstr);

    payload[0] = fedaprot_cr_plzcont;
    payload[1] = 0;
    /*memcpy(payload+2, requester->key, 6);*/
    ipport2mem(payload+2, ip, port);    
    send_encrypted(feda_rx, target, payload, sizeof(payload), 0);
}

void send_encrypted_nodect_re(struct feda_udp_receiver *feda_rx,
                              struct feda_peer *fp,
                              const unsigned char *cookie,
                              const unsigned char *node_id,
                              int rank,
                              const unsigned char *pubkey,
                              const unsigned char *hash,
                              const unsigned char *hsign)
{
    unsigned char payload[214];  /* payload[0] is the 24th byte */
    unsigned char *p = payload;
    unsigned char *signed_part;

    *p = fedaprot_cr_nodect_re;
    p++;
    *p = 0;
    p++;
    memcpy(p, cookie, 4);
    p += 4;
    memcpy(p, node_id, node_id_size);
    p += node_id_size;
    *p = 1;   /* yes, we know the node */
    p++;
    signed_part = p;
    *p = rank;
    p++;
    memcpy(p, pubkey, public_key_size);
    p += public_key_size;
    memcpy(p, hash, yespower_hash_size);
    p += yespower_hash_size;
    memcpy(p, hsign, signature_size);
    p += signature_size;
    place_timemark(timemark_minutes(feda_rx->pir), p);
    p += 4;
    crypto_eddsa_sign(p, feda_rx->comctx.point_secret,
                      signed_part, p - signed_part);

#if 1
    if((p - payload) + signature_size != sizeof(payload) ||
        (p - payload) + signature_size + 24 != 238
    ) {
        servlog_message(srvl_alert,
                        "sizes mismatch in send_encrypted_nodect_re");
    }
#endif

    send_encrypted(feda_rx, fp, payload, sizeof(payload), 0);
}

void send_encrypted_nodect_neg(struct feda_udp_receiver *feda_rx,
                               struct feda_peer *fp,
                               const unsigned char *cookie,
                               const unsigned char *node_id)
{
    unsigned char payload[214];  /* payload[0] is the 24th byte */
    unsigned char *p = payload;

    *p = fedaprot_cr_nodect_re;
    p++;
    *p = 0;
    p++;
    memcpy(p, cookie, 4);
    p += 4;
    memcpy(p, node_id, node_id_size);
    p += node_id_size;
    *p = 0;   /* no, we don't know the node */
    p++;
    *p = feda_rx->comctx.point->minpref;
    p++;
    memset(p, 0, sizeof(payload) - (p - payload));

    send_encrypted(feda_rx, fp, payload, sizeof(payload), 0);
}


static void send_echo_request(struct feda_udp_receiver *feda_rx,
                              struct feda_peer *fp)
{
    unsigned char payload[12];
    unsigned int ip;
    unsigned short port;

    place_timemark(timemark_minutes(feda_rx->pir), payload);
    comctx_fill_nonce(&feda_rx->comctx, payload + 4);
    feda_peer_getaddr(fp, &ip, &port);
    send_plaintext_with_pub(feda_rx, ip, port, fedaprot_echo_req,
                            payload, sizeof(payload));
}

static void send_assoc_request(struct feda_udp_receiver *feda_rx,
                               struct feda_peer *fp)
{
    unsigned char pld[320];
    unsigned char signed_info[public_key_size + 4 + cipher_nonce_used];

    memcpy(pld, feda_rx->comctx.point->node_id, node_id_size);
    pld[10] = feda_rx->comctx.point->point;
    pld[11] = feda_rx->comctx.point->signed_by == 0 ? 0x00 : 0xff;
    if(pld[11] == 0x00) {
        memcpy(pld+12, feda_rx->comctx.point->zp_certbody, feda_cert_size);
        memcpy(pld+64, feda_rx->comctx.point->zp_signature, signature_size);
    } else {
        fill_noise(pld+12, feda_cert_size + signature_size);
    }
    memcpy(pld+128, feda_rx->comctx.point->certbody, feda_cert_size);
    memcpy(pld+180, feda_rx->comctx.point->signature, signature_size);
    place_timemark(timemark_minutes(feda_rx->pir), pld+244);
    comctx_fill_nonce(&feda_rx->comctx, pld+248);

    memcpy(signed_info, feda_rx->comctx.kex_public, kex_public_size);
    memcpy(signed_info + kex_public_size, pld+244, 12);
    crypto_eddsa_sign(pld+256, feda_rx->comctx.point_secret,
                      signed_info, sizeof(signed_info));

    send_semiencrypted_128(feda_rx, fp, fedaprot_associate,
                           pld, sizeof(pld));
}



/* ---------------------------------------------------------------
   INCOMING REQUEST HANDLERS
   --------------------------------------------------------------- */


static void handle_intro_request(struct feda_udp_receiver *feda_rx,
                                 unsigned int ip, unsigned short port,
                                 int payload_len)
{
    struct feda_peer *fp;

    if(payload_len < 362) {
        servlog_message(srvl_normal, "broken intro req. from %s (%d bytes)",
                                     ipport2a(ip, port), payload_len+2);
        send_exchange_error(feda_rx, ip, port, fedaprot_err_broken);
        return;
        /* YYY definitely we should implement some kind of counters and
           limitations for things like this */
    }

    fp = get_peer_rec(feda_rx->pir, ip, port, 1);
    send_self_introduction(feda_rx, fp);
}

static int kndbres2fedaproterr(int r)
{
    switch(r) {
    case kndb_res_success:          return 0;
    case kndb_res_node_unknown:     return fedaprot_err_node_unkn;
    case kndb_res_node_rank_low:    return fedaprot_err_rank_low;
    case kndb_res_wrong_node_key:
    case kndb_res_cert_outdated:
    case kndb_res_zpcert_outdated:
    case kndb_res_node_key_differs:
    case kndb_res_signature_fail:   return fedaprot_err_signature;
    case kndb_res_point_unknown:
    case kndb_res_file_not_found:
    case kndb_res_file_error:
    case kndb_res_format_error:
    case kndb_res_unexpected_thing:
    case kndb_res_not_implemented:
    default:                        return fedaprot_err_internal;
    }
}

static void handle_introduction_dgram(struct feda_udp_receiver *feda_rx,
                                      unsigned int ip, unsigned short port,
                                      unsigned char *payload,
                                      int payload_len)
{
    struct feda_peer *fp;
    unsigned const char *master_pub, *master_hash, *master_hsign,
        *zp_certbody, *zp_signature, *certbody, *signature, *node_id, *p;
    int point;
    unsigned char pubkey[public_key_size];
    int r;

    if(payload_len < 362) {
        servlog_message(srvl_normal, "broken intro dgram from %s (%d bytes)",
                                     ipport2a(ip, port), payload_len+2);
        send_exchange_error(feda_rx, ip, port, fedaprot_err_broken);
        return;
        /* YYY definitely we should implement some kind of counters and
           limitations for things like this */
    }

    fp = get_peer_rec(feda_rx->pir, ip, port, 0);
    if(!fp) {
        servlog_message(srvl_debug, "ignored stray intro dgram from %s",
                                    ipport2a(ip, port));
        return;
    }

    if(feda_peer_assoc_status(fp) != fpas_intro_requested) {
        servlog_message(srvl_debug, "unrequested intro dgram from %s",
                                    feda_peer_description(fp));
        send_exchange_error(feda_rx, ip, port, fedaprot_err_unexpected);
        return;
    }

    master_pub = payload;
    node_id = master_pub + (public_key_size - node_id_size);
    master_hash = master_pub + public_key_size;
    master_hsign = master_hash + yespower_hash_size;
    p = master_hsign + signature_size;
    point = *p;
    p++;
    /* signed_by = *p;  // we ignore it */
    p++;
    zp_certbody = p;
    zp_signature = zp_certbody + feda_cert_size;
    certbody = zp_signature + signature_size;
    signature = certbody + feda_cert_size;

    r = kndb_did_we_know(feda_rx->kndb, node_id, point,
                         zp_certbody, zp_signature, certbody, signature,
                         pubkey);
    if(r != kndb_res_success) {
        servlog_message(srvl_debug, "intro dgram from %s rejected (%s)",
                        feda_peer_description(fp), kndb_result_message(r));
        feda_peer_set_assocstatus(fp, fpas_gave_up);
        send_exchange_error(feda_rx, ip, port, kndbres2fedaproterr(r));
        return;
    }

    servlog_message(srvl_info, "intro dgram accepted (%s.%d)",
                    hexdata2a(node_id, node_id_size), point);

    r = feda_peer_set_identity(fp, node_id, point, pubkey);
    if(!r) {
        servlog_message(srvl_debug, "giving up association with %s",
                                    feda_peer_description(fp));
        feda_peer_set_assocstatus(fp, fpas_gave_up);
        send_exchange_error(feda_rx, ip, port, fedaprot_err_unexp_point);
        return;
    }

    feda_peer_set_assocstatus(fp, fpas_echo_request_sent);
    send_echo_request(feda_rx, fp);
}


static void handle_associate_request(struct feda_udp_receiver *feda_rx,
                                     unsigned int ip, unsigned short port,
                                     unsigned char *payload,
                                     int payload_len)
{                                             /* payload here is 2..end */
    int r;
    struct feda_peer *fp;
    unsigned char decrypt_key[cipher_key_size];
    const unsigned char *noncepart, *mac;
    unsigned char *ctext;
    unsigned char *node_id, *point_p, *sigby_p;
    unsigned char *zp_certbody;
    unsigned char *zp_signature;
    unsigned char *point_certbody;
    unsigned char *point_signature;
    unsigned char nonce[cipher_nonce_total];
    unsigned char point_pubkey[public_key_size];
    unsigned char signed_stuff[public_key_size + 4 + cipher_nonce_used];

    fp = get_peer_rec(feda_rx->pir, ip, port, 1);

        /* it is very possible that this peer is either new, or we didn't
           intend to associate with it; however, let's first see what's
           inside the request!
         */

    if(payload_len < 318) {
        servlog_message(srvl_normal, "broken assoc. req. from %s (%d bytes)",
                                     ipport2a(ip, port), payload_len+2);
        send_exchange_error(feda_rx, ip, port, fedaprot_err_broken);
        return;
        /* YYY definitely we should implement some kind of counters and
           limitations for things like this */
    }

        /* we don't even try to install the new remote public key,
           we'd better understand if it is valid and then install it
           forcibly
         */
    derive_cipher_keys(feda_rx->comctx.kex_secret, feda_rx->comctx.kex_public,
                       payload, NULL, decrypt_key);

    noncepart = payload + kex_public_size;
    mac = noncepart + cipher_nonce_used;
    ctext = payload + kex_public_size + cipher_nonce_used + cipher_mac_size;
    memset(nonce, 0, cipher_nonce_offset);
    memcpy(nonce + cipher_nonce_offset, noncepart, cipher_nonce_used);
    deobfuscate_buffer(nonce + cipher_nonce_offset, cipher_nonce_used);
    r = crypto_aead_unlock(ctext, mac, decrypt_key, nonce, NULL, 0,
                           ctext, payload_len - (ctext - payload));
    if(r != 0  /* sic! */) {  /* decrypting failed */
        servlog_message(srvl_normal,
                        "failed decrypting association request from %s",
                        feda_peer_description(fp));
        servlog_message(srvl_debug|srvl_private, "decr. key used: %s",
                    hexdata2a(decrypt_key, cipher_key_size));
        servlog_message(srvl_debug, "our public kex key: %s",
                    hexdata2a(feda_rx->comctx.kex_public, public_key_size));
        servlog_message(srvl_debug, "their public kex key: %s",
                    hexdata2a(payload, public_key_size));
        send_exchange_error(feda_rx, ip, port, fedaprot_err_decrypt);
        return;

        /* YYY do something on this! */
    }

    /* now that is has unlocked, let's verify the signature */

    node_id = ctext;
    point_p = node_id + node_id_size;
    sigby_p = point_p + 1;
    zp_certbody = sigby_p + 1;
    zp_signature = zp_certbody + feda_cert_size;
    point_certbody = zp_signature + signature_size;
    point_signature = point_certbody + feda_cert_size;

    r = kndb_did_we_know(feda_rx->kndb, node_id, *point_p,
                         zp_certbody, zp_signature,
                         point_certbody, point_signature,
                         point_pubkey);
    switch(r) {
    case kndb_res_success:
        break;
    case kndb_res_node_unknown:
    case kndb_res_file_error:
        send_exchange_error(feda_rx, ip, port, fedaprot_err_node_unkn);
        return;
    case kndb_res_node_rank_low:
        send_exchange_error(feda_rx, ip, port, fedaprot_err_rank_low);
        return;
    case kndb_res_cert_outdated:
    case kndb_res_zpcert_outdated:
    case kndb_res_signature_fail:
        send_exchange_error(feda_rx, ip, port, fedaprot_err_signature);
        return;
    case kndb_res_point_unknown:   /* this must not happen _here_ */
        servlog_message(srvl_alert, "BUG: unexpected _point_unknown");
        send_exchange_error(feda_rx, ip, port, fedaprot_err_internal);
        return;
    default:
        servlog_message(srvl_alert,
            "kndb_did_we_know says %s (%d), which we can't handle",
            kndb_result_message(r), r);
        send_exchange_error(feda_rx, ip, port, fedaprot_err_internal);
        return;
    }

    /* we have everything we need to verify the signature, let's do it */

    memcpy(signed_stuff, payload, kex_public_size);
    memcpy(signed_stuff+kex_public_size, ctext+244, 4+cipher_nonce_used);
    r = crypto_eddsa_check(ctext+256, point_pubkey,
                           signed_stuff, sizeof(signed_stuff));
    if(r != 0 /* sic */) {
        char nodeid_str[node_id_size * 2 + 1];
        hexdata2str(nodeid_str, node_id, node_id_size);
        servlog_message(srvl_normal,
            "assoc. request signature check failed (%s, node %s, point %d)",
            feda_peer_description(fp), nodeid_str, *point_p);
        servlog_message(srvl_debug, "we used %s as the remote public key",
                        hexdata2a(point_pubkey, public_key_size));
                        
        send_exchange_error(feda_rx, ip, port, fedaprot_err_signature);
        return;
    }

    peer_set_kex_public(feda_rx->pir, fp, payload, 1 /* signature verified */);
    r = feda_peer_set_identity(fp, node_id, *point_p, point_pubkey);
    if(!r) {
        servlog_message(srvl_debug, "giving up association with %s",
                                    ipport2a(ip, port));
        feda_peer_set_assocstatus(fp, fpas_gave_up);
        send_exchange_error(feda_rx, ip, port, fedaprot_err_unexp_point);
        return;
    }

    set_destination_peer(feda_rx->the_map, node_id, *point_p, fp);
    feda_peer_set_assocstatus(fp, fpas_established);
    servlog_message(srvl_normal,
                    "association with %s established (remote req.)",
                    feda_peer_description(fp));
    update_peer_last_rx(fp);
    send_keepalive_dgram(feda_rx, fp);
}

static void handle_change_key_request(struct feda_udp_receiver *feda_rx,
                                      unsigned int ip, unsigned short port,
                                      unsigned char *payload,
                                      int payload_len)
{
    enum { signed_part_len =
        kex_public_size + 2*cipher_nonce_used + 4 /* timemart */ };

    int r;
    struct feda_peer *fp;
    int assocstat;
    unsigned char node_id[node_id_size];
    int point;

    fp = get_peer_rec(feda_rx->pir, ip, port, 0);
    if(!fp) {
            /* we didn't have a crypto assoc with this peer, ignore it */
        servlog_message(srvl_info, "stray ``change key'' from %s",
                                   ipport2a(ip, port));
        return;
        /* YYY definitely we should implement some kind of counters and
           limitations for things like this */
    }

    assocstat = feda_peer_assoc_status(fp);
    switch(assocstat) {
    case fpas_not_desired:
        servlog_message(srvl_info,
            "ignoring change key request from %s (not desired to assoc.)",
            feda_peer_description(fp));
        return;
    case fpas_none:
    case fpas_echo_request_sent:
    case fpas_intro_requested:
    case fpas_assoc_request_sent:
    case fpas_established:
        break;
    case fpas_gave_up:    /* okay, let's give it another try */
        feda_peer_set_assocstatus(fp, fpas_none);
        break;
    default:
        servlog_message(srvl_debug,
                        "unknown value (%d) for assoc. status of %s",
                        assocstat, feda_peer_description(fp));
        return;
    }

    /* YYY damn check the timemark! and the returned nonce! HERE!!! */

    /* now we need to verify the signature; if there really was an active
       crypto association, the public key of the point must reside in
       the peer object, but let's see first */

    /* if(all_zeroes(fp->remote_public, sizeof(fp->remote_public))) { */
    if(!feda_peer_ever_had_assoc(fp)) {
            /* looks like there never was an association with this peer */
            /* we just initiate one, ignoring the request we've just got */
        servlog_message(srvl_info,
            "change key req. from %s but no memorized pubkey; will ping",
            feda_peer_description(fp));
        feda_peer_set_assocstatus(fp, fpas_echo_request_sent);
        send_echo_request(feda_rx, fp);
        return;
    }

    r = crypto_eddsa_check(payload + signed_part_len,
                           feda_peer_remote_public(fp),
                           payload, signed_part_len);
    if(r != 0 /* sic */) {
        servlog_message(srvl_info,
            "change key request from %s signature failed, ignoring",
            feda_peer_description(fp));
        return;
    }

    /* well, now we know for sure */

    peer_set_kex_public(feda_rx->pir, fp, payload, 1 /* signature ok */);

    feda_peer_get_point(fp, node_id, &point);
    servlog_message(srvl_info,
                    "sending assoc. req. to %s to finish key change",
                    feda_peer_description(fp));
    feda_peer_set_assocstatus(fp, fpas_assoc_request_sent);
    send_assoc_request(feda_rx, fp);
}

static void handle_echo_reply(struct feda_udp_receiver *feda_rx,
                              unsigned int ip, unsigned short port,
                              unsigned char *payload,
                              int payload_len)
{
    int r;
    struct feda_peer *fp;
    unsigned char nonce[cipher_nonce_total];
    const unsigned char *noncepart, *mac, *node_id, *point_p, *sign_p;
    unsigned char *ctext;
    unsigned char remote_pubkey[public_key_size];
    char nodeid_str[node_id_size * 2 + 1];
    int assocstat;

    fp = get_peer_rec(feda_rx->pir, ip, port, 0);
    if(!fp) {
        /* we definitely didn't send echo request to an unknown peer */
        servlog_message(srvl_info, "stray ``echo reply'' from %s",
                                   ipport2a(ip, port));
        send_exchange_error(feda_rx, ip, port, fedaprot_err_unexpected);
        return;
        /* YYY definitely we should implement some kind of counters and
           limitations for things like this */
    }

    if(payload_len < 158) {
        servlog_message(srvl_normal, "broken echo reply dgram from %s",
                                     feda_peer_description(fp));
        send_exchange_error(feda_rx, ip, port, fedaprot_err_broken);
        return;
        /* YYY definitely we should implement some kind of counters and
           limitations for things like this */
    }

    assocstat = feda_peer_assoc_status(fp);
    switch(assocstat) {
    case fpas_none:                /* unexpected, but... why no? */
    case fpas_echo_request_sent:   /* yes, we were looking for this reply */
        break;
    case fpas_gave_up:             /* well if they want, heh */
        feda_peer_set_assocstatus(fp, fpas_none);
        break;
    case fpas_not_desired:         /* we definitely didn't send echo req. */
        servlog_message(srvl_info,
            "ignoring unrequested echo reply from %s (not desired to assoc.)",
            feda_peer_description(fp));
        return;
    case fpas_assoc_request_sent:  /* we're not so sure but... no */
    case fpas_intro_requested:
        servlog_message(srvl_info,
            "ignoring unrequested echo reply from %s (other req. was sent)",
            feda_peer_description(fp));
        return;
    case fpas_established:
        servlog_message(srvl_info,
            "ignoring unrequested echo reply from %s (assoc. established)",
            feda_peer_description(fp));
        return;
    default:
        servlog_message(srvl_debug,
                        "unknown value (%d) for assoc. status of %s",
                        assocstat, feda_peer_description(fp));
        return;
    }

    /* okay, now it's desirable to try decrypting it */
    /* no sence to bother protecting existing assoc., as there's none */

    r = peer_set_kex_public(feda_rx->pir, fp, payload, 0 /* no sign. here */);
    if(!r) {  /* this must never happen! just check it out of paranoia */
        servlog_message(srvl_normal,
                        "set_kex_public op refused for %s, unexpectedly",
                        feda_peer_description(fp));
        return;
    }

    noncepart = payload + kex_public_size;
    mac = noncepart + cipher_nonce_used;
    ctext = payload + kex_public_size + cipher_nonce_used + cipher_mac_size;

    memset(nonce, 0, cipher_nonce_offset);
    memcpy(nonce + cipher_nonce_offset, noncepart, cipher_nonce_used);
    deobfuscate_buffer(nonce + cipher_nonce_offset, cipher_nonce_used);
    r = crypto_aead_unlock(ctext, mac, feda_peer_decrypt_key(fp),
                           nonce, NULL, 0,
                           ctext, payload_len - (ctext - payload));
    if(r != 0  /* sic! */) {  /* decrypting failed */
        servlog_message(srvl_normal, "failed decrypting echo reply from %s",
                                     feda_peer_description(fp));
        servlog_message(srvl_debug|srvl_private, "decrypt_key: %s",
                        hexdata2a(feda_peer_decrypt_key(fp), cipher_key_size)); 
        send_exchange_error(feda_rx, ip, port, fedaprot_err_decrypt);
        return;

        /* YYY do something on this! */
    }

    /* decrypted successfully! the last thing to check is the signature,
       but we can only do this in case we have the remote node:point
       in our ``known'' database; YYY actually, we'd better also check
       the timemark, our nonce and their nonce, but not now  */

    node_id = ctext + 18;
    point_p = ctext + 28;
    sign_p = ctext + 38;
    hexdata2str(nodeid_str, node_id, node_id_size);
    r = kndb_get_point_pubkey(feda_rx->kndb, node_id, *point_p,
                              remote_pubkey, NULL);
    if(r != kndb_res_success) {
        servlog_message(srvl_debug,
            "echo reply from %s, but we don't know their key (%s)",
            feda_peer_description(fp), kndb_result_message(r));

        if(r == kndb_res_point_unknown) {
            servlog_message(srvl_debug, "will request certs from %s",
                                        feda_peer_description(fp));
            send_introduction_request(feda_rx, fp);
            feda_peer_set_assocstatus(fp, fpas_intro_requested);
        } else {
            servlog_message(srvl_debug, "gave up, will not retry with %s",
                                        feda_peer_description(fp));
            feda_peer_set_assocstatus(fp, fpas_gave_up);
        }
        return;
    }
    r = crypto_eddsa_check(sign_p, remote_pubkey, payload, kex_public_size);
    if(r != 0 /* sic */) {
        servlog_message(srvl_normal,
            "echo repl. signature check failed (%s, %s.%d), will request cert",
            ipport2a(ip, port), nodeid_str, *point_p);

            /* this may be because of newer point key, request their cert
               just in case XXX */

        return;
    }

    /* now that we know for sure */

    r = feda_peer_set_identity(fp, node_id, *point_p, remote_pubkey);
    if(!r) {
        servlog_message(srvl_debug, "giving up association with %s",
                                    feda_peer_description(fp));
        feda_peer_set_assocstatus(fp, fpas_gave_up);
        send_exchange_error(feda_rx, ip, port, fedaprot_err_unexp_point);
        return;
    }

    servlog_message(srvl_debug, "sending assoc. request to %s",
                                feda_peer_description(fp));
    feda_peer_set_assocstatus(fp, fpas_assoc_request_sent);
    send_assoc_request(feda_rx, fp);
}

static void handle_stub_request(struct feda_udp_receiver *feda_rx,
                                unsigned int ip, unsigned short port)
{
    struct feda_peer *fp;

    fp = get_peer_rec(feda_rx->pir, ip, port, 1);
    feda_peer_inc_stubcred(fp);
}


static void handle_plain_error(struct feda_udp_receiver *feda_rx,
                               unsigned int ip, unsigned short port,
                               unsigned char *payload,
                               int payload_len)
{
    servlog_message(srvl_alert,
                    "peer %s says (as plain) we caused error (%02x, %s)",
                    ipport2a(ip, port), payload[0],
                    fedaprot_err_diags(payload[0]));
    /* not much else we can do here */
    if(payload[0] == fedaprot_err_decrypt) {
        struct feda_peer *fp;
        servlog_message(srvl_debug, "this means decrypting error");
        fp = get_peer_rec(feda_rx->pir, ip, port, 0);
        if(fp) {
            servlog_message(srvl_debug|srvl_private, "our encr. key: %s",
                hexdata2a(feda_peer_encrypt_key(fp), cipher_key_size));
            servlog_message(srvl_debug2, "our public kex key: %s",
                hexdata2a(feda_rx->comctx.kex_public, public_key_size));
            servlog_message(srvl_debug2, "their public kex key: %s",
                hexdata2a(feda_peer_remote_kex_pub(fp), public_key_size));
        } else {
            servlog_message(srvl_debug, "... but no peer record is found");
        }
    }
}





static void handle_natcheck_request(struct feda_udp_receiver *feda_rx,
                                    unsigned int ip, unsigned short port,
                                    const unsigned char *kex_public,
                                    int payload_len)
{
    /* we don't check the payload_len, because dgrams shorter than 128
       are rejected earlier, and we only use the kex_public, which is
       way shorter than 128 */

    int r, i;
    struct feda_peer *fp;
    struct feda_peer *siblings[10]; 
    int sibcnt;

    fp = get_peer_rec(feda_rx->pir, ip, port, 1);
    r = peer_set_kex_public(feda_rx->pir, fp, kex_public, 0 /* no sign. */);
    if(!r) {
        send_exchange_error(feda_rx, ip, port, fedaprot_err_kex_refused);
        return;
    }

    choose_peers_for_natcheck(feda_rx->pir, feda_peer_stub_credit(fp),
                              siblings, &sibcnt);

    servlog_message(srvl_normal, "starting natcheck proc. for %s (%d sibls)",
                                 feda_peer_description(fp), sibcnt);

    send_natcheck_response(feda_rx, fp, siblings, sibcnt);

    for(i = 0; i < sibcnt; i++)
        send_encrypted_please_contact(feda_rx, fp, siblings[i]);
}

static void handle_echo_request(struct feda_udp_receiver *feda_rx,
                                unsigned int ip, unsigned short port,
                                const unsigned char *body, int len)
{
    int r;
    struct feda_peer *fp;
    fp = get_peer_rec(feda_rx->pir, ip, port, 1);
    r = peer_set_kex_public(feda_rx->pir, fp, body, 0 /* no sign. here */);
    servlog_message(srvl_normal, "echo request from %s, will respond",
                                 ipport2a(ip, port));
    if(!r) {
        unsigned char encrypt_key[cipher_key_size];
        servlog_message(srvl_normal,
            "refused to replace kex_public for %s on echo request",
            feda_peer_description(fp));
        /* we'll continue anyway -- for the case we still have a valid
           association, but our peer has restarted and now uses another
           kex keypair; however, we won't replace the kex public until
           we have a valid association request, or else someone might
           spoof the addr/port of our peer and break the association
           by fooling us into replacing the key */

        derive_cipher_keys(feda_rx->comctx.kex_secret,
                           feda_rx->comctx.kex_public, 
                           body, encrypt_key, NULL);
        send_echo_response(feda_rx, fp, body + kex_public_size, encrypt_key);
        return;
    }
    send_echo_response(feda_rx, fp, body + kex_public_size, NULL);
}


static void handle_plain_dgram(struct feda_udp_receiver *feda_rx,
                               unsigned int ip, unsigned short port,
                               unsigned char *dgram, int len)
{
    unsigned char cmd;

    cmd = get_plain_dgram_cmd(dgram);
    servlog_message(srvl_debug2, "plain dgram; cmd %02x", cmd);
    switch(cmd) {
    case fedaprot_intro_req:
        handle_intro_request(feda_rx, ip, port, len-2);
        break;
    case fedaprot_i_am:
        handle_introduction_dgram(feda_rx, ip, port, dgram+2, len-2);
        break;
    case fedaprot_associate:
        handle_associate_request(feda_rx, ip, port, dgram+2, len-2);
        break;
    case fedaprot_change_key:
        handle_change_key_request(feda_rx, ip, port, dgram+2, len-2);
        break;
    case fedaprot_stub:
        handle_stub_request(feda_rx, ip, port);
        break;
    case fedaprot_test:
        handle_natcheck_request(feda_rx, ip, port, dgram+2, len-2);
        break;
    case fedaprot_echo_req:
        handle_echo_request(feda_rx, ip, port, dgram+2, len-2);
        break;
    case fedaprot_echo_reply:
        handle_echo_reply(feda_rx, ip, port, dgram+2, len-2);
        break;
    case fedaprot_error:
        handle_plain_error(feda_rx, ip, port, dgram+2, len-2);
        break;
    case fedaprot_test_reply:
    case fedaprot_forward:
        send_exchange_error(feda_rx, ip, port, fedaprot_err_unexpected);
        break;
    default:
        send_exchange_error(feda_rx, ip, port, fedaprot_err_unknown);
        break;
    }
}


static void handle_cr_keepalive(struct feda_udp_receiver *feda_rx,
                                struct feda_peer *fp)
{
    servlog_message(srvl_debug, "keepalive from %s",
                                feda_peer_description(fp));

    /* nothing else to do; time and state are already updated by
       handle_encrypted_dgram */
}

static void handle_cr_error(struct feda_udp_receiver *feda_rx,
                            struct feda_peer *fp,
                            unsigned char *payload,
                            int payload_len)
{
    servlog_message(srvl_alert, "peer %s says we caused error (code=%02x)",
                                feda_peer_description(fp), payload[0]);

    /* not much else we can do here */
}

static void handle_cr_plzcont(struct feda_udp_receiver *feda_rx,
                              struct feda_peer *fp,
                              unsigned char *addrport)
{
    unsigned int ip;
    unsigned short port;
    mem2ipport(addrport, &ip, &port);
    servlog_message(srvl_debug, "got please_contact, will send a stub to %s",
                                ipport2a(ip, port));
   
    send_plaintext_128(feda_rx, ip, port, fedaprot_stub, NULL, 0);
}

/* payload here starts with the byte #25 */
static void handle_cr_nodect_q(struct feda_udp_receiver *feda_rx,
                               struct feda_peer *fp,
                               unsigned char *payload,
                               int payload_len)
{
    unsigned char *cookie;
    unsigned char *node_id;
    int rank;
    unsigned char pubkey[public_key_size];
    unsigned char hash[yespower_hash_size];
    unsigned char hsign[signature_size];
    int r;

    cookie = payload + 1;
    node_id = payload + 5;

    r = kndb_get_node(feda_rx->kndb, node_id, &rank, pubkey, hash, hsign);

    servlog_message(srvl_debug,
                    "peer %s wants the %s node cert [q=%x], which we%s know",
                    feda_peer_description(fp),
                    hexdata2a(payload+5, node_id_size),
                    u32_from_big_endian(cookie),
                    r == kndb_res_success ? "" : " don't");

    if(r == kndb_res_success)
        send_encrypted_nodect_re(feda_rx, fp, cookie, node_id,
                                 rank, pubkey, hash, hsign);
    else
        send_encrypted_nodect_neg(feda_rx, fp, cookie, node_id);
}

/* payload here starts with the byte #25 */
static void handle_cr_nodect_re(struct feda_udp_receiver *feda_rx,
                                struct feda_peer *fp,
                                unsigned char *payload,
                                int payload_len)
{
    servlog_message(srvl_debug, "peer %s has sent us the cert for node %s",
                                feda_peer_description(fp),
                                hexdata2a(payload+5, node_id_size));

    /* XXX we're not ready to do anything on this;
       we need first to modify the known cert database so that it can
       hold unverified node certs, telling them from verified ones,
       and hold the information on which keys we trusted and when, so
       that the cert is in use despite we never took the hash of it
     */

//#error

}

static void handle_cr_pxy_outb(struct feda_udp_receiver *feda_rx,
                               struct feda_peer *fp,
                               unsigned char *payload,
                               int payload_len)
{
    /* this must mean we act as a proxy, AND one of the peers who
       are proxy users sent us a dgram which we should send through
       the respective proxy port
     */

    unsigned int ip;
    unsigned short port;
    struct feda_proxy *pxy;

    pxy = feda_peer_get_proxy_object(fp);
    if(!pxy) {
        servlog_message(srvl_normal,
            "peer %s sent us a proxy_inbound dgram, but it isn't proxyuser",
            feda_peer_description(fp));
        send_cr_error(feda_rx, fp, fedaprot_err_wont_proxy);
        return;
    }

    mem2ipport(payload, &ip, &port);

    feda_proxy_send_outbound(pxy, ip, port, payload + 6, payload_len - 6);
}

static void handle_incoming_dgram(struct feda_udp_receiver *feda_rx,
                                  unsigned int ip, unsigned short port,
                                  unsigned char *dgram, int len);

static void handle_cr_pxy_inb(struct feda_udp_receiver *feda_rx,
                              struct feda_peer *fp,
                              unsigned char *payload,
                              int payload_len)
{
    /* normally, this means WE use proxy, the proxy received a dgram
       for us, wrapped it into its own encrypted dgram (see the
       handle_proxy_inbound_packet function) and sent to us; so,
       perhaps the first thing to check is if we really do use a proxy
       AND the peer which sent us the dgram IS really our proxy
     */

    unsigned int ip;
    unsigned short port;

    if(fp != feda_rx->the_proxy_peer) {
        servlog_message(srvl_normal,
            "peer %s sent us a proxy_inbound dgram, but %s",
            feda_peer_description(fp),
            feda_rx->the_proxy_peer ?
                "a different peer is configured as the proxy" :
                "we don't use a proxy (at all)");
        return;
    }

    mem2ipport(payload, &ip, &port);
    handle_incoming_dgram(feda_rx, ip, port, payload+6, payload_len-6);
}


static void handle_assocpeer_error(struct feda_udp_receiver *feda_rx,
                                   struct feda_peer *fp,
                                   int code)
{
    unsigned char payload[2];

    payload[0] = fedaprot_cr_error;
    payload[1] = code;
    servlog_message(srvl_debug,
                    "sending error notification (code=%02x) to %s",
                    code, feda_peer_description(fp));
    send_encrypted(feda_rx, fp, payload, sizeof(payload), 0);
}


static void deliver_nonlocal_packet(struct feda_udp_receiver *feda_rx,
                                    struct feda_peer *orig_peer,
                                    unsigned char *buf, int size)
{
    const unsigned char *node_id;
    unsigned char point;
    struct feda_peer *fp;

    node_id = buf + 26;
    point = buf[36];

    fp = find_dest_peer(feda_rx->the_map, node_id, point);
    if(fp == FEDA_DEST_NO_NODE || fp == FEDA_DEST_NO_NODE_CERT) {
        servlog_message(srvl_debug,
                        "dest %s:%02x unreachable, packet dropped",
                        hexdata2a(node_id, node_id_size), point);
        return;
    }
    if(fp == FEDA_DEST_NO_POINT || !fp) {
        servlog_message(srvl_debug,
                        "no peer for dest %s:%02x, packet dropped",
                        hexdata2a(node_id, node_id_size), point);
        return;
    }
    if(fp == FEDA_DEST_IS_LOCAL) {
        if(!feda_rx->the_tun) {
            servlog_message(srvl_debug,
                            "%s:%02x is local, but no tun_iface; dropped",
                            hexdata2a(node_id, node_id_size), point);
            return;
        }
        push_packet_locally(feda_rx->the_tun, buf, size);
        return;
    }
    if(fp == orig_peer) {
        servlog_message(srvl_debug,
                        "dest %s:%02x: refused to send back to %s",
                        hexdata2a(node_id, node_id_size), point,
                        feda_peer_description(fp));
        return;
    }

    send_packet_to_peer(feda_rx, fp, 0xF083A8D0 /*forward*/, buf, size);
}

static void handle_cr_data(struct feda_udp_receiver *feda_rx,
                           struct feda_peer *fp,
                           unsigned char *payload,
                           int payload_len)
{
    static unsigned char pktbuf[4096];
    int res;

    servlog_message(srvl_debug2, "data from %s [%s...]",
                    feda_peer_description(fp), hexdata2a(payload, 5));

    if(!feda_rx->the_tun && !feda_rx->the_map) {
        /* we don't have any sink for these data, neither a local
           tun interface, nor a possibility to route the data further;
           in this version we simply drop the packets; perhaps we
           should do smth. more, like sending a ICMPv6 to the source,
           notifying the sending peer, etc              YYY YYY */

        return;
    }

    res = feda_peer_handle_traffic(fp, payload, payload_len,
                                   pktbuf, sizeof(pktbuf));
    if(!res)
        return;

    /* so we have a packet to dispatch */
    /* YYY YYY let's act as an open relay for now; this is definitely to be
       changed once we have automated credentials exchange */

    deliver_nonlocal_packet(feda_rx, fp, pktbuf, res);
}


static void handle_encrypted_dgram(struct feda_udp_receiver *feda_rx,
                                   unsigned int ip, unsigned short port,
                                   unsigned char *dgbuf, int len)
{
    int r;
    struct feda_peer *fp;
    unsigned char nonce[cipher_nonce_total];
    unsigned char *mac = dgbuf + cipher_nonce_used;
    unsigned char *ct = mac + cipher_mac_size;
    int ctlen = len - cipher_mac_size - cipher_nonce_used;

    fp = get_peer_rec(feda_rx->pir, ip, port, 0);
    if(!fp) {
        servlog_message(srvl_info,
            "got encrypted dgram from %s which is unknown for us",
            ipport2a(ip, port));
#if 0
        send_exchange_error(feda_rx, ip, port, fedaprot_err_no_assoc);
#else
        send_change_key(feda_rx, ip, port, nonce + cipher_nonce_offset);
#endif
        return;

        /* perhaps we should do something more sophisticated here, e.g.,
           remember the peer, count the packets, and on a certain threshold
           send the address to cooldown; well, not now */
    }

    memset(nonce, 0, cipher_nonce_offset);
    memcpy(nonce + cipher_nonce_offset, dgbuf, cipher_nonce_used);
    deobfuscate_buffer(nonce + cipher_nonce_offset, cipher_nonce_used);
        /* YYY at this moment, we might want to check the remote nonce and
         * refuse to work in case it is not greater than the previous one,
         * while allowing some gap for dgrams delivered out of the order
         */

    r = crypto_aead_unlock(ct, mac, feda_peer_decrypt_key(fp), nonce, NULL, 0,
                           ct, ctlen);
    if(r != 0  /* sic */) {
        servlog_message(srvl_info, "failed to decrypt dgram from %s",
                                   feda_peer_description(fp));
        servlog_message(srvl_debug|srvl_private, "decrypt_key: %s",
            hexdata2a(feda_peer_decrypt_key(fp), cipher_key_size));
#if 0
        send_exchange_error(feda_rx, ip, port, fedaprot_err_decrypt);
#else
        send_change_key(feda_rx, ip, port, nonce + cipher_nonce_offset);
#endif
        return;
    }

    update_peer_last_rx(fp);

    switch(ct[0]) {
    case fedaprot_cr_keepalive:
        handle_cr_keepalive(feda_rx, fp);
        break;
    case fedaprot_cr_plzcont:
        handle_cr_plzcont(feda_rx, fp, ct + 2);
        break;
    case fedaprot_cr_error:
        handle_cr_error(feda_rx, fp, ct + 1, ctlen - 1);
        break;
    case fedaprot_cr_nodect_q:
        handle_cr_nodect_q(feda_rx, fp, ct + 1, ctlen - 1);
        break;
    case fedaprot_cr_nodect_re:
        handle_cr_nodect_re(feda_rx, fp, ct + 1, ctlen - 1);
        break;
    case fedaprot_cr_pxy_outb:
        handle_cr_pxy_outb(feda_rx, fp, ct + 1, ctlen - 1);
        break;
    case fedaprot_cr_pxy_inb:
        handle_cr_pxy_inb(feda_rx, fp, ct + 1, ctlen - 1);
        break;
    default:
        if((ct[0] & 0xf0) == 0xd0)
            handle_cr_data(feda_rx, fp, ct, ctlen);
        else
            handle_assocpeer_error(feda_rx, fp, fedaprot_err_unknown);
    }

    /* now we have a successfully decrypted datagram, so we know at least
       that our peer is still active and we know for sure we still have
       a cryptografically established association with it */
    if(feda_peer_assoc_status(fp) != fpas_established) {
        unsigned char node_id[node_id_size];
        int point;
        feda_peer_get_point(fp, node_id, &point);
        feda_peer_set_assocstatus(fp, fpas_established);
        set_destination_peer(feda_rx->the_map, node_id, point, fp);
        servlog_message(srvl_info,
                        "association with %s established (!)",
                        feda_peer_description(fp));
    }
}





/* ---------------------------------------------------------------
   EVENT HANDLERS
   --------------------------------------------------------------- */

static void handle_incoming_dgram(struct feda_udp_receiver *feda_rx,
                                  unsigned int ip, unsigned short port,
                                  unsigned char *dgram, int len)
{
    if(len < fedaprot_min_dgram || len > fedaprot_max_dgram) {
        /* XXX if it is received through proxy, we should ask the proxy
           to cooldown the host to save some traffic... well, not now
         */
        servlog_message(srvl_debug,
                        "Invalid dgram size (%d), cooling down %s",
                        len, ipport2a(ip, 0));
        set_to_cooldown(feda_rx->pir, ip, 0);
        return;
    }
    if(is_cooldown(feda_rx->pir, ip, 0)) {
        servlog_message(srvl_debug, "host %s is cooled down", ipport2a(ip, 0));
        return;
    }
    if(dgram[0] >= fedaprot_zb_minplain && dgram[0] <= fedaprot_zb_maxplain) {
        handle_plain_dgram(feda_rx, ip, port, dgram, len);
        return;
    }
    /* must be an encrypted one then */
    if(dgram[0] >= fedaprot_zb_minenc && dgram[0] <= fedaprot_zb_maxenc)
        handle_encrypted_dgram(feda_rx, ip, port, dgram + 1, len - 1);
    else
        handle_encrypted_dgram(feda_rx, ip, port, dgram, len);

}

static void the_fd_handler_read(struct sue_fd_handler *h)
{
    unsigned char buf[2048];
    struct sockaddr_in sa;
    socklen_t sln;
    int rc;
    struct feda_udp_receiver *feda_rx = h->userdata;
    unsigned int ip;
    unsigned short port;

    servlog_message(srvl_debug2, "the_fd_handler_read called");

    sln = sizeof(sa);
    rc = recvfrom(h->fd, buf, sizeof(buf), 0, (struct sockaddr*)&sa, &sln);
    if(rc < 0) {
        servlog_perror(srvl_alert, "recvfrom", "rx_socket");
        servlog_message(srvl_alert, "recvfrom returned -1");
        return;
    }
    ip = ntohl(sa.sin_addr.s_addr);
    port = ntohs(sa.sin_port);
    servlog_message(srvl_debug2|srvl_private,
                    "UDP (%d bytes) from %s [%s...]",
                    rc, ipport2a(ip, port), hexdata2a(buf, 5));

    handle_incoming_dgram(feda_rx, ip, port, buf, rc);
}

static void send_through_proxy(struct feda_udp_receiver *feda_rx,
                               struct feda_transmit_item *item)
{
    unsigned char payload[2048];
    int mlen;

    mlen = item->len - item->offset + 7;
    if(mlen > sizeof(payload)) {
        servlog_message(srvl_alert,
            "dgram too long, can't send through the proxy (%d)", mlen);
        return;
    }
    payload[0] = fedaprot_cr_pxy_outb;
    ipport2mem(payload+1, item->ip, item->port);    
    memcpy(payload+7, item->buf + item->offset, item->len - item->offset);
    send_encrypted(feda_rx, feda_rx->the_proxy_peer, payload, mlen, 1);
                                                          /* 1 == direct */
}

static void the_fd_handler_write(struct sue_fd_handler *h)
{
    struct feda_udp_receiver *feda_rx = h->userdata;
    struct feda_transmit_item *item;

    servlog_message(srvl_debug2, "the_fd_handler_write called");

    item = fetch_item_to_transmit(feda_rx->txq);
    if(!item)  /* diags already logged */
        return;

    if(feda_rx->the_proxy_peer && !item->direct) {
        send_through_proxy(feda_rx, item);
    } else {
        servlog_message(srvl_debug2, "message for %s will go directly",
                                     ipport2a(item->ip, item->port));
        perform_send_to(feda_rx->fdh.fd, item->ip, item->port,
                        item->buf + item->offset,
                        item->len - item->offset);
    }
    feda_txitem_sent(item);
}

static void the_fd_handler(struct sue_fd_handler *h, int r, int w, int x)
{
    struct feda_udp_receiver *feda_rx = h->userdata;

    servlog_message(srvl_debug2, "the_fd_handler called (%s)(%s)",
                                 r ? "r" : "-", w ? "w" : "-");

    if(r)
        the_fd_handler_read(h);
    if(w)
        the_fd_handler_write(h);
    if(x)  /* we don't handle any exceptional conditions on the socket */
        servlog_message(srvl_debug, "the_fd_handler: x is true? strange");

    h->want_read = 1;
    h->want_write = feda_txq_want_write(feda_rx->txq);
    h->want_except = 0;
}


/* It is possible we'd better move the t/o handler object to the 
   peer collection (fsrv_pir module, feda_pir_collection structure),
   because, as a matter of fact, fsrv_pir is the only "user" of the
   hook (if we don't count the updating of the fdh_want_* fields)
 */
static void the_timeout_hdl(struct sue_timeout_handler *hdl)
{
    struct feda_udp_receiver *feda_rx = hdl->userdata;

    servlog_message(srvl_debug2, "the_timeout_hdl called");

    feda_pir_timer_hook(feda_rx->pir);

    feda_rx->fdh.want_read = 1;
    feda_rx->fdh.want_write = feda_txq_want_write(feda_rx->txq);
    feda_rx->fdh.want_except = 0;

    sue_timeout_set_from_now(hdl, fsrv_housekeeping_interval, 0);
    sue_sel_register_timeout(feda_rx->the_selector, &feda_rx->tmoh);
}


/* ---------------------------------------------------------------
   INITIALIZATION
   --------------------------------------------------------------- */


struct feda_udp_receiver *
make_udp_receiver(struct sue_event_selector *s, struct server_conf_info *conf)
{
    int r;
    struct feda_udp_receiver *res;
    char nodeid_str[node_id_size * 2 + 1];

    res = malloc(sizeof(*res));
    res->fdh.fd = -1;
    res->fdh.want_read = 1;
    res->fdh.want_write = 0;
    res->fdh.want_except = 0;
    res->fdh.userdata = res;
    res->fdh.handle_fd_event = &the_fd_handler;
    res->tmoh.userdata = res;
    res->tmoh.handle_timeout = &the_timeout_hdl;
    res->the_selector = s;
    res->the_config = conf;
    res->the_map = NULL;
    res->the_proxyset = NULL;

    r = comctx_init(&res->comctx);
    if(!r) {
        servlog_message(srvl_alert, "problems initializing crypto comm ctx");
        free(res);
        return NULL;
    }
    r = cpmctx_init_point(&res->comctx, conf->keys_dir);
    if(!r) {
        servlog_message(srvl_alert, "can't work without a deployed point");
        free(res);
        return NULL;
    }
    if(res->comctx.point->point == 0) {
        servlog_message(srvl_alert, "I'm not allowed to run as ZeroPoint!");
        free(res->comctx.point);
        free(res);
        return NULL;
    }

    servlog_message(srvl_debug, "kex pub key: %s",
                hexdata2a(res->comctx.kex_public, kex_public_size));
    servlog_message(srvl_debug, "point pub key: %s",
                hexdata2a(res->comctx.point->public_key, public_key_size));

    res->kndb = make_kndb(conf->keys_dir, res->comctx.point->minpref);

    hexdata2str(nodeid_str, res->comctx.point->node_id, node_id_size);
    servlog_message(srvl_info, "node_id %s point %d",
                    nodeid_str, res->comctx.point->point);

    res->txq = make_transmit_queue(s);

    res->the_proxy_peer = NULL;
    res->the_nodenets_peer = NULL;

        /* this should be the very last step of the construction as
           make_pir_collection affects the *res object heavily
         */
    res->pir = make_pir_collection(res, conf, &res->comctx);

    return res;
}

int launch_udp_receiver(struct feda_udp_receiver *feda_rx)
{
    int sfd, res;
    struct sockaddr_in sa;

    sfd = socket(AF_INET, SOCK_DGRAM, 0);

    if(sfd == -1) {
        servlog_perror(srvl_alert, "socket", NULL);
        return 0;
    }

#if 0  /* no need for this, it's not TCP, doesn't have that TIME_WAIT */
    opt = 1;
    setsockopt(sfd, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));
#endif

    sa.sin_family = AF_INET;
    sa.sin_addr.s_addr = htonl(feda_rx->the_config->listen_address);
    sa.sin_port = htons(feda_rx->the_config->listen_port);

    res = bind(sfd, (const struct sockaddr*)&sa, sizeof(sa));
    if(res == -1) {
        servlog_perror(srvl_alert, NULL, "bind");
        close(sfd);
        return 0;
    }

    /* success */

    feda_rx->fdh.fd = sfd;
    feda_rx->fdh.want_read = 1;
    sue_sel_register_fd(feda_rx->the_selector, &feda_rx->fdh);

    sue_timeout_set_from_now(&feda_rx->tmoh, fsrv_housekeeping_start_delay, 0);
    sue_sel_register_timeout(feda_rx->the_selector, &feda_rx->tmoh);

    return 1;
}

void udp_receiver_set_tun(struct feda_udp_receiver *feda_rx,
                          struct feda_tunnel_gate *tun,
                          struct feda_destination_map *destmap)
{
    feda_rx->the_tun = tun;
    feda_rx->the_map = destmap;

        /* definitely we know our point, as it is configured in the
           make_udp_receiver function, much earlier... the object can't
           exist without it */

    set_destination_peer(feda_rx->the_map,
                         feda_rx->comctx.point->node_id,
                         feda_rx->comctx.point->point,
                         FEDA_DEST_IS_LOCAL);

    switch(feda_rx->the_config->nodenets) {
    case nodenets_none:
        break;
    case nodenets_some:
        destmap_set_nodenets_peer(feda_rx->the_map, FEDA_DEST_IS_LOCAL, 0);
        break;
    case nodenets_all:
        destmap_set_nodenets_peer(feda_rx->the_map, FEDA_DEST_IS_LOCAL, 1);
        break;
    }
}

void udp_receiver_set_pxyset(struct feda_udp_receiver *feda_rx,
                             struct feda_proxy_set *pxyset)
{
    feda_rx->the_proxyset = pxyset;
    feda_pir_set_pxyset(feda_rx->pir, pxyset);
}


/* ---------------------------------------------------------------
   HANDLE ASSOCIATION ESTABLISHING FOR A GIVEN PEER
   --------------------------------------------------------------- */

/* called only by the fsrv_pir module, but belongs more here not there */

void handle_association_process(struct feda_udp_receiver *feda_rx,
                                struct feda_peer *fp)
{
    int since_last_rx, since_last_tx;
    unsigned int ip;
    unsigned short port;

    if(!can_send_to(feda_rx, fp))
        return;

    feda_peer_get_idle(fp, &since_last_rx, &since_last_tx);
    feda_peer_getaddr(fp, &ip, &port);

    servlog_message(srvl_debug2, "peer %s since_last_rx/tx %d/%d",
                    ipport2a(ip, port), since_last_rx, since_last_tx);

    switch(feda_peer_assoc_status(fp)) {
    case fpas_not_desired:
    case fpas_gave_up:
        return; /* should never happen, we aren't called in these states */
    case fpas_none:
        if(!feda_peer_should_initiate_assoc(fp))
            return;
#if 0
        if(since_last_rx < since_last_tx || since_last_tx < min_retry_time)
            return;
#endif
        send_echo_request(feda_rx, fp);
        feda_peer_set_assocstatus(fp, fpas_echo_request_sent);
        return;
    case fpas_echo_request_sent:
        if(since_last_rx < since_last_tx || since_last_tx < min_retry_time)
            return;
        send_echo_request(feda_rx, fp);
        return;
    case fpas_intro_requested:
        if(since_last_rx < since_last_tx || since_last_tx < min_retry_time)
            return;
        send_introduction_request(feda_rx, fp);
        return;
    case fpas_assoc_request_sent:
        if(since_last_rx < since_last_tx || since_last_tx < min_retry_time)
            return;
        servlog_message(srvl_debug, "resending the assoc. request");
        send_assoc_request(feda_rx, fp);
        return;
    case fpas_established:
        if(since_last_rx > feda_rx->the_config->peer_timeout) {
            unsigned char node_id[node_id_size];
            int point;

            feda_peer_get_point(fp, node_id, &point);
            servlog_message(srvl_info, "association with %s seems dead",
                                       ipport2a(ip, port));
            feda_peer_set_assocstatus(fp, fpas_none);
            set_destination_peer(feda_rx->the_map, node_id, point, NULL);
            return;
        }
        if(since_last_tx > feda_rx->the_config->keepalive_interval)
            send_keepalive_dgram(feda_rx, fp);
    }
}

/* ---------------------------------------------------------------
   HANDLING PACKETS RECEIVED BY PROXY PORTS (INBOUND PXY TRAFFIC)
   --------------------------------------------------------------- */

void
handle_proxy_inbound_packet(struct feda_udp_receiver *feda_rx,
                            struct feda_peer *fp_for,
                            unsigned int from_ip, unsigned short from_port,
                            unsigned char *buf, int offset, int msglen)
{
    unsigned char *payload;
    enum { hdrinfolen = 7 };

    if(offset < hdrinfolen) {  /* bug, this must never happen */
        servlog_message(srvl_alert,
            "handle_proxy_inbound_packet called with offset too short (BUG)");
        return;
    }

    if(feda_peer_assoc_status(fp_for) != fpas_established) {
        servlog_message(srvl_info,
            "the peer using the proxy isn't associated, dropping the dgram");
        return;
    }

    payload = buf + (offset - hdrinfolen);
    *payload = fedaprot_cr_pxy_inb;
    ipport2mem(payload+1, from_ip, from_port);    
    send_encrypted(feda_rx, fp_for, payload, hdrinfolen + msglen, 0);
}


/* ---------------------------------------------------------------
   SENDING OUTGOING PACKETS TO A GIVEN PEER
   --------------------------------------------------------------- */

    /* the TTL / hop limit is decremented already */
void send_packet_to_peer(struct feda_udp_receiver *feda_rx,
                         struct feda_peer *fp, unsigned int packid,
                         const unsigned char *buf, int size)
{
    static unsigned char payload[512];  /* slightly more than we need */
    int dgrams, dgpsize, last_dgpsize, cmdbyte, i;
    unsigned int seq;

    if(feda_peer_get_unsent(fp) >= max_unsent_per_peer) {
        servlog_message(srvl_debug, "[%08x] packet dropped due to digestion",
                                    packid);
        return;
    }

    dgrams = (size-1) / 448 + 1;
    if(dgrams > 4) {
        servlog_message(srvl_alert, "[%08x] %d B is too big (it's a bug)",
                                    packid, size);
        return;
    }
    dgpsize = size / dgrams;
    last_dgpsize = size - dgpsize*(dgrams-1);
    cmdbyte = 0xD0 | (((dgrams-1) & 0x03) << 2);
    seq = feda_peer_get_sequential(fp);

    servlog_message(srvl_debug2, "[%08x] sending with seq. %d cmdbyte %02X",
                                 packid, seq, cmdbyte);

    payload[1] = (seq >> 16) & 0xff;
    payload[2] = (seq >>  8) & 0xff;
    payload[3] = seq         & 0xff;
    for(i = 0; i < dgrams; i++) {
        int pldsz = i == dgrams-1 ? last_dgpsize : dgpsize;
        payload[0] = (cmdbyte & 0xfc) | (i & 0x03);
        memcpy(payload+4, buf + i * dgpsize, pldsz);
        send_encrypted(feda_rx, fp, payload, pldsz + 4, 0);
    }

    feda_rx->fdh.want_write = 1;
}


/* ---------------------------------------------------------------
   STATUS REPORTS
   --------------------------------------------------------------- */

static void do_rx_report(struct feda_udp_receiver *feda_rx,
                         report_callback f, void *userdata)
{
    f(userdata, "I'm running as %s.%d",
                hexdata2a(feda_rx->comctx.point->node_id, node_id_size),
                feda_rx->comctx.point->point);
    if(feda_rx->the_proxy_peer)
        f(userdata, "using proxy %s\n",
                    feda_peer_description(feda_rx->the_proxy_peer));
    if(feda_rx->the_proxyset)
        feda_proxy_set_report(feda_rx->the_proxyset, f, userdata);

    feda_pir_report(feda_rx->pir, f, userdata);
}

void feda_udp_receiver_report(struct feda_udp_receiver *feda_rx)
{
    int level = srvl_normal | srvl_private;
    servlog_message(srvl_normal, "Got SIGUSR1");
    do_rx_report(feda_rx, report_to_log_cb, &level);
}

void feda_udp_receiver_streamrep(struct feda_udp_receiver *feda_rx,
                                 void *stream)
{
    do_rx_report(feda_rx, report_to_stream_cb, stream);
}


/* ---------------------------------------------------------------
   SOME GETTERS
   --------------------------------------------------------------- */

const struct point_config_file *
feda_rx_get_point_cfg(struct feda_udp_receiver *feda_rx)
{
    return feda_rx->comctx.point;
}

struct feda_pir_collection *feda_rx_get_pir(struct feda_udp_receiver *feda_rx)
{
    return feda_rx->pir;
}

struct feda_destination_map *
feda_rx_get_destmap(struct feda_udp_receiver *feda_rx)
{
    return feda_rx->the_map;
}
