#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <sue/sue_base.h>

#include "hexdata.h"
#include "addrport.h"
#include "servlog.h"
#include "servlog.h"
#include "comcrypt.h"
#include "fsrv_cfg.h"
#include "fsrv_txq.h"
#include "fsrv_pir.h"
#include "fsrv_rx.h"

#include "fsrv_pxy.h"


struct feda_proxy {
    struct feda_proxy_set *the_master;
    struct peer_conf *the_peerconf;
    struct feda_peer *the_peer;
    struct sue_fd_handler fdh;
    int port;
    struct feda_transmit_queue *txq;
    struct feda_proxy *next;
};

struct feda_proxy_set {
    struct sue_event_selector *the_selector;
    struct feda_udp_receiver *the_feda_rx;
    struct feda_pir_collection *the_pir;
    struct server_conf_info *the_conf;
    struct feda_proxy *first;
};



static void pxy_fd_handler_read(struct feda_proxy *pxy)
{
    unsigned char buf[2048];
    int offset = 8;
    struct sockaddr_in sa;
    socklen_t sln;
    int rc;
    unsigned int ip;
    unsigned short port;

    servlog_message(srvl_debug2, "pxy_fd_handler_read called");

    sln = sizeof(sa);
    rc = recvfrom(pxy->fdh.fd, buf + offset, sizeof(buf) - offset,
                  0, (struct sockaddr*)&sa, &sln);
    if(rc < 0) {
        servlog_perror(srvl_alert, "recvfrom", "proxy");
        servlog_message(srvl_alert, "recvfrom returned -1");
        return;
    }
    ip = ntohl(sa.sin_addr.s_addr);
    port = ntohs(sa.sin_port);
    servlog_message(srvl_debug2|srvl_private,
                    "UDP @port %d (%d bytes) from %s [%s...]",
                    pxy->port, rc, ipport2a(ip, port),
                    hexdata2a(buf + offset, 5));
    if(rc < fedaprot_min_dgram || rc > fedaprot_max_dgram) {
        servlog_message(srvl_debug,
                        "Invalid dgram size (%d), cooling down %s",
                        rc, ipport2a(ip, 0));
        set_to_cooldown(pxy->the_master->the_pir, ip, 0);
        return;
    }
    if(is_cooldown(pxy->the_master->the_pir, ip, 0)) {
        servlog_message(srvl_debug, "host %s is cooled down", ipport2a(ip, 0));
        return;
    }
    if(!pxy->the_peer) {
        servlog_message(srvl_debug,
            "proxy @port %d: no peer, dropping packet", pxy->port);
        return;
    }
    handle_proxy_inbound_packet(pxy->the_master->the_feda_rx, pxy->the_peer,
                                ip, port, buf, offset, rc);
}

static void pxy_fd_handler_write(struct feda_proxy *pxy)
{
    struct feda_transmit_item *item;

    servlog_message(srvl_debug2, "pxy_fd_handler_write called");

    item = fetch_item_to_transmit(pxy->txq);
    if(!item)  /* diags already logged */
        return;

    perform_send_to(pxy->fdh.fd, item->ip, item->port,
                    item->buf + item->offset,
                    item->len - item->offset);
    feda_txitem_sent(item);
}

static void pxy_fd_handler(struct sue_fd_handler *h, int r, int w, int x)
{
    struct feda_proxy *pxy = h->userdata;

    servlog_message(srvl_debug2, "pxy_fd_handler called (%s)(%s)",
                                 r ? "r" : "-", w ? "w" : "-");

    if(r)
        pxy_fd_handler_read(pxy);
    if(w)
        pxy_fd_handler_write(pxy);
    if(x)  /* we don't handle any exceptional conditions on the socket */
        servlog_message(srvl_debug, "pxy_fd_handler: x is true? strange");

    h->want_read = 1;
    h->want_write = feda_txq_want_write(pxy->txq);
    h->want_except = 0;
}


static struct feda_proxy *do_mk_proxy(struct feda_proxy_set *master,
                                      struct sue_event_selector *sel,
                                      struct server_conf_info *conf,
                                      struct peer_conf *peercfg)
{
    int sfd, r;
    struct sockaddr_in sa;
    struct feda_proxy *res;

    sfd = socket(AF_INET, SOCK_DGRAM, 0);

    if(sfd == -1) {
        servlog_perror(srvl_alert, "proxy port", "socket");
        return NULL;
    }

    if(peercfg->proxyport > 0) {   /* this condition might change in future */
        sa.sin_family = AF_INET;
        sa.sin_addr.s_addr = htonl(conf->listen_address);
        sa.sin_port = htons(peercfg->proxyport);
    
        r = bind(sfd, (const struct sockaddr*)&sa, sizeof(sa));
        if(r == -1) {
            servlog_perror(srvl_alert, "proxy port", "bind");
            close(sfd);
            return NULL;
        }
    }

    /* success */
    res = malloc(sizeof(*res));
    res->the_master = master;
    res->the_peerconf = peercfg;
    res->the_peer = NULL;
    res->fdh.fd = sfd;
    res->fdh.want_read = 1;
    res->fdh.want_write = 0;
    res->fdh.want_except = 0;
    res->fdh.userdata = res;
    res->fdh.handle_fd_event = &pxy_fd_handler;
    res->port = peercfg->proxyport;
    res->txq = make_transmit_queue(sel);
    res->next = NULL;
    sue_sel_register_fd(sel, &res->fdh);

    return res;
}


void feda_proxy_peer_gone(struct feda_proxy *pxy)
{
    pxy->the_peer = NULL;
}

void feda_proxy_set_peer(struct feda_proxy *pxy, struct feda_peer *fp)
{
    pxy->the_peer = fp;
}

/* ---------- proxy set methods -------------------- */


static struct feda_proxy_set *do_mk_proxyset(struct sue_event_selector *sel,
                                             struct feda_udp_receiver *rx,
                                             struct feda_pir_collection *pir,
                                             struct server_conf_info *conf)
{
    struct feda_proxy_set *res;

    res = malloc(sizeof(*res));
    res->the_selector = sel;
    res->the_feda_rx = rx;
    res->the_pir = pir;
    res->the_conf = conf;
    res->first = NULL;

    return res;
}


struct feda_proxy_set *make_proxy_set(struct sue_event_selector *sel,
                                      struct feda_udp_receiver *feda_rx,
                                      struct server_conf_info *conf)
{
    struct feda_proxy_set *res = NULL;
    struct peer_conf *p;

    for(p = conf->first_peer; p; p = p->next) {
        struct feda_proxy *pxy;

        if(!(p->type & ptp_proxyuser))
            continue;
        if(!res)
            res = do_mk_proxyset(sel, feda_rx, feda_rx_get_pir(feda_rx), conf);
        pxy = do_mk_proxy(res, sel, conf, p);
        if(!pxy)
            continue;
        pxy->next = res->first;
        res->first = pxy;
    }

    return res;
}

struct feda_proxy *find_feda_proxy_by_conf(const struct feda_proxy_set *ps,
                                           const struct peer_conf *cfg1,
                                           const struct peer_conf *cfg2)
{
    struct feda_proxy *p;

    if(!ps)
        return NULL;

    for(p = ps->first; p; p = p->next)
        if(p->the_peerconf == cfg1 || p->the_peerconf == cfg2)
            return p;

    return NULL;
}

void feda_proxy_send_outbound(struct feda_proxy *pxy,
                              unsigned int ip, unsigned short port,
                              const unsigned char *data, int len)
{
    struct feda_transmit_item *msg;

    msg = make_txitem_4ip(pxy->txq, len, 0, ip, port);
    memcpy(msg->buf, data, len);
    feda_txq_enqueue(msg);
    pxy->fdh.want_write = 1;
}


void feda_proxy_set_report(struct feda_proxy_set *ps,
                           report_callback f, void *userdata)
{
    struct feda_proxy *p;

    if(!ps)
        return;

    for(p = ps->first; p; p = p->next) {
        f(userdata, "proxy @%d fd=%d for %s", p->port, p->fdh.fd,
            p->the_peer ?
                feda_peer_description(p->the_peer) :
                p->the_peerconf->name);
    }
}
