#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "keyutils.h"
#include "addrport.h"
#include "hexdata.h"
#include "ip6struc.h"
#include "comcrypt.h"
#include "servlog.h"
#include "inadrcol.h"
#include "fsrv_cfg.h"
#include "fsrv_pad.h"
#include "fsrv_rx.h"
#include "fsrv_pxy.h"

#if 0
#include "fsrv_dst.h"
#endif

#include "fsrv_pir.h"


/* See the fsrv_pad.h file for explanation */
enum { defrag_seq_bits = 8 };


struct feda_peer {    /* created by feda_peer_init, exclusively */
                      /* destroyed exclusively by fedapeer_destruction_hook
                         so any ``destructor'' stuff should go there */
    struct feda_pir_collection *the_master;
    struct inaddr_item *the_item;

    unsigned int ip;
    unsigned short port;

    unsigned char node_id[node_id_size];
    int point;
    unsigned char remote_public[public_key_size];   /* sig.key, not kex */

    struct peer_conf *the_conf_by_ip, *the_conf_by_point;

    unsigned char remote_kex_pub[kex_public_size];
    unsigned char encrypt_key[cipher_key_size];
    unsigned char decrypt_key[cipher_key_size];

    long last_rx, last_tx;  /* last_tx updated on any sent dgram,
                               last_rx is only updated on successfully
                               decrypted cryptographic messages */
    char initiate_assoc;  /* boolean; should we initiate association pr. */
    int assoc_status;
    int stub_credit;
    int unsent_packs;
    unsigned int sequential;  /* unsigned 'cause it's intended to wrap */
    struct feda_peer_defrag *defrag_buf;

    struct feda_proxy *the_proxy;  /* the proxy object used by this peer */
};


struct peer_item {
    struct feda_peer *item;
    struct peer_item *next;
};

struct feda_pir_collection {
    struct server_conf_info *the_conf;
    struct feda_udp_receiver *the_feda_rx;
    struct feda_proxy_set *the_pxyset;
    struct crypto_comm_context *the_comctx;
#if 0
    struct feda_destination_map *the_destmap;
#endif
    struct inaddr_collection cooldown, peers;
    int natcheck_peer_count;
    struct feda_peer **natcheck_peers;
    struct peer_item *permpeer_first;
};

/*
   The cooldown collection holds addresses that sent us incorrect dgrams or
   are temporarily banned for other reasons.  The userdata and both hooks are
   always NULL for all the cooldown collection items.
   
   The peers collection contains data for IP/PORT pairs which either the
   server received any MEANINGFUL datagrams from, or they are configured as
   peers in the configuration file, or the server acted as a client to them.
   In each of the collection members, the userdata field points to a 
   feda_peer structure.
*/

static void fedapeer_timeout_hook(struct inaddr_item *item)
{
    unsigned int ip;
    unsigned short port;
    inaddritem_getaddr(item, &ip, &port);
    servlog_message(srvl_info, "peer %s timed out", ipport2a(ip, port));
    inaddritem_remove(item);
}

static void fedapeer_destruction_hook(struct inaddr_item *item)
{
    if(item->userdata) {
        struct feda_peer *fp = item->userdata;
        /* ``destructor'' stuff for the feda_peer structure goes here */
        servlog_message(srvl_debug, "removing peer %s",
                                    feda_peer_description(fp));
        if(fp->the_proxy)
            feda_proxy_peer_gone(fp->the_proxy);
        feda_rx_peer_gone(fp->the_master->the_feda_rx, fp);
        if(fp->defrag_buf)
            dispose_peer_defrag(fp->defrag_buf);
        free(item->userdata);
    } else {
        unsigned int ip;
        unsigned short port;
        inaddritem_getaddr(item, &ip, &port);
        servlog_message(srvl_debug, "removing peer %s (?!)",
                                    ipport2a(ip, port));
    }
}

static void
feda_peer_init(struct feda_pir_collection *pir, struct inaddr_item *peer)
{
    struct feda_peer *fp;
    unsigned int ip;
    unsigned short port;

    if(peer->userdata)
        return;

    inaddritem_getaddr(peer, &ip, &port);

    fp = malloc(sizeof(*fp));
    memset(fp, 0, sizeof(*fp));
    fp->the_master = pir;
    fp->the_item = peer;
    fp->ip = ip;
    fp->port = port;
    fp->point = -1;
    fp->last_rx = -1;
    fp->last_tx = -1;
    fp->initiate_assoc = 0;
    fp->assoc_status = fpas_not_desired;
    fp->unsent_packs = 0;
    fp->sequential = 0;
    fp->defrag_buf = NULL;
    fp->the_proxy = NULL;
    peer->userdata = fp;
    peer->timeout_hook = fedapeer_timeout_hook;
    peer->destruction_hook = fedapeer_destruction_hook;
}

static int feda_peer_identified(const struct feda_peer *fp)
{
    return !all_zeroes(fp->node_id, node_id_size) && fp->point != -1;
}

static int peer_has_type(const struct feda_peer *fp, int t)
{
    return
        (fp->the_conf_by_ip && (fp->the_conf_by_ip->type & t)) ||
        (fp->the_conf_by_point && (fp->the_conf_by_point->type & t));
}


    /* This function is called from within the feda_peer_set_identity
       func., that is, only upon establishing connectivity; further,
       it is only called in case for the peer object its pair of
       configuration records has just been changed.

       Peers may connect to us from different locations and perform
       certain service or play a kind of special role for us; this
       function is the place to settle all this stuff.
     */
static void feda_peer_reconfigure(struct feda_peer *fp)
{
    if(peer_has_type(fp, ptp_nodenets)) {
        servlog_message(srvl_debug,
            "Setting nodenets peer: %s", feda_peer_description(fp));
        feda_rx_set_nodenets_peer(fp->the_master->the_feda_rx, fp);
    }
    if(peer_has_type(fp, ptp_proxyuser)) {
        struct feda_proxy *pxy =
            find_feda_proxy_by_conf(fp->the_master->the_pxyset,
                                    fp->the_conf_by_ip,
                                    fp->the_conf_by_point);
        if(!pxy) {  /* must not happen! */
            servlog_message(srvl_alert,
                "peer %s conf'd as proxyuser but pxy obj not found (BUG)",
                feda_peer_description(fp));
            return;
        }
        fp->the_proxy = pxy;
        feda_proxy_set_peer(pxy, fp);
    }
}

int feda_peer_set_identity(struct feda_peer *fp,
                           const unsigned char *node_id, int point,
                           const unsigned char *pubkey)
{
    struct peer_conf *p;
    unsigned int ip = fp->ip;
    unsigned short port = fp->port;
    int reconf;

    if(feda_peer_identified(fp) &&
        (0 != memcmp(fp->node_id, node_id, node_id_size) || fp->point != point)
    ) {
        char nid_str[node_id_size * 2 + 1];
        char cur_nid_str[node_id_size * 2 + 1];
        hexdata2str(nid_str, node_id, node_id_size);
        hexdata2str(cur_nid_str, fp->node_id, node_id_size);
        servlog_message(srvl_normal,
            "for peer %s: refusing to replace %s.%d with %s.%d",
            ipport2a(ip, port), cur_nid_str, fp->point, nid_str, point);
        return 0;
    }

    reconf = 0;
    for(p = fp->the_master->the_conf->first_peer; p; p = p->next) {
        int have_ip, have_point, match_ip, match_point;

        have_ip = p->ip != PEER_IP_UNDEF;
        match_ip = ip == p->ip && (port == p->port || p->port == 0);
        have_point = peer_conf_has_point(p);
        match_point = have_point &&
            0 == memcmp(node_id, p->node_id, node_id_size) &&
            point == p->point;

        if(!match_ip && !match_point)
            continue;
        if(have_ip && match_ip && have_point && !match_point) {
                /* we have to refuse this peer! */
            char nid_str[node_id_size * 2 + 1];
            char conf_nid_str[node_id_size * 2 + 1];
            hexdata2str(nid_str, node_id, node_id_size);
            hexdata2str(conf_nid_str, p->node_id, node_id_size);
            servlog_message(srvl_normal,
                "refusing assoc. with %s: %s.%d mismatches our config %s.%d",
                ipport2a(ip, port), nid_str, point, conf_nid_str, p->point);
            return 0;
        }
        if(have_ip && match_ip && !fp->the_conf_by_ip) {
            fp->the_conf_by_ip = p;
            reconf = 1;
        }
        if(have_point && match_point && !fp->the_conf_by_point) {
            fp->the_conf_by_point = p;
            reconf = 1;
        }
    }
    if(reconf)
        feda_peer_reconfigure(fp);

    memcpy(fp->node_id, node_id, node_id_size);
    fp->point = point;
    if(pubkey)
        memcpy(fp->remote_public, pubkey, public_key_size);

#if 0  /* this is not our responsibility zone */
    set_destination_peer(fp->the_master->the_destmap, node_id, point, fp);
#endif
    return 1;
}


static void enlist_permpeer(struct feda_pir_collection *pir,
                            struct feda_peer *p)
{
    struct peer_item *tmp;
    tmp = malloc(sizeof(*tmp));
    tmp->item = p;
    tmp->next = pir->permpeer_first;
    pir->permpeer_first = tmp;
}

static void add_configured_peers(struct feda_pir_collection *pir)
{
    struct peer_conf *p;
    int i;

        /* first, just count the number of natcheck peers */
    pir->natcheck_peer_count = 0;
    for(p = pir->the_conf->first_peer; p; p = p->next) {
        if(p->type & ptp_natcheck) {
            if(p->port == 0 || p->ip == (unsigned int)(-1)) {
                servlog_message(srvl_normal,
                    "peer %s: natcheck on, but no ip/port; ignored",
                    p->name);
                continue;
            }
            pir->natcheck_peer_count++;
        }
    }

        /* now, create and register all peers (not only natcheck ones)
           as appropriate
         */
    pir->natcheck_peers =
        malloc(sizeof(*pir->natcheck_peers) * (pir->natcheck_peer_count+1));
    i = 0;  /* for natcheck array */
    for(p = pir->the_conf->first_peer; p; p = p->next) {
        struct inaddr_item *ia;
        struct feda_peer *fp;
        if(!peer_conf_has_ip(p))
            continue;
        ia = inaddrcoll_permadd(&pir->peers, p->ip, p->port);
        if(!ia->userdata)
            feda_peer_init(pir, ia);
        fp = ia->userdata;
        fp->the_conf_by_ip = p;
        if(peer_conf_has_point(p)) {
            fp->the_conf_by_point = p;
            memcpy(fp->node_id, p->node_id, node_id_size);
            fp->point = p->point;
        }

        fp->initiate_assoc = 0;
        fp->assoc_status = fpas_not_desired;
        enlist_permpeer(pir, fp);
        if(p->type & ptp_natcheck) {
            fp->initiate_assoc = 1;
            fp->assoc_status = fpas_none;
            pir->natcheck_peers[i] = fp;
            i++;
        }
        if(p->type & (ptp_mynode | ptp_proxy | ptp_persist)) {
            fp->initiate_assoc = 1;
            fp->assoc_status = fpas_none;
            if(p->type & ptp_proxy)
                feda_rx_set_proxy_peer(pir->the_feda_rx, fp);
        }
    }
    pir->natcheck_peers[i] = NULL;

    servlog_message(srvl_debug, "natcheck peers configured: %d (%d)",
                                pir->natcheck_peer_count, i);
}

struct feda_pir_collection *make_pir_collection(struct feda_udp_receiver *rx,
           struct server_conf_info *conf, struct crypto_comm_context *comctx)
{
    struct feda_pir_collection *p;
    long long tm;

    p = malloc(sizeof(*p));
    p->the_conf = conf;
    p->the_feda_rx = rx;
    p->the_pxyset = NULL;
    p->the_comctx = comctx;
#if 0
    p->the_destmap = NULL;
#endif

    tm = time(NULL);
    inaddrcoll_init(&p->cooldown, tm, conf->cooldown_timeout);
    inaddrcoll_init(&p->peers, tm, conf->peer_timeout);

    p->permpeer_first = NULL;

    add_configured_peers(p);

    return p;
}

#if 0
void feda_pir_set_map(struct feda_pir_collection *pir,
                      struct feda_destination_map *destmap)
{
    pir->the_destmap = destmap;
}
#endif

void feda_pir_set_pxyset(struct feda_pir_collection *pir,
                         struct feda_proxy_set *pxyset)
{
    pir->the_pxyset = pxyset;
}




void feda_pir_timer_hook(struct feda_pir_collection *pir)
{
    long long tm;
    int res;
    struct peer_item *p;

    tm = time(NULL);

    servlog_message(srvl_debug2, "feda_pir_timer_hook called, tm=%lld", tm);

    res = inaddrcoll_update_time(&pir->cooldown, tm);
    if(res)
        inaddrcoll_process(&pir->cooldown);
    res = inaddrcoll_update_time(&pir->peers, tm);
    if(res)
        inaddrcoll_process(&pir->peers);

    for(p = pir->permpeer_first; p; p = p->next) {
        struct feda_peer *fp = p->item;
        if(fp->assoc_status != fpas_not_desired &&
            fp->assoc_status != fpas_gave_up
        ) {
            handle_association_process(pir->the_feda_rx, p->item);
        }
    }
}

static const char *decimal2a(unsigned int n)
{
    static char res[16];
    char *p = res + (sizeof(res)-1);

    if(n == 0)  /* yes, special case */
        return "0";

    *p = 0;
    while(n > 0) {
        p--;
        *p = '0' + n % 10;
        n /= 10;
    }

    return p;
}

static void
single_peer_report(struct feda_peer *fp, long t, report_callback f, void *ud)
{
    f(ud, "[%s] %s", t == -1 ? "-" : decimal2a(t), feda_peer_description(fp));
}

void
feda_pir_report(struct feda_pir_collection *pir, report_callback f, void *ud)
{
    struct peer_item *p;
    struct inaddr_item *ia;

    for(p = pir->permpeer_first; p; p = p->next) {
        struct feda_peer *fp = p->item;
        single_peer_report(fp, -1, f, ud);
    }
    for(ia = pir->peers.first; ia; ia = ia->next) {
        struct feda_peer *fp = ia->userdata;
        single_peer_report(fp, pir->peers.curtime - ia->timemark, f, ud);
    }
    for(ia = pir->cooldown.first; ia; ia = ia->next) {
        f(ud, "cooldown [%d] %s",
              pir->cooldown.timeout - (pir->cooldown.curtime - ia->timemark),
              ipport_mem2a(ia->key));
    }
}

void update_peer_last_rx(struct feda_peer *fp)
{
    fp->last_rx = fp->the_master->peers.curtime;
}

void update_peer_last_tx(struct feda_peer *fp)
{
    fp->last_tx = fp->the_master->peers.curtime;
}

int feda_peer_get_point(const struct feda_peer *fp,
                        unsigned char *node_id, int *point)
{
    memcpy(node_id, fp->node_id, node_id_size);
    *point = fp->point;
    return !all_zeroes(fp->node_id, node_id_size);
}

int feda_peer_inc_unsent(struct feda_peer *fp)
{
    fp->unsent_packs++;
    return fp->unsent_packs;
}

int feda_peer_dec_unsent(struct feda_peer *fp)
{
    fp->unsent_packs--;
    return fp->unsent_packs;
}

int feda_peer_get_unsent(const struct feda_peer *fp)
{
    return fp->unsent_packs;
}

int feda_peer_inc_stubcred(struct feda_peer *fp)
{
    fp->stub_credit++;
    return fp->stub_credit;
}

int feda_peer_dec_stubcred(struct feda_peer *fp)
{
    fp->stub_credit--;
    return fp->stub_credit;
}

int feda_peer_stub_credit(const struct feda_peer *fp)
{
    return fp->stub_credit;
}

void feda_peer_getaddr(const struct feda_peer *fp,
                       unsigned *ip, unsigned short *port)
{
    *ip = fp->ip;
    *port = fp->port;
}

#if 0
const unsigned char *feda_peer_node_id(const struct feda_peer *fp)
{
    return fp->node_id;
}
#endif

const unsigned char *feda_peer_encrypt_key(const struct feda_peer *fp)
{
    return fp->encrypt_key;
}

const unsigned char *feda_peer_decrypt_key(const struct feda_peer *fp)
{
    return fp->decrypt_key;
}

const unsigned char *feda_peer_remote_kex_pub(const struct feda_peer *fp)
{
    return fp->remote_kex_pub;
}

const unsigned char *feda_peer_remote_public(const struct feda_peer *fp)
{
    return fp->remote_public;
}

struct feda_peer *get_peer_rec(struct feda_pir_collection *pir,
                               unsigned int ip, unsigned short port, int add)
{
    struct inaddr_item *p;
    p = inaddrcoll_find(&pir->peers, ip, port, add);
    if(!p)
        return NULL;
    if(!p->userdata) {    
        feda_peer_init(pir, p);
        servlog_message(srvl_info, "new peer %s", ipport2a(ip, port));
    }
    /* !!! the following block may be removed at any time, it's debug only */
    {
        unsigned int aip;
        unsigned short apt;
        mem2ipport(p->key, &aip, &apt);
        if(ip != aip || port != apt)
            servlog_message(srvl_debug, "get_peer_rec: %d!=%d or %d!=%d",
                            ip, aip, port, apt);
        else
            servlog_message(srvl_debug2, "peer %s object address %llx",
                            ipport2a(ip, port), (unsigned long long)p);
    }
    return p->userdata;
}

void set_to_cooldown(struct feda_pir_collection *pir,
                     unsigned long ip, unsigned short port)
{
    struct inaddr_item *cls, *p;

    cls = pir->cooldown.last;
    p = inaddrcoll_find(&pir->cooldown, ip, port, 1);
    if(cls == pir->cooldown.last)  /* it was not added */
        inaddritem_reset(p);
}

int is_cooldown(struct feda_pir_collection *pir,
                unsigned long ip, unsigned short port)
{
    struct inaddr_item *p;
    p = inaddrcoll_find(&pir->cooldown, ip, port, 0);
    return p != NULL;
}

int timemark_minutes(const struct feda_pir_collection *pir)
{
    return (pir->peers.starttime + (long long)pir->peers.curtime) / 60;
}

int feda_peer_is_direct(const struct feda_peer *fp)
{
    int flags = 0;
    if(fp->the_conf_by_ip)
        flags = fp->the_conf_by_ip->type;
    if(fp->the_conf_by_point)
        flags |= fp->the_conf_by_point->type;
    return flags & (ptp_direct | ptp_proxy);
}

void feda_peer_get_idle(const struct feda_peer *fp,
                        int *since_last_rx, int *since_last_tx)
{
    int curtime = fp->the_master->peers.curtime;
    *since_last_rx = curtime - fp->last_rx;
    *since_last_tx = curtime - fp->last_tx;
}

unsigned int feda_peer_get_sequential(struct feda_peer *fp)
{
    unsigned int res = fp->sequential;
    fp->sequential++;
    return res;
        /* yes, we DO know; for more details read
         * http://www.rebuildworld.net/taboo/coding_style.html#sideeffects
         */
}

    /* signchecked (boolean) means that the public key was signed by
       its owner and the signature has been checked
     */
int peer_set_kex_public(struct feda_pir_collection *pir, struct feda_peer *fp,
                        const unsigned char *kex_public, int signchecked)
{
    if(0 == memcmp(fp->remote_kex_pub, kex_public, kex_public_size))
        return 1;    /* nothing new, nothing to replace, that's good! */

    if(!signchecked && fp->assoc_status == fpas_established)
        return 0;    /* this means established cryptographic association */

    memcpy(fp->remote_kex_pub, kex_public, kex_public_size);

    derive_cipher_keys(pir->the_comctx->kex_secret,
                       pir->the_comctx->kex_public, fp->remote_kex_pub,
                       fp->encrypt_key, fp->decrypt_key);

    servlog_message(srvl_debug, "set kex pub %s for %s",
                    hexdata2a(fp->remote_kex_pub, kex_public_size),
                    ipport2a(fp->ip, fp->port));
    return 1;
}

int feda_peer_should_initiate_assoc(const struct feda_peer *fp)
{
    return fp->initiate_assoc;
}

int feda_peer_assoc_status(const struct feda_peer *fp)
{
    return fp->assoc_status;
}

int feda_peer_set_assocstatus(struct feda_peer *fp, int newstat)
{
    int r = fp->assoc_status;
    fp->assoc_status = newstat;
    return r;
}

int feda_peer_ever_had_assoc(const struct feda_peer *fp)
{
    return !all_zeroes(fp->remote_public, sizeof(fp->remote_public));
}

struct feda_proxy *feda_peer_get_proxy_object(const struct feda_peer *fp)
{
    return fp->the_proxy;
}

static void find_termz(char **p)
{
    while(**p)
        (*p)++;
}

const char *feda_peer_description(const struct feda_peer *fp)
{
    static char res[256 /* peer_name_length_limit*2 + 21 + 24 + 12 + ... */];
    char *p = res;

    strcpy(p, ipport2a(fp->ip, fp->port));
    find_termz(&p);

    if(fp->the_conf_by_ip) {
        *p = ' ';
        p++;
        *p = '[';
        p++;
        strcpy(p, fp->the_conf_by_ip->name);
        find_termz(&p);
        *p = ']';
        p++;
    }

    if(feda_peer_identified(fp)) {
        *p = ' ';
        p++;
        *p = '(';
        p++;
        if(fp->assoc_status != fpas_established) {
            *p = feda_peer_ever_had_assoc(fp) ? '~' : '?';
            p++;
            *p = ':';
            p++;
        }
        hexdata2str(p, fp->node_id, node_id_size);
        p += node_id_size*2;
        *p = '.';
        p++;
        strcpy(p, decimal2a(fp->point));
        find_termz(&p);
        if(fp->the_conf_by_point) {
            *p = ' ';
            p++;
            *p = '[';
            p++;
            strcpy(p, fp->the_conf_by_point->name);
            find_termz(&p);
            *p = ']';
            p++;
        }
        *p = ')';
        p++;
    }

    *p = 0;
    return res;
}

void choose_peers_for_natcheck(struct feda_pir_collection *pir, int stubcred,
                               struct feda_peer *peers[10], int *num)
{
    int i, count;

    memset(peers, 0, sizeof(*peers) * 10);
    count = 0;
    for(i = 0; i < pir->natcheck_peer_count; i++) {
        struct feda_peer *fp = pir->natcheck_peers[i];
        if(!fp)
            continue;
        if(fp->assoc_status != fpas_established)
            continue;
        peers[count] = pir->natcheck_peers[i];
        count++;
        if(count >= stubcred)
            break;
    }
    *num = count;
}



int feda_peer_handle_traffic(struct feda_peer *fp,
                             const unsigned char *packet, int packetsize,
                             unsigned char *buf, int bufsize)
{
    int res, r;

    if(!fp->defrag_buf)
        fp->defrag_buf = make_peer_defrag(defrag_seq_bits);

    res = peer_defrag_proc(fp->defrag_buf, packet, packetsize, buf, bufsize);

    switch(res) {
    case fpdstat_go_on:           /* ok, no more actions needed */
        return 0;
    case fpdstat_dropped_this:    /* probably duplicate, just ignore */
        return 0;
    case fpdstat_dropped_earlier:
        servlog_message(srvl_debug, "dropped earlier packet from peer %s",
                                    ipport2a(fp->ip, fp->port));
        return 0;
    case fpdstat_buffer_short:
        servlog_message(srvl_debug, "we're short of buffer (pkt from %s)",
                                    ipport2a(fp->ip, fp->port));
        return 0;
    }

    if(res < 1) {
        servlog_message(srvl_alert, "handle_cr_data: unexpected value (bug)");
        return 0;
    }

    if(res < 40) {
        servlog_message(srvl_info,
            "peer %s sent us a packet shorter than IPv6 header (%d bytes)",
            ipport2a(fp->ip, fp->port), res);
        return 0;
    }

    r = ipv6_decrement_hop_limit(buf);
    if(!r) {
        servlog_message(srvl_info, "packet from peer %s dropped (TTL expired)",
                                   ipport2a(fp->ip, fp->port));
        return 0;
    }

    return res;
}
