#include <stdlib.h>
#include <string.h>

#include "hexdata.h"
#include "treebyte.h"
#include "cryptodf.h"
#include "fsrv_rep.h"
#include "fsrv_pir.h"  /* for report only */

#include "fsrv_dst.h"


static struct destination_node *make_dest_node(const unsigned char *node_id)
{
    struct destination_node *p = malloc(sizeof(*p));
    memcpy(p->node_id, node_id, node_id_size);
    p->node_cert_status = ncstat_unknown;
    p->home_node = 0;
    p->node_peer = NULL;
    p->point_peers = NULL;
    return p;
}

struct feda_destination_map {
    struct treebyte tree;
    struct destination_node *my_node;
    int my_point;
    struct feda_peer *nodenets;
    char fe_too;   /* whether nodenets should serve the 0xFE point as well */
};


struct feda_destination_map *make_destination_map()
{
    struct feda_destination_map *p;
    p = malloc(sizeof(*p));
    treebyte_init(&p->tree, node_id_size);
    p->my_node = NULL;
    p->my_point = -1;
    p->nodenets = NULL;
    p->fe_too = 0;

    return p;
}


void set_destination_peer(struct feda_destination_map *map,
                          const unsigned char *node_id, int point,
                          struct feda_peer *peer)
{
    void **loc;
    struct destination_node *dn;

    if(!map)
        return;

    loc = treebyte_provide(&map->tree, node_id);
    if(!*loc) {
        if(!peer)
            return;
        dn = make_dest_node(node_id);
        *loc = dn;
    } else {
        dn = *loc;
    }
        /* we're only called when the crypto association has just been
           successfully established (or just gone), both are only possible
           when the cert is there */
    dn->node_cert_status = ncstat_present;

    if(point == 0x00 || point == 0xFE) {
        dn->node_peer = peer;
    } else {
        if(!dn->point_peers) {
                /* NB: 254 is because we don't need slots for 0xFE, 0xFF */
            int sz = 254 * sizeof(void*);
            dn->point_peers = malloc(sz);
            memset(dn->point_peers, 0, sz);
        }
        dn->point_peers[point] = peer;
    }

    if(peer == FEDA_DEST_IS_LOCAL) {
        dn->home_node = 1;
        map->my_node = dn;
        map->my_point = point;
    }
}

void destmap_set_nodenets_peer(struct feda_destination_map *map,
                               struct feda_peer *peer, int fe_too)
{
    map->nodenets = peer;
    map->fe_too = fe_too;
}

struct destination_node *find_destination(struct feda_destination_map *map,
                                          const unsigned char *node_id)
{
    void **loc;
    struct destination_node *dn;

    if(!map)
        return NULL;

    loc = treebyte_provide(&map->tree, node_id);
    if(!*loc) {
        dn = make_dest_node(node_id);
        dn->node_cert_status = ncstat_unknown;
        *loc = dn;
    } else {
        dn = *loc;
    }

    return dn;
}

static
struct feda_peer* choose_dest_peer(struct destination_node *dn, int point)
{
    if(!dn->point_peers || point < 1 || point >= 0xFE)
        return dn->node_peer;
    if(dn->point_peers[point])
        return dn->point_peers[point];
    return dn->node_peer;
}

static struct feda_peer *default_peer(struct feda_destination_map *map)
{
    if(!map->my_node || map->my_point == 0xFE || map->my_point == 0)
        return NULL;
    return map->my_node->node_peer;
}

struct feda_peer *find_dest_peer(struct feda_destination_map *map,
                                 const unsigned char *node_id, int point)
{
    struct feda_peer *dp;
    struct destination_node *dn;
    struct feda_peer *peer;

    dp = default_peer(map);

    dn = find_destination(map, node_id);
    if(!dn)
        return dp ? dp : FEDA_DEST_NO_NODE;

    /* well, we _do_ have the destination node */
    if(dn->home_node && map->nodenets &&
        (point == 0x00 || point == 0xff || (point == 0xfe && map->fe_too))
    ) {
        return map->nodenets;
    }

    if(dn->node_cert_status != ncstat_present)
        return dp ? dp : FEDA_DEST_NO_NODE_CERT;

    peer = choose_dest_peer(dn, point);
    if(!peer)
        return dp ? dp : FEDA_DEST_NO_POINT;

    return peer;
}


/* ---------------------------------------------------------------
   THE REPORTS
   --------------------------------------------------------------- */

static const char *node_cert_status_str(int s)
{
    switch(s) {
    case ncstat_unknown: return "unknown";
    case ncstat_present: return "present";
    case ncstat_absent:  return "absent";
    default:             return "BUG";
    }
}

static const char *ext_peer_description(const struct feda_peer *peer)
{
    if(!peer)
        return "NULL";
    if(peer == FEDA_DEST_NO_NODE)
        return "NO_NODE";
    if(peer == FEDA_DEST_NO_NODE_CERT)
        return "NO_NODE_CERT";
    if(peer == FEDA_DEST_NO_POINT)
        return "NO_POINT";
    if(peer == FEDA_DEST_IS_LOCAL)
        return "IS_LOCAL";
    return feda_peer_description(peer);
}

static void destmap_node_report(struct destination_node *np,
                                report_callback f, void *userdata)
{
     f(userdata, "= %s%s, cert: %s, ",
         hexdata2a(np->node_id, node_id_size),
         np->home_node ? " (HOME)" : "",
         node_cert_status_str(np->node_cert_status));
     if(np->node_peer)
         f(userdata, "  node->     %s", ext_peer_description(np->node_peer));
     if(np->point_peers) {
         int i;
         for(i = 0; i < 254; i++) {
             if(np->point_peers[i]) {
                 f(userdata, "  %3d(x%02x)-> %s", i, i,
                             ext_peer_description(np->point_peers[i]));
             }
         }
     }
}

int destmap_choose_nodes(struct feda_destination_map *map,
                         const unsigned char *pref, int preflen,
                         unsigned char (*nodes)[node_id_size], int max)
{
    int n;
    void **vpp;
    struct treebyte_iter *iter;

    n = 0;
    iter = treebyte_make_iter(&map->tree, pref, preflen);
    while((vpp = treebyte_iter_next(iter)) != NULL) {
        struct destination_node *np;
        int idx;
        if(!*vpp)
            continue;
        np = *vpp;
        idx = n >= max ? max-1 : n;
        memcpy(nodes[idx], np->node_id, node_id_size);
        n++;
    }
    treebyte_dispose_iter(iter);
    return n;
}

void destmap_report(struct feda_destination_map *map,
                    const unsigned char *pref, int preflen,
                    report_callback f, void *userdata)
{
    void **vpp;
    struct treebyte_iter *iter;

    if(map->nodenets) {
        f(userdata, "%% nodenets: %s%s", 
                    ext_peer_description(map->nodenets),
                    map->fe_too ? " xFE_too" : "");
    }
    iter = treebyte_make_iter(&map->tree, pref, preflen);
    while((vpp = treebyte_iter_next(iter)) != NULL) {
        struct destination_node *np;
        if(!*vpp)
            continue;
        np = *vpp;
        destmap_node_report(np, f, userdata);
    }
    treebyte_dispose_iter(iter);
}

    /* preflen is in HALFbytes! */
void destmap_streamrep(struct feda_destination_map *map,
                       const unsigned char *pref, int preflen, void *stream)
{
    destmap_report(map, pref, preflen, report_to_stream_cb, stream);
}

void destmap_pointrep(struct feda_destination_map *map,
                      const unsigned char *nid, int point,
                      report_callback f, void *userdata)
{
    struct feda_peer *peer;

    peer = find_dest_peer(map, nid, point);
    f(userdata, "%s\n", ext_peer_description(peer));
}

void destmap_stream_pointrep(struct feda_destination_map *map,
                             const unsigned char *nid, int point, void *stream)
{
    destmap_pointrep(map, nid, point, report_to_stream_cb, stream);
}
