#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/un.h>

#include <sue/sue_base.h>

#include "strargv.h"
#include "hexdata.h"
#include "servlog.h"
#include "pointcfg.h"
#include "fsrv_cfg.h"
#include "fsrv_cfd.h"
#include "fsrv_rx.h"
#include "fsrv_dst.h"
#include "_version.h"

#include "fsrv_con.h"


struct feda_con_session {
    struct feda_console *master;
    struct sue_fd_handler fdh;
    FILE *stream;
    struct feda_con_session *next;
    extra_log_id log_id;
    char skipping;
    char to_close;
    int buf_used;
    char buf[4000];   /* the idea is to be less than 4096 in total */
};

struct feda_console {
    struct sue_fd_handler fdh;
    struct sue_event_selector *the_selector;
    struct server_conf_info *the_config;
    struct feda_udp_receiver *the_feda_rx;
    struct feda_con_session *first;
    char *path;
};



static void send_commit(struct feda_con_session *ses)
{
    if(!ses->to_close)
        fputs("== ", ses->stream);
    fflush(ses->stream);
}

static const char text_help[] =
    "The following commands are recognized:\n"
    "    exit              close the session\n"
    "    help [<command>]  show help text\n"
    "    log <level>       enable/disable log messages (see ``help log'')\n"
    "    quit              close the session (just like 'exit')\n"
    "    show <what>       show various information (see ``help show'')\n"
    "    shutdown          shut the server down immediately\n"
    "Type ``help <command>'' to get help on a specific command.\n"
    ;

static const char text_help_help[] =
    "The ``help'' command shows help texts; with no parameters, shows the\n"
    "general help text, which includes the list of the available commands.\n"
    "Type ``help <command>'' to get help on a specific command.\n"
    ;

static const char text_help_exit_quit[] =
    "Both ``exit'' and ``quit'' commands cause the server to close your\n"
    "control console session.  No arguments are expected.\n"
    ;

static const char text_help_log[] =
    "The ``log'' command sets the level of the log messages you'd like to\n"
    "see.  It requires exactly one parameter, one of the following:\n"
    "    no/off/disable         turn log messages off completely\n"
    "    q/quiet                only see the alert level messages\n"
    "    yes/on/enable/normal   set level to ``normal''\n"
    "    verb/verbose/info      see alert, normal and info messages\n"
    "    debug                  see everything up to the debug level\n"
    "    debug2                 see REALLY everything\n"
    "Please note your console is always considered private enough to show\n"
    "sensitive data.  If you work under an unwanted supervision, avoid\n"
    "turning the log on (well, as long as it is possible)\n"
    ;

static const char text_help_shutdown[] =
    "The ``shutdown'' command causes the server to shut down.  Please\n"
    "specify the word ``really'' as the only parameter to confirm you\n"
    "really want it.  Otherwise, the command will tell you it needs the\n"
    "word to be specified.\n"
    ;

static const char text_help_show[] =
    "The ``show'' shows information depending on its subcommand.  The\n"
    "following subcommands are recognized:\n"
    "    conf                 dump the configuration\n"
    "    dest [<prefix> [<point>]]    display the existing dest. map\n"
    "                         or a part of it, or a route for the point;\n"
    "                         point may be given decimal or hex as xNN\n"
    "    pid                  show the daemon's process ID (PID)\n"
    "    point                show the daemon's point information\n"
    "    report               show the peer report (the same info as the\n"
    "                         server writes to the logs upon SIGUSR1)\n"
    "    version              show the daemon's version and the build date\n"
    ;

static void con_ses_handle_help(struct feda_con_session *ses, char **argv)
{
    if(!argv[1])
        fputs(text_help, ses->stream);
    else
    if(0 == strcmp(argv[1], "help"))
        fputs(text_help_help, ses->stream);
    else
    if(0 == strcmp(argv[1], "exit") || 0 == strcmp(argv[1], "quit"))
        fputs(text_help_exit_quit, ses->stream);
    else
    if(0 == strcmp(argv[1], "shutdown"))
        fputs(text_help_shutdown, ses->stream);
    else
    if(0 == strcmp(argv[1], "show"))
        fputs(text_help_show, ses->stream);
    else
    if(0 == strcmp(argv[1], "log"))
        fputs(text_help_log, ses->stream);
    else
        fprintf(ses->stream, "* Sorry, nothing is known about ``%s''\n",
                             argv[1]);
}

    /* dsize will be in HALFbytes!  returns boolean */
static int hex2nodepref(unsigned char *node_id, int *dsize, const char *str)
{
    int dest, i, half;
    dest = 0;
    half = 0;
    for(i = 0; str[i]; i++) {
        char c = str[i];
        int dig = hexdigval(c);
        if(dig == -1 || i >= 2*node_id_size)
            return 0;
        if(!half) {
            node_id[dest] = (dig << 4) & 0xf0;
        } else {
            node_id[dest] |= (dig & 0x0f);
            dest++;
        }
        half = !half;
    }
    *dsize = i;
    return 1;
}

static void print_node_choice(FILE *stream, const unsigned char nid[])
{
    fprintf(stream, " |  %s\n", hexdata2a(nid, node_id_size));
}

static void con_ses_show_destpoint(struct feda_con_session *ses,
                       struct feda_destination_map *map,
                       const unsigned char *nipref, int preflen, int point)
{
    enum { max_to_choose = 6 };
    unsigned char nids[max_to_choose][node_id_size];
    int cnt, i;

    cnt = destmap_choose_nodes(map, nipref, preflen, nids, max_to_choose);
    if(cnt < 1) {
        fprintf(ses->stream, "* No nodes by this prefix ``%s''/%d\n",
                             hexdata2a(nipref, (preflen+1)/2), preflen);
        return;
    }
    if(cnt > 1) {
        fprintf(ses->stream, "* Which node do you mean?\n");
        if(cnt <= max_to_choose) {
            for(i = 0; i < cnt; i++)
                print_node_choice(ses->stream, nids[i]);
        } else {
            for(i = 0; i < max_to_choose - 1; i++)
                print_node_choice(ses->stream, nids[i]);
            fprintf(ses->stream, " ...\n");
            print_node_choice(ses->stream, nids[max_to_choose - 1]);
        }
        return;
    }
    destmap_stream_pointrep(map, nids[0], point, ses->stream);
}

static void con_ses_show_dest(struct feda_con_session *ses,
                              const char *nidstr, const char *pointstr)
{
    struct feda_destination_map *map;
    unsigned char destpref[node_id_size];
    int preflen;
    int ok;

    map = feda_rx_get_destmap(ses->master->the_feda_rx);

    if(nidstr) {
        ok = hex2nodepref(destpref, &preflen, nidstr);
        if(!ok) {
            fprintf(ses->stream, "* Incorrect node prefix ``%s''\n", nidstr);
            return;
        }
    } else {
        preflen = 0;
    }

    if(pointstr) {
        int point;
        char *err;
        point = *pointstr == 'x' ? 
            strtol(pointstr + 1, &err, 16) :
            strtol(pointstr, &err, 10);
        if(*err || point < 0 || point > 255) {
            fprintf(ses->stream, "* Invalid point number [%s]\n", pointstr);
            return;
        }
#if 0     /* debug message, remove it later */
        fprintf(ses->stream, "** [%s]/%d\n", hexdata2a(destpref, 5), preflen);
#endif
        con_ses_show_destpoint(ses, map, destpref, preflen, point);
    } else {
        destmap_streamrep(map, destpref, preflen, ses->stream);
    }
}

static void con_ses_show_point(struct feda_con_session *ses)
{
    const struct point_config_file *pcf;
    pcf = feda_rx_get_point_cfg(ses->master->the_feda_rx);
    fprintf(ses->stream, "%s.%d\n",
                         hexdata2a(pcf->node_id, node_id_size), pcf->point);
}

static void con_ses_handle_show(struct feda_con_session *ses, char **argv)
{
    if(!argv[1]) {
        fputs(text_help_show, ses->stream);
        return;
    }
    if(0 == strcmp(argv[1], "conf")) {
        dump_configuration_to_stream(ses->master->the_config, ses->stream);
    } else
    if(0 == strcmp(argv[1], "dest")) {
        con_ses_show_dest(ses, argv[2], argv[2] ? argv[3] : NULL);
    } else
    if(0 == strcmp(argv[1], "pid")) {
        fprintf(ses->stream, "%ld\n", (long)getpid());
    } else
    if(0 == strcmp(argv[1], "point")) {
        con_ses_show_point(ses);
    } else
    if(0 == strcmp(argv[1], "report")) {
        feda_udp_receiver_streamrep(ses->master->the_feda_rx, ses->stream);
    } else
    if(0 == strcmp(argv[1], "version")) {
        fputs("FEDAnet server (fedaserv) vers. " FEDA_VERSION
              " (compiled " __DATE__ ")\n",
              ses->stream);
    } else
    {
        fprintf(ses->stream, "* Sorry, don't know how to show ``%s''\n",
                             argv[1]);
    }
}

static void con_ses_log_cb(void *userdata, const char *message)
{
    struct feda_con_session *ses = userdata;
    fputs(message, ses->stream);
    fflush(ses->stream);
}

static void set_log_level(struct feda_con_session *ses, int level)
{
    int r;
    if(level == srvl_disable) {
        if(ses->log_id) {
            r = remove_extra_log(ses->log_id);
            ses->log_id = NULL;
            if(r) {
                fputs("Log messages disabled\n", ses->stream);
            } else {
                fputs("Error removing the log channel (bug)\n", ses->stream);
                servlog_message(srvl_alert, "bug in set_log_level (1)");
            }
        } else {
            fputs("Log messages are already off\n", ses->stream);
        }
    } else {
        if(ses->log_id) {
            r = change_extra_log(ses->log_id, level | srvl_private);
            if(r) {
                fputs("OK\n", ses->stream);
            } else {
                fputs("Log channel not found (bug)\n", ses->stream);
                servlog_message(srvl_alert, "bug in set_log_level (2)");
            }
        } else {
            ses->log_id =
                add_extra_log(&con_ses_log_cb, ses, level | srvl_private);
        }
    }
}

static void con_ses_handle_log(struct feda_con_session *ses, char **argv)
{
    struct levit {
        const char *name;
        int level;
    };
    static const struct levit names[] = {
        { "no",      srvl_disable },
        { "off",     srvl_disable },
        { "disable", srvl_disable },
        { "alert",   srvl_alert   },
        { "quiet",   srvl_alert   },
        { "q",       srvl_alert   },
        { "yes",     srvl_normal  },
        { "on",      srvl_normal  },
        { "normal",  srvl_normal  },
        { "enable",  srvl_normal  },
        { "info",    srvl_info    },
        { "verb",    srvl_info    },
        { "verbose", srvl_info    },
        { "debug",   srvl_debug   },
        { "debug2",  srvl_debug2  },
        { NULL,      0            }
    };

    int idx;

    if(!argv[1]) {
        fputs(text_help_log, ses->stream);
        return;
    }
    for(idx = 0; names[idx].name; idx++)
        if(0 == strcmp(argv[1], names[idx].name)) {  /* found! */
            set_log_level(ses, names[idx].level);
            return;
        }
    fprintf(ses->stream, "Level ``%s'' unknown, try ``help log''\n", argv[1]);
}

static void con_ses_handle_shutdown(struct feda_con_session *ses, char **argv)
{
    if(argv[1] && 0 == strcmp(argv[1], "really")) {
        fputs("Breaking the main loop, bye-bye\n", ses->stream);
        servlog_message(srvl_normal,
            "Shutting down on command from the control console");
        sue_sel_break(ses->master->the_selector);
    } else {
        fputs("* Please give me the word ``really'' to confirm\n",
              ses->stream);
    }
}

static void
con_ses_handle_command(struct feda_con_session *ses, const char *cmd, int len)
{
    char **argv;
    argv = make_argv(cmd);
    if(!argv) {
        fprintf(ses->stream, "* PARSE ERROR (%s)\n",
                             mkargv_diags(make_argv_errno));
        send_commit(ses);
        return;
    }
    if(!argv[0]) {
        /* actually nothing to do */
    } else
    if(0 == strcmp(argv[0], "exit") || 0 == strcmp(argv[0], "quit")) {
        fputs("Bye.\n", ses->stream);
        ses->to_close = 1;
    } else
    if(0 == strcmp(argv[0], "help")) {
        con_ses_handle_help(ses, argv);
    } else
    if(0 == strcmp(argv[0], "show")) {
        con_ses_handle_show(ses, argv);
    } else
    if(0 == strcmp(argv[0], "shutdown")) {
        con_ses_handle_shutdown(ses, argv);
    } else
    if(0 == strcmp(argv[0], "log")) {
        con_ses_handle_log(ses, argv);
    } else
    {
        fprintf(ses->stream, "* ERROR command (%s) unknown, try ``help''\n",
                             argv[0]);
    }

    dispose_argv(argv);
    send_commit(ses);
}

static void destroy_session(struct feda_con_session *ses)
{
    struct feda_console *fc = ses->master;
    struct sue_event_selector *sel = fc->the_selector;
    struct feda_con_session **p;

    sue_sel_remove_fd(sel, &ses->fdh);
    if(ses->stream)
        fclose(ses->stream);
    else
        close(ses->fdh.fd); /* this means fdopen failed on start */

    if(ses->log_id) {
        int ok;
        ok = remove_extra_log(ses->log_id);
        if(!ok)
            servlog_message(srvl_alert,
                            "destroy_session: can't remove_extra_log (bug)");
        ses->log_id = NULL;
    }

    p = &fc->first;
    while(*p) {
        if(*p == ses) { /* found! */
            *p = (*p)->next;
            free(ses);
            return;
        }
        p = &(*p)->next;
    }

    servlog_message(srvl_alert,
                    "destroy_session: couldn't find it in the list! (bug)");
}

static int find_eol(const char *buf, int buflen)
{
    int i;
    for(i = 0; i < buflen; i++)
        if(buf[i] == '\n')
            return i;
    return -1;
}

static void con_data_fd_handler(struct sue_fd_handler *h, int r, int w, int x)
{
    struct feda_con_session *ses;
    int res;

    if(!r || w || x) {
        servlog_message(srvl_alert,
            "con_data_fd_handler: unexpected combination %d %d %d", r, w, x);
        return;
    }

    ses = h->userdata;
    res = read(h->fd, ses->buf, sizeof(ses->buf) - ses->buf_used);
    if(res <= 0) {
        if(res == -1)
            servlog_perror(srvl_normal, "con_data_fd_handler", "read");
        destroy_session(ses);
        return;
    }

    ses->buf_used += res;

    while(ses->buf_used > 0) {
        int pos = find_eol(ses->buf, ses->buf_used);
        int rest;
        if(pos == -1) {
            if(ses->buf_used > sizeof(ses->buf) - 5) {
                static const char errmsg[] =
                    "** LINE TOO LONG **\n"
                    "Skipping; send a EOL char to recover\n";
                write(h->fd, errmsg, sizeof(errmsg)-1);
                ses->skipping = 1;
                ses->buf_used = 0;
            }
            break;
        }
        ses->buf[pos] = 0;
        if(!ses->skipping)
            con_ses_handle_command(ses, ses->buf, pos);
        rest = ses->buf_used - pos - 1;
        if(rest > 0)
            memmove(ses->buf, ses->buf + pos + 1, rest);
        ses->buf_used -= (pos + 1);
        ses->skipping = 0;   /* we've just seen EOL, anyway! */
    }
    if(ses->to_close)
        destroy_session(ses);
}

static void listen_fd_handler(struct sue_fd_handler *h, int r, int w, int x)
{
    struct feda_console *fc;
    struct feda_con_session *ses;
    int fd;

    if(!r || w || x) {
        servlog_message(srvl_alert,
            "listen_fd_handler: unexpected combination %d %d %d", r, w, x);
        return;
    }

    fc = h->userdata;

    fd = accept(h->fd, NULL, NULL);
    if(fd == -1) {
        servlog_perror(srvl_alert, NULL, "accept");
        return;
    }

    ses = malloc(sizeof(*ses));
    ses->master = fc;
    ses->fdh.fd = fd;
    ses->fdh.want_read = 1;
    ses->fdh.want_write = 0;
    ses->fdh.want_except = 0;
    ses->fdh.userdata = ses;
    ses->fdh.handle_fd_event = &con_data_fd_handler;
    ses->skipping = 0;
    ses->to_close = 0;
    ses->buf_used = 0;
    ses->next = fc->first;
    ses->log_id = NULL;
    ses->stream = fdopen(fd, "w");
    fc->first = ses;

    sue_sel_register_fd(fc->the_selector, &ses->fdh);
    send_commit(ses);  /* to display the prompt */
}





static int
set_socket_name(struct sockaddr_un *sun, struct server_conf_info *conf)
{
    int len;

    settle_ctlsockpath(conf);
    if(!conf->control_socket_path || !*conf->control_socket_path) {
        servlog_message(srvl_alert, "don't have the control socket path");
        return 0;
    }
    len = strlen(conf->control_socket_path);
    if(len + 1 > sizeof(sun->sun_path)) {
        servlog_message(srvl_alert, "control socket path is too long");
        return 0;
    }
    strcpy(sun->sun_path, conf->control_socket_path);
    return 1;
}

static int cleanup_old_socket(const char *path)
{
    int r;
    struct stat sb;

    r = stat(path, &sb);
    if(r == -1) {
        if(errno == ENOENT) /* no file, nothing to do, everything's fine */
            return 1;
        servlog_perror(srvl_alert, "stat", path);
        return 0;
    }
    if((sb.st_mode & S_IFMT) != S_IFSOCK) {
        servlog_message(srvl_alert, "%s exists and is not a socket", path);
        return 0;
    }
    r = unlink(path);
    if(r == -1) {
        servlog_perror(srvl_alert, "unlink", path);
        return 0;
    }
    return 1;
}

struct feda_console *
launch_control_console(struct sue_event_selector *sel,
                       struct server_conf_info *conf,
                       struct feda_udp_receiver *feda_rx)
{
    struct feda_console *fc;
    int sd, r, save_umask;
    struct sockaddr_un sun;

    sd = socket(AF_UNIX, SOCK_STREAM, 0);
    if(!sd) {
        servlog_perror(srvl_alert, "launch_control_console", "socket");
        return NULL;
    }
    sun.sun_family = AF_UNIX;
    r = set_socket_name(&sun, conf);
    if(!r) {
        /* diagnostic is printed already */
        close(sd);
        return NULL;
    }
    r = cleanup_old_socket(sun.sun_path);
    if(!r) {
        servlog_message(srvl_alert,
            "socket cleanup failed, will run with no control socket");
        close(sd);
        return NULL;
    }
    save_umask = umask(077);
    r = bind(sd, (struct sockaddr*)&sun, sizeof(sun));
    if(r == -1) {
        servlog_perror(srvl_alert, "launch_control_console", sun.sun_path);
        close(sd);
        umask(save_umask);
        return NULL;
    }
    umask(save_umask);
    listen(sd, 5);

    fc = malloc(sizeof(*fc));
    fc->fdh.fd = sd;
    fc->fdh.want_read = 1;
    fc->fdh.want_write = 0;
    fc->fdh.want_except = 0;
    fc->fdh.userdata = fc;
    fc->fdh.handle_fd_event = &listen_fd_handler;
    fc->the_selector = sel;
    fc->the_config = conf;
    fc->the_feda_rx = feda_rx;
    fc->path = strdup(sun.sun_path);
    fc->first = NULL;
    sue_sel_register_fd(sel, &fc->fdh);

    return fc;
}

void destroy_control_console(struct feda_console *con)
{
    int r;

    while(con->first)
        destroy_session(con->first);

    sue_sel_remove_fd(con->the_selector, &con->fdh);
    close(con->fdh.fd);

    r = unlink(con->path);
    if(r == -1)
        servlog_perror(srvl_normal, "warning: unlink", con->path);

    free(con->path);
    free(con);
}
