#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "hexdata.h"
#include "keyutils.h"
#include "textrec.h"
#include "strargv.h"
#include "servlog.h"

#include "fsrv_cfg.h"


const char *peer_type_str(int ptp)
{
    switch(ptp) {
    case ptp_default:    return "default";
    case ptp_mynode:     return "mynode";
    case ptp_natcheck:   return "natcheck";
    case ptp_persist:    return "persist";
    case ptp_nodenets:   return "nodenets";
    case ptp_trustncc:   return "trustncc";
    case ptp_certhub:    return "certhub";
    case ptp_introducer: return "introducer";
    case ptp_proxy:      return "proxy";
    case ptp_proxyuser:  return "proxyuser";
    case ptp_direct:     return "direct";
    default:             return NULL;
    }
}

int peer_conf_has_ip(const struct peer_conf *pc)
{
    return pc->port != 0 && pc->ip != PEER_IP_UNDEF;
}

int peer_conf_has_point(const struct peer_conf *pc)
{
    return
        !all_zeroes(pc->node_id, node_id_size)
        && pc->point != PEER_POINT_UNDEF;
}

/*
    This implementation handles all configuration parameters as strings,
    ignoring the textrec module's type capability.  This may look weird
    but it saves a lot of coding; see the configfile_{type,string}_cb
    functions to understand why.
 */

struct server_conf_info *make_servconf()
{
    struct server_conf_info *p;
    p = malloc(sizeof(*p));
    p->log_syslog_level = srvl_normal;
    p->log_syslog_priv = 0;
    p->log_syslog_facility = srvl_facil_default;
    p->log_syslog_ident = NULL;
    p->log_file_level = srvl_disable;
    p->log_file_priv = 0;
    p->log_file_name = NULL;
    p->log_stderr_level = srvl_disable;
    p->log_stderr_priv = 0;
    p->keys_dir = NULL;
    p->control_socket = 0;
    p->control_socket_path = NULL;
    p->listen_address = ntohl(INADDR_ANY);
    p->listen_port = 0xFEDA;
    p->cooldown_timeout = 3600;
    p->peer_timeout = 36000;
    p->keepalive_interval = 120;
    p->first_peer = NULL;
    p->forwarding = 0;
    p->tun_iface = NULL;
    p->nodenets = nodenets_none;
    return p;
}

void dispose_servconf(struct server_conf_info *p)
{
    if(p->log_syslog_ident)
        free(p->log_syslog_ident);
    if(p->log_file_name)
        free(p->log_file_name);
    if(p->keys_dir)
        free(p->keys_dir);
    if(p->control_socket_path)
        free(p->control_socket_path);
    while(p->first_peer) {
        struct peer_conf *tmp = p->first_peer;
        p->first_peer = p->first_peer->next;
        free(tmp);
    }
    if(p->tun_iface)
        free(p->tun_iface);
    free(p);
}



/* ---- read config ---------- */

static void message_parser_abort(const char *fname, int line, int code)
{
    if(code < 0) {
        servlog_message(srvl_normal, "%s:%d: parse error [%s]", fname, line,
                                     textrec_error_message(code));
    } else {
        servlog_message(srvl_normal, "%s:%d: the problem is here",
                                     fname, line);
    }
}

static int run_parser_on_file(struct textrec_parser *parser, const char *name,
                              int suppress_messages)
{
    FILE *f;
    int c, res;

    f = fopen(name, "r");
    if(!f) {
        if(!suppress_messages)
            servlog_perror(srvl_normal, name, NULL);
        return 0;
    }

    while((c = fgetc(f)) != EOF) {
        res = textrec_feedchar(parser, c);
        if(res != 0) {
            if(!suppress_messages)
                message_parser_abort(name, parser->line, res);
            break;
        }
    }
    if(c == EOF)
        res = textrec_feedchar(parser, EOF);
    fclose(f);
    return res == textrec_res_ok || res == textrec_res_eof;
}



typedef struct config_reading_context {
    struct server_conf_info *info;
    struct peer_conf *current_peer;
} cfgctx;

struct field_handler {
    const char *name;
    int (*func)(cfgctx *, const char *, int);
};


#define HFLD_LOGLEVEL(Field) \
{ \
    if(0 == strcmp(s, "none")) \
        c->info->Field = srvl_disable; \
    else \
    if(0 == strcmp(s, "alert")) \
        c->info->Field = srvl_alert; \
    else \
    if(0 == strcmp(s, "normal")) \
        c->info->Field = srvl_normal; \
    else \
    if(0 == strcmp(s, "info")) \
        c->info->Field = srvl_info; \
    else \
    if(0 == strcmp(s, "debug")) \
        c->info->Field = srvl_debug; \
    else \
    if(0 == strcmp(s, "debug2")) \
        c->info->Field = srvl_debug2; \
    else { \
        servlog_message(srvl_normal, "unknown log level [%s]", s); \
        return 1; \
    } \
    return 0; \
}

#define HFLD_BOOLEAN(Field) \
{ \
    c->info->Field = 0 == strcmp(s, "yes"); \
    return 0; \
}

#define HFLD_STRING(Field) \
{ \
    if(c->info->Field) free(c->info->Field); \
    c->info->Field = s ? strdup(s) : NULL; \
    return 0; \
}

#define HFLD_NONNEG_INTEGER(Field, FieldName) \
{ \
    long nn; \
    char *err; \
    nn = strtol(s, &err, 10); \
    if(*s != '\0' && *err == '\0' && nn >= 0) { \
        c->info->Field = nn; \
        return 0; \
    } \
    servlog_message(srvl_normal, "invalid " FieldName " [%s]", s); \
    return 1; \
} \

static int handle_log_syslog_level(cfgctx *c, const char *s, int len)
HFLD_LOGLEVEL(log_syslog_level)

static int handle_log_syslog_priv(cfgctx *c, const char *s, int len)
HFLD_BOOLEAN(log_syslog_priv)

static int handle_log_syslog_facility(cfgctx *c, const char *s, int len)
{
    struct nm {
        const char *name;
        int facil;
    };
    static const struct nm facils[] = {
        { "default", srvl_facil_default },
        { "user",    srvl_facil_user    },
        { "daemon",  srvl_facil_daemon  },
        { "local0",  srvl_facil_local0  },
        { "local1",  srvl_facil_local1  },
        { "local2",  srvl_facil_local2  },
        { "local3",  srvl_facil_local3  },
        { "local4",  srvl_facil_local4  },
        { "local5",  srvl_facil_local5  },
        { "local6",  srvl_facil_local6  },
        { "local7",  srvl_facil_local7  },
        { NULL,      -1 }
    };
    int i;
    for(i = 0; facils[i].name; i++)
        if(0 == strcmp(s, facils[i].name)) {
            c->info->log_syslog_facility = facils[i].facil;
            return 0;
        }
    servlog_message(srvl_normal, "unknown syslog facility name [%s]", s);
    return 1;
}

static int handle_log_syslog_ident(cfgctx *c, const char *s, int len)
HFLD_STRING(log_syslog_ident)

static int handle_log_file_level(cfgctx *c, const char *s, int len)
HFLD_LOGLEVEL(log_file_level)

static int handle_log_file_priv(cfgctx *c, const char *s, int len)
HFLD_BOOLEAN(log_file_priv)

static int handle_log_file_name(cfgctx *c, const char *s, int len)
HFLD_STRING(log_file_name)

static int handle_log_stderr_level(cfgctx *c, const char *s, int len)
HFLD_LOGLEVEL(log_stderr_level)

static int handle_log_stderr_priv(cfgctx *c, const char *s, int len)
HFLD_BOOLEAN(log_stderr_priv)

static int handle_listen_address(cfgctx *c, const char *s, int len)
{
    unsigned long addr = inet_addr(s);
    if(addr == INADDR_NONE) {
        servlog_message(srvl_normal, "invalid ip address [%s]", s);
        return 1;
    }
    c->info->listen_address = ntohl(addr);
    return 0;
}

static int handle_listen_port(cfgctx *c, const char *s, int len)
{
    long pn;
    char *err;
    pn = strtol(s, &err, 10);
    if(*s != '\0' && *err == '\0' && pn > 0 && pn < 65535) {
        c->info->listen_port = pn;
        return 0;
    }
    servlog_message(srvl_normal, "invalid port number [%s]", s);
    return 1;
}

static int handle_cooldown_timeout(cfgctx *c, const char *s, int len)
HFLD_NONNEG_INTEGER(cooldown_timeout, "timeout")

static int handle_peer_timeout(cfgctx *c, const char *s, int len)
HFLD_NONNEG_INTEGER(peer_timeout, "timeout")

static int handle_keepalive_interval(cfgctx *c, const char *s, int len)
HFLD_NONNEG_INTEGER(keepalive_interval, "interval")

static int handle_keys_dir(cfgctx *c, const char *s, int len)
HFLD_STRING(keys_dir)

static int handle_control_socket(cfgctx *c, const char *s, int len)
HFLD_BOOLEAN(control_socket)

static int handle_control_socket_path(cfgctx *c, const char *s, int len)
HFLD_STRING(control_socket_path)


static int handle_peer(cfgctx *c, const char *s, int len)
{
    struct peer_conf *p, *last;

    if(len > peer_name_length_limit) {
        servlog_message(srvl_normal, "peer name too long [%s]", s);
        return 1;
    }

    last = NULL;
    for(p = c->info->first_peer; p; p = p->next) {
        if(0 == strcmp(p->name, s)) {
            servlog_message(srvl_normal, "duplicate peer name [%s]", s);
            return 1;
        }
        last = p;
    }

    p = malloc(sizeof(*p));
    strcpy(p->name, s);
    p->type = 0;
    p->ip = PEER_IP_UNDEF;
    p->port = 0;
    memset(p->node_id, 0, node_id_size);
    p->point = PEER_POINT_UNDEF;
    p->proxyport = -1;
    p->next = NULL;

    if(last)
        last->next = p;
    else
        c->info->first_peer = p;

    c->current_peer = p;

    return 0;
}

static int handle_peer_type(cfgctx *c, const char *arg, int len)
{
    char **words;
    int res, i;

    if(!c->current_peer) {
        servlog_message(srvl_normal, "`type' directive without `peer'");
        return 1;
    }
    words = make_argv(arg);
    if(!words) {
        servlog_message(srvl_normal, "invalid data for `type' (%s)",
                                     mkargv_diags(make_argv_errno));
        return 1;
    }

    for(i = 0; words[i]; i++) {
        const char *s = words[i];

#define TYPECHECK(tt) \
        if(0 == strcmp(s, #tt))   \
            c->current_peer->type |= ptp_##tt;  \
        else


        TYPECHECK(default)
        TYPECHECK(mynode)
        TYPECHECK(natcheck)
        TYPECHECK(persist)
        TYPECHECK(nodenets)
        TYPECHECK(trustncc)
        TYPECHECK(certhub)
        TYPECHECK(introducer)
        TYPECHECK(proxy)
        TYPECHECK(proxyuser)
        TYPECHECK(direct)
        {
            servlog_message(srvl_normal, "unknown peer type [%s]", s);
            res = 1;
            goto quit;
        }

#undef TYPECHECK
    }

    res = 0;
quit:
    dispose_argv(words);
    return res;
}

static int handle_peer_ip(cfgctx *c, const char *s, int len)
{
    unsigned long addr;
    if(!c->current_peer) {
        servlog_message(srvl_normal, "`ip' directive without `peer'");
        return 1;
    }
    addr = inet_addr(s);
    if(addr == INADDR_NONE) {
        servlog_message(srvl_normal, "invalid ip address [%s]", s);
        return 1;
    }
    c->current_peer->ip = ntohl(addr);
    return 0;
}

static int handle_peer_port(cfgctx *c, const char *s, int len)
{
    long pn;
    char *err;
    if(!c->current_peer) {
        servlog_message(srvl_normal, "`port' directive without `peer'");
        return 1;
    }
    pn = strtol(s, &err, 10);
    if(*s != '\0' && *err == '\0' && pn > 0 && pn < 65535) {
        c->current_peer->port = pn;
        return 0;
    }
    servlog_message(srvl_normal, "invalid port number [%s]", s);
    return 1;
}

static int handle_peer_node_id(cfgctx *c, const char *s, int len)
{
    int res;
    if(!c->current_peer) {
        servlog_message(srvl_normal, "`node_id' directive without `peer'");
        return 1;
    }
    if(len != node_id_size * 2) {
        servlog_message(srvl_normal, "`node_id' of invalid size [%s]", s);
        return 1;
    }
    res = hexstr2data(c->current_peer->node_id, node_id_size, s);
    if(res != node_id_size) {
        servlog_message(srvl_normal, "invalid `node_id' [%s]", s);
        return 1;
    }
    return 0;
}

static int handle_peer_point(cfgctx *c, const char *s, int len)
{
    long pn;
    char *err;
    if(!c->current_peer) {
        servlog_message(srvl_normal, "`point' directive without `peer'");
        return 1;
    }
    pn = strtol(s, &err, 10);
    if(*s != '\0' && *err == '\0' && pn >= 0 && pn <= 253) {
        c->current_peer->point = pn;
        return 0;
    }
    servlog_message(srvl_normal, "invalid point number [%s]", s);
    return 1;
}

static int handle_peer_proxyport(cfgctx *c, const char *s, int len)
{
    long pn;
    char *err;
    if(!c->current_peer) {
        servlog_message(srvl_normal, "`proxyport' directive without `peer'");
        return 1;
    }
    pn = strtol(s, &err, 10);
    if(*s != '\0' && *err == '\0' && pn >= 0 && pn < 65535) {
        c->current_peer->proxyport = pn;
        return 0;
    }
    servlog_message(srvl_normal, "invalid proxyport number [%s]", s);
    return 1;
}

static int handle_endpeer(cfgctx *c, const char *s, int len)
{
    if(!c->current_peer) {
        servlog_message(srvl_normal, "`endpeer' directive without `peer'");
        return 1;
    }
    c->current_peer = NULL;
    return 0;
}

static int handle_forwarding(cfgctx *c, const char *s, int len)
HFLD_BOOLEAN(forwarding)

static int handle_tun_iface(cfgctx *c, const char *s, int len)
HFLD_STRING(tun_iface)

static int handle_nodenets(cfgctx *c, const char *s, int len)
{
    struct nm {
        const char *name;
        int val;
    };
    static const struct nm tokens[] = {
        { "none", nodenets_none },
        { "some", nodenets_some },
        { "all",  nodenets_all  },
        { NULL,   -1 }
    };
    int i;
    for(i = 0; tokens[i].name; i++)
        if(0 == strcmp(s, tokens[i].name)) {
            c->info->nodenets = tokens[i].val;
            return 0;
        }
    servlog_message(srvl_normal, "unknown nodenets mode [%s]", s);
    return 1;
}


static struct field_handler fields[] =
{
    { "log_syslog_level",    &handle_log_syslog_level },
    { "log_syslog_priv",     &handle_log_syslog_priv },
    { "log_syslog_facility", &handle_log_syslog_facility },
    { "log_syslog_ident",    &handle_log_syslog_ident },
    { "log_file_level",      &handle_log_file_level },
    { "log_file_priv",       &handle_log_file_priv },
    { "log_file_name",       &handle_log_file_name },
    { "log_stderr_level",    &handle_log_stderr_level },
    { "log_stderr_priv",     &handle_log_stderr_priv },
    { "listen_address",      &handle_listen_address },
    { "listen_port",         &handle_listen_port },
    { "cooldown_timeout",    &handle_cooldown_timeout },
    { "peer_timeout",        &handle_peer_timeout },
    { "keepalive_interval",  &handle_keepalive_interval },
    { "keys_dir",            &handle_keys_dir },
    { "control_socket",      &handle_control_socket },
    { "control_socket_path", &handle_control_socket_path },
    { "peer",                &handle_peer },
    { "type",                &handle_peer_type },
    { "ip",                  &handle_peer_ip },
    { "port",                &handle_peer_port },
    { "node_id",             &handle_peer_node_id },
    { "point",               &handle_peer_point },
    { "proxyport",           &handle_peer_proxyport },
    { "endpeer",             &handle_endpeer },
    { "forwarding",          &handle_forwarding },
    { "tun_iface",           &handle_tun_iface },
    { "nodenets",            &handle_nodenets },
    { NULL,                  NULL }
};



static int configfile_type_cb(const char *field_name, void *user_data)
{
#if 0   /* this is permanent, just kept as a reminder not to do so */
    int i;
    for(i = 0; fields[i].name; i++)
        if(0 == strcmp(field_name, fields[i].name))
            return ftype_string;
    return ftype_unknown;
#endif
    /* consider all fields valid here, to issue a warning for unknown ones */
    return ftype_string;
}

static int
configfile_string_cb(const char *fnm, const uchar *buf, int len, void *ud)
{
    int i;
    cfgctx *ctx = ud;
    for(i = 0; fields[i].name; i++)
        if(0 == strcmp(fnm, fields[i].name))
            return (*fields[i].func)(ctx, (const char*)buf, len);

    servlog_message(srvl_info, "WARNING: config: field unknown [%s]", fnm);
    return 0;
}

int read_config(const char *fname, struct server_conf_info *p)
{
    struct textrec_parser parser;
    cfgctx ctx;
    int res;

    textrec_init(&parser);

    textrec_set_type_cb(&parser, configfile_type_cb);
    textrec_set_string_cb(&parser, configfile_string_cb);

    ctx.info = p;
    textrec_set_user_data(&parser, &ctx);

    res = run_parser_on_file(&parser, fname, 0);

    textrec_cleanup(&parser);
    return res;
}

void settle_localpath(char **target, const char *localpath)
{
    const char *home;
    char *dir;
    int homelen, lplen;

    if(*target)
        return;

    home = getenv("HOME");
    if(!home || !*home) {  /* rare case; fallback to the current directory */
        *target = strdup(localpath);
        return;
    }
    homelen = strlen(home);
    lplen = strlen(localpath);
    dir = malloc(homelen + 1 + lplen + 1);
    strcpy(dir, home);
    dir[homelen] = '/';
    strcpy(dir + homelen + 1, localpath);
    *target = dir;
}

void settle_keydir(struct server_conf_info *p)
{
    settle_localpath(&p->keys_dir, ".fedanet/keys");
}

void settle_ctlsockpath(struct server_conf_info *p)
{
    settle_localpath(&p->control_socket_path, ".fedanet/servctl");
}
