#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <monocypher/monocypher.h>

#include "message.h"
#include "fileutil.h"
#include "keyutils.h"
#include "txtr_out.h"
#include "hexdata.h"
#include "kfiles.h"
#include "mastrkey.h"
#include "pointcfg.h"
#include "xmonocyp.h"
#include "knowndb.h"
#include "fk_args.h"
#include "fk_data.h"
#include "fk_diags.h"
#include "fk_point.h"



static void show_node_props(const struct point_config_file *conf)
{
    message(mlv_info, "rank %d; minimum rank for others is set to %d.\n",
                      conf->rank, conf->minpref);
}

static void explain_signedby(const struct point_config_file *conf)
{
    message(mlv_info, "This point's key certificate is signed by the %s.\n",
            conf->signed_by == 0 ? "ZeroPoint key" : "node master key");
}

static void display_zp_ts(const struct point_config_file *conf)
{
    const unsigned char *zpc = conf->zp_certbody;
    message(mlv_info, "ZeroPoint key timestamp %d\n",
                      get_cert_timestamp((const struct feda_cert_info*)zpc));
}

int main_status(const struct fedakeys_args *args)
{
    int res, errline;
    char *fname;
    struct point_config_file conf;
    unsigned char secret_key[secret_key_size];
    struct kex_keypair_file kex_data;

    fname = make_config_fname(args->dir);
    res = read_config_file(fname, &conf, &errline);
    if(res != rcf_res_ok) {
        pointcfg_message(mlv_alert, res, fname, errline);
        dispose_fname(fname);
        message(mlv_normal, "no cryptographic configuration available\n");
        return 1;
    }
    dispose_fname(fname);

    if(conf.point == -1) {             /* master workplace? */
        char *masterkey_fname;
        unsigned char master_secret[secret_key_size];

        masterkey_fname = make_masterkey_fname(args->dir);
        res = read_single_blob(masterkey_fname, "secret",
                            master_secret, sizeof(master_secret), &errline);
        dispose_fname(masterkey_fname);
        if(res == kfiles_res_ok) {
            crypto_wipe(master_secret, sizeof(master_secret));
            message(mlv_alert, "MASTER WORKPLACE for node %s\n",
                                hexdata2a(conf.node_id, sizeof(conf.node_id)));
            message(mlv_normal,
                "Extreme care has to be taken. No networking is desired.\n");
            show_node_props(&conf);
            return 0;
        } else {
            message(mlv_normal,
                "Looks like a master workplace, but no secret key found\n");
            message(mlv_info, "The node id is %s\n",
                              hexdata2a(conf.node_id, sizeof(conf.node_id)));
            return 1;
        }
    }

    message(mlv_alert, "Node %s point %d\n",
            hexdata2a(conf.node_id, sizeof(conf.node_id)), conf.point);
    show_node_props(&conf);
    message(mlv_normal, "Key timestamp %d\n",
            get_cert_timestamp((const struct feda_cert_info*)conf.certbody));
    if(conf.point != 0 && conf.signed_by == 0)
        display_zp_ts(&conf);

    fname = make_secretkey_fname(args->dir);
    res = read_single_blob(fname, "secret", secret_key, sizeof(secret_key),
                           &errline);
    if(res != kfiles_res_ok) {
        kfiles_message(mlv_alert, res, fname, errline);
        message(mlv_alert, "can't read the point's secret key\n");
        dispose_fname(fname);
        return 1;
    }
    dispose_fname(fname);
    crypto_wipe(secret_key, sizeof(secret_key));

    if(conf.point == 0) {
        message(mlv_normal, "This is the ZeroPoint workplace\n");
    } else {
        fname = make_zeropointkey_fname(args->dir);
        res = read_single_blob(fname, "secret", secret_key, secret_key_size,
                               &errline);
        dispose_fname(fname);
        if(res == kfiles_res_ok) {              /* ZP secret key is there! */
            crypto_wipe(secret_key, sizeof(secret_key));
            message(mlv_normal, "This is a combined ZP+point workplace\n");
            if(conf.signed_by == -1) /* we didn't yet display ZP ts */
                display_zp_ts(&conf);
        }
        explain_signedby(&conf);
    }

    fname = make_msgkey_fname(args->dir, NULL);
    res = read_kex_keypair_file(fname, &kex_data, 1);
    dispose_fname(fname);
    if(res) {
        crypto_wipe(kex_data.kex_secret, kex_secret_size);
        message(mlv_normal, "Key exchange (kex) key found, timestamp %d\n",
                            kex_data.created);
    } else {
        message(mlv_normal, "No key exchange (kex) key information found\n");
    }

    return 0;
}

static int read_verify_point(const char *fname, const char *secret_fname,
                  struct point_file *pf, int may_take_hash, const char **err)
{
    int res, errline, preflen;

    res = read_point_file(fname, pf);
    if(!res) {
        *err = "the file is not in proper format";
        return 0;
    }

    if(all_zeroes(pf->secret, secret_key_size)) {
        if(!secret_fname) {
            *err = "no secret key in the file and no separate key file name";
            return 0;
        }
        res = read_single_blob(secret_fname, "secret",
                               pf->secret, secret_key_size, &errline);
        if(res != kfiles_res_ok) {
            *err = "couldn't get the secret key from the separate file";
            return 0;
        }
    }

    res = check_keypair(pf->secret, pf->public_key);
    if(!res) {
        *err = "public and secret keys don't match";
        return 0;
    }

    preflen = check_master_pub(pf->node_id, pf->master_pub, pf->master_hash,
                               pf->master_hsign, may_take_hash);
    if(preflen < 0) {
        *err = checkmasterpub_message(preflen);
        return 0;
    }

    if(preflen != pf->rank) {
        *err = "the rank as specified doesn't match the hash";
        return 0;
    }

    if(preflen < pf->minpref) {
        *err = "the master's 0xFEDA prefix is shorter than the minimum";
        return 0;
    }

    res = compare_feda_cert((struct feda_cert_info *)pf->certbody,
        pf->node_id, pf->point, pf->signed_by, pf->timestamp,
        pf->public_key);
    if(!res) {
        *err = "certbody malformed";
        return 0;
    }

    if(pf->signed_by == -1) { /* signed by master, this is simple */
        res = crypto_eddsa_check(pf->signature, pf->master_pub,
                                 pf->certbody, feda_cert_size);
        if(res /* sic! */) {
            *err = "signature check failed";
            return 0;
        }
    } else
    if(pf->signed_by == 0) { /* signed by Zero Point, check in two steps */
        res = crypto_eddsa_check(pf->zp_signature, pf->master_pub,
                                 pf->zp_certbody, feda_cert_size);
        if(res /* sic! */) {
            *err = "zero point certificate signature check failed";
            return 0;
        }
        res = compare_feda_cert((struct feda_cert_info *)pf->zp_certbody,
            pf->node_id, 0, -1, -1 /* ignore timestamp */, pf->zp_pub);
        if(!res) {
            *err = "zero point certbody malformed";
            return 0;
        }
        res = crypto_eddsa_check(pf->signature, pf->zp_pub,
                                 pf->certbody, feda_cert_size);
        if(res /* sic! */) {
            *err = "certificate signature check failed";
            return 0;
        }
    } else {
        *err = "unsupported signed_by value";
        return 0;
    }

    /* all checks passed, everything's fine */
    return 1;
}

int main_pverify(const struct fedakeys_args *args)
{
    struct point_file mf;
    int res;
    const char *err = "";  /* initialized to avoid a stupid warning */

    if(args->cmdargc < 2) {
        message(mlv_alert, "\"fedakeys mverify\" needs a file name, try -h\n");
        return 1;
    }
    res = read_verify_point(args->cmdargs[1], args->cmdargs[2], &mf,
                            args->may_take_hash, &err);
    if(!res) {
        message(mlv_normal, "VERIFICATION FAILED: %s\n", err);
        return 1;
    }

    message(mlv_normal, "verification passed\n");
    return 0;
}

static int
can_upgrade_zp(const struct point_config_file *conf, const struct point_file *pf)
{
    if(pf->point == 0 || conf->point != 0)
        return 0;
    if(0 != memcmp(conf->node_id, pf->node_id, node_id_size))
        return 0;
    if(pf->signed_by == 0) {
        /* it must be the same version of the Zero Point key */
        if(0 != memcmp(conf->public_key, pf->zp_pub, public_key_size))
            return 0;
    }
    return 1;
}

int main_deploy(const struct fedakeys_args *args)
{
    struct point_file mf;
    int res, errline;
    const char *err;
    char *secretkey_fname, *config_fname;
    FILE *f;
    int saved_umask;
    int zeropoint_upgrade = 0;
    struct known_nodes_db *p;

    if(args->cmdargc < 2) {
        message(mlv_alert, "\"fedakeys deploy\" needs a file name, try -h\n");
        return 1;
    }

        /* now we grab all we need, and only quit via 'quit:' from now on */

    secretkey_fname = make_secretkey_fname(args->dir);
    config_fname = make_config_fname(args->dir);
    saved_umask = umask(0077);
    res = read_verify_point(args->cmdargs[1], args->cmdargs[2], &mf,
                            args->may_take_hash, &err);
    if(!res) {
        message(mlv_alert, "point file verification FAILED: %s\n", err);
        res = 1;
        goto quit;
    }

    if(file_is_there(config_fname)) {
        struct point_config_file conf;
        char *zeropointkey_fname;
        res = read_config_file(config_fname, &conf, &errline);
        if(res != rcf_res_ok) {
            pointcfg_message(mlv_normal, res, config_fname, errline);
            message(mlv_alert, "FATAL: can't read %s, despite it exists\n",
                    config_fname);
            res = 1;
            goto quit;
        }
        if(!can_upgrade_zp(&conf, &mf)) {
            message(mlv_alert,
                    "FATAL: %s already exists, move/delete it and retry\n",
                    config_fname);
            res = 1;
            goto quit;
        }
        /* okay, we'll upgrade ZP-only configuration to P+ZP */
        zeropoint_upgrade = 1;
        zeropointkey_fname = make_zeropointkey_fname(args->dir);
        res = rename(secretkey_fname, zeropointkey_fname);
        dispose_fname(zeropointkey_fname);
        if(res == -1) {
            message_perror(mlv_alert, "FATAL: rename", secretkey_fname);
            res = 1;
            goto quit;
        }
        /* if we don't have the zero point info within the point file,
           which is the case if it is signed by the node's master key,
           we need to keep zero point info from the current config file
         */
        if(mf.signed_by != 0) {
            /* memcpy(mf.zp_pub, conf.public_key, public_key_size); */
            memcpy(mf.zp_certbody, conf.certbody, feda_cert_size);
            memcpy(mf.zp_signature, conf.signature, signature_size);
        }
    }

    if(file_is_there(secretkey_fname)) {
        message(mlv_alert,
                "FATAL: %s already exists, move/delete it and retry\n",
                secretkey_fname);
        res = 1;
        goto quit;
    }

    res = make_directory_path(args->dir, 0);
    if(res == -1) {
        message_perror(mlv_alert, "FATAL", args->dir);
        res = 1;
        goto quit;
    }

    f = fopen(secretkey_fname, "w");
    if(!f) {
        message_perror(mlv_alert, "FATAL", secretkey_fname);
        res = 1;
        goto quit;
    }
    put_hex_field(f, "secret", mf.secret, sizeof(mf.secret));
    fclose(f);

    f = fopen(config_fname, "w");
    if(!f) {
        message_perror(mlv_alert, "FATAL", config_fname);
        res = 1;
        goto quit;
    }
    put_hex_field(f, "node_id", mf.node_id, sizeof(mf.node_id));
    put_int_field(f, "point", mf.point);
    put_int_field(f, "rank", mf.rank);
    put_int_field(f, "minpref", mf.minpref);
    put_hex_field(f, "public", mf.public_key, sizeof(mf.public_key));
    put_int_field(f, "signed_by", mf.signed_by);
    put_hex_field(f, "master_pub", mf.master_pub, sizeof(mf.master_pub));
    put_hex_field(f, "master_hash", mf.master_hash, sizeof(mf.master_hash));
    put_hex_field(f, "master_hsign", mf.master_hsign, sizeof(mf.master_hsign));
/*  put_int_field(f, "timestamp", mf.timestamp); */
    put_hex_field(f, "certbody", mf.certbody, sizeof(mf.certbody));
    put_hex_field(f, "signature", mf.signature, sizeof(mf.signature));
    if(mf.signed_by != -1 || zeropoint_upgrade) {
/*      put_hex_field(f, "zp_pub", mf.zp_pub, sizeof(mf.zp_pub)); */
        put_hex_field(f, "zp_certbody", mf.zp_certbody, sizeof(mf.zp_certbody));
        put_hex_field(f, "zp_signature", mf.zp_signature,
                                         sizeof(mf.zp_signature));
    }
    fclose(f);

    p = make_kndb(args->dir, mf.minpref);
    res = kndb_save_node(p, mf.node_id, mf.master_pub, mf.rank,
                         mf.master_hash, mf.master_hsign);
    if(res != kndb_res_success)
        message(mlv_normal, "WARNING: couldn't save the node as known (%s)\n",
                            kndb_result_message(res));
    res = kndb_did_we_know(p, mf.node_id, mf.point,
                           mf.zp_certbody, mf.zp_signature,
                           mf.certbody, mf.signature, NULL);
    if(res != kndb_res_success)
        message(mlv_normal, "WARNING: couldn't save point(s) as known (%s)\n",
                            kndb_result_message(res));
    dispose_kndb(p); 


    res = 0;

quit:
    crypto_wipe(&mf, sizeof(mf));
    umask(saved_umask);
    dispose_fname(config_fname);
    dispose_fname(secretkey_fname);
    return res;
}


/* secret key isn't const because it gets wiped once used */
static int sign_and_save_point(const char *output_fn, const char *args_dir,
    int point, u_chr *secret_key, const u_chr *public_key, long timestamp)
{
    char *fname;
    FILE *f;
    struct point_config_file conf;
    unsigned char signer_key[secret_key_size];
    char output_fname[32]; /* 20 for node id, 7..9 for '_pNNN.key', 2 resvd */
    const char *ofn;
    int res, errline;
    struct feda_cert_info certinfo;
    unsigned char signature[signature_size];
    int saved_umask;

    fname = make_config_fname(args_dir);
    res = read_config_file(fname, &conf, &errline);
    if(res != rcf_res_ok) {
        pointcfg_message(mlv_normal, res, fname, errline);
        message(mlv_alert, "Couldn't get the point configuration\n");
        message(mlv_info, "do you have the workplace deployed here?\n");
        dispose_fname(fname);
        return 1;
    }
    dispose_fname(fname);
    if(conf.point == -1) {
        message(mlv_alert, "can't zcrpoint/zsign at the master workplace\n");
        return 1;
    }
    if(conf.point != 0 && (
        all_zeroes(conf.zp_certbody, sizeof(conf.zp_certbody)) ||
        all_zeroes(conf.zp_signature, sizeof(conf.zp_signature))))
    {
        message(mlv_alert, "seems we don't have zero point deployed here\n");
        return 1;
    }

    compose_feda_cert(&certinfo,
                      conf.node_id, point, 0, timestamp, public_key);

    fname = conf.point == 0 ?
        make_secretkey_fname(args_dir) : make_zeropointkey_fname(args_dir);
    res = read_single_blob(fname, "secret", signer_key, sizeof(signer_key),
                           &errline);
    if(res != kfiles_res_ok) {
        kfiles_message(mlv_alert, res, fname, errline);
        message(mlv_alert, "can't read the file or there's no secret key\n");
        message(mlv_info, "can't get the zero point secret key\n");
        dispose_fname(fname);
        return 1;
    }
    dispose_fname(fname);

    crypto_eddsa_sign(signature, signer_key,
                      (unsigned char *)&certinfo, sizeof(certinfo));
    crypto_wipe(signer_key, sizeof(signer_key));


    /* okay, now try to save */

    if(output_fn) {
        ofn = output_fn;
    } else {
        hexdata2str(output_fname, conf.node_id, sizeof(conf.node_id));
        sprintf(output_fname+2*node_id_size, "_p%d.key", (int)point);
        ofn = output_fname;
    }

    saved_umask = umask(0077);
    f = fopen(ofn, "w");
    umask(saved_umask);
    if(!f) {
        message_perror(mlv_alert, "FATAL", ofn);
        return 1;
    }

    put_hex_field(f, "node_id", conf.node_id, node_id_size);
    put_int_field(f, "point", point);
    put_int_field(f, "rank", conf.rank);
    put_int_field(f, "minpref", conf.minpref);
    if(secret_key) {
        put_hex_field(f, "secret", secret_key, secret_key_size);
        crypto_wipe(secret_key, secret_key_size);
    }
    put_hex_field(f, "public", public_key, public_key_size);
    put_int_field(f, "signed_by", 0);
    put_hex_field(f, "master_pub", conf.master_pub, public_key_size);
    put_hex_field(f, "master_hash", conf.master_hash, yespower_hash_size);
    put_hex_field(f, "master_hsign", conf.master_hsign, signature_size);
    put_int_field(f, "timestamp", timestamp);
    put_hex_field(f, "certbody", &certinfo, sizeof(certinfo));
    put_hex_field(f, "signature", signature, sizeof(signature));
    if(conf.point == 0) {
        put_hex_field(f, "zp_pub", conf.public_key, public_key_size);
        put_hex_field(f, "zp_certbody", conf.certbody, feda_cert_size);
        put_hex_field(f, "zp_signature", conf.signature, signature_size);
    } else {
        put_hex_field(f, "zp_pub", get_cert_pubkey(conf.zp_certbody),
                                   public_key_size);
        put_hex_field(f, "zp_certbody", conf.zp_certbody, feda_cert_size);
        put_hex_field(f, "zp_signature", conf.zp_signature, signature_size);
    }
    fclose(f);

    message(mlv_normal, "wrote %s; it contains SECRET information!\n", ofn);

    return 0;
}

int main_zcrpoint(const struct fedakeys_args *args)
{
    int res;
    unsigned char seed[seed_size];
    unsigned char secret_key[secret_key_size];
    unsigned char public_key[public_key_size];
    long point;
    char *err;

    if(args->cmdargc < 2 || !args->cmdargs[1] || !*args->cmdargs[1]) {
        message(mlv_alert, "\"fedakeys zcrpoint\" needs the point id\n");
        return 1;
    }
    point = strtol(args->cmdargs[1], &err, 10);
    if(*err != 0 || point < 0 || point > 254) {
        message(mlv_alert, "invalid point id [%s]\n", args->cmdargs[1]);
        return 1;
    }

    res = get_random(seed, sizeof(seed));
    if(!res) {
        message(mlv_alert, "can't read random numbers\n");
        return 1;
    }

    crypto_eddsa_key_pair(secret_key, public_key, seed);

    return sign_and_save_point(args->out_file, args->dir, point,
                               secret_key, public_key, args->timestamp);
}

int main_zsign(const struct fedakeys_args *args)
{
    int res, errline;
    unsigned char public_key[public_key_size];
    long point;
    char *err;

    if(args->cmdargc != 3 || !args->cmdargs[1] || !*args->cmdargs[1]) {
        message(mlv_alert, "\"fedakeys zsign\" needs exactly two args\n");
        return 1;
    }
    point = strtol(args->cmdargs[2], &err, 10);
    if(*err != 0 || point < 0 || point > 253) {
        message(mlv_alert, "invalid point id [%s]\n", args->cmdargs[2]);
        return 1;
    }

    res = read_single_blob(args->cmdargs[1], "public",
                           public_key, sizeof(public_key), &errline);
    if(res != kfiles_res_ok) {
        kfiles_message(mlv_alert, res, args->cmdargs[1], errline);
        message(mlv_alert, "can't read the file or there's no public key\n");
        message(mlv_info, "can't continue without the public key\n");
        return 1;
    }

    return sign_and_save_point(args->out_file, args->dir,
                               point, NULL, public_key, args->timestamp);
}

static char *key2pub_fname(const char *keyfname)
{
    int len;
    char *res;

    len = strlen(keyfname);
    if(len >= 5 && 0 == strcmp(keyfname + len - 4, ".key")) {
        res = malloc(len + 1);
        strcpy(res, keyfname);
        strcpy(res + len - 3, "pub");
    } else {
        res = malloc(len + 5);
        strcpy(res, keyfname);
        strcpy(res + len, ".pub");
    }
    return res;
}

int main_key2pub(const struct fedakeys_args *args)
{
    struct point_file mf;
    int res;
    const char *err;
    char *output_fname;
    const char *ofn;
    FILE *f;

    if(args->cmdargc < 2) {
        message(mlv_alert,
            "\"fedakeys key2pub\" needs a file name, try -h\n");
        return 1;
    }

    res = read_verify_point(args->cmdargs[1], NULL, &mf,
                            args->may_take_hash, &err);
    if(!res) {
        message(mlv_alert, "point file verification FAILED: %s\n", err);
        return 1;
    }

    if(args->out_file) {
        ofn = args->out_file;
    } else {
        output_fname = key2pub_fname(args->cmdargs[1]);
        ofn = output_fname;
    }

    f = fopen(ofn, "w");
    if(!f) {
        message_perror(mlv_alert, "FATAL", ofn);
        res = 1;
        goto quit;
    }
    put_hex_field(f, "node_id", mf.node_id, sizeof(mf.node_id));
    put_int_field(f, "point", mf.point);
    put_int_field(f, "rank", mf.rank);
    put_int_field(f, "minpref", mf.minpref);
    put_int_field(f, "signed_by", mf.signed_by);
    put_hex_field(f, "master_pub", mf.master_pub, sizeof(mf.master_pub));
    put_hex_field(f, "master_hash", mf.master_hash, sizeof(mf.master_hash));
    put_hex_field(f, "master_hsign", mf.master_hsign, sizeof(mf.master_hsign));
    if(mf.signed_by != -1) {
        put_hex_field(f, "zp_certbody", mf.zp_certbody,
                                        sizeof(mf.zp_certbody));
        put_hex_field(f, "zp_signature", mf.zp_signature,
                                         sizeof(mf.zp_signature));
    }
    put_hex_field(f, "certbody", mf.certbody, sizeof(mf.certbody));
    put_int_field(f, "timestamp", mf.timestamp);
    put_hex_field(f, "signature", mf.signature, sizeof(mf.signature));
    put_hex_field(f, "public", mf.public_key, sizeof(mf.public_key));
    fclose(f);
    res = 0;

quit:
    if(output_fname)
        free(output_fname);
    return res;
}

static void get_and_save_keypair(u_chr *seed, FILE *priv, FILE *pub)
{
    unsigned char secret_key[secret_key_size];
    unsigned char public_key[public_key_size];
    crypto_eddsa_key_pair(secret_key, public_key, seed);
    put_hex_field(priv, "secret", secret_key, sizeof(secret_key));
    crypto_wipe(secret_key, sizeof(secret_key));
    put_hex_field(pub, "public", public_key, sizeof(public_key));
}

int main_keypair(const struct fedakeys_args *args)
{
    unsigned char seed[seed_size];
    int res, saved_umask;

    if(args->cmdargc < 2 && !args->out_file) {
        message(mlv_alert, "\"fedakeys keypair\" needs a basename, try -h\n");
        return 1;
    }

    res = get_random(seed, sizeof(seed));
    if(!res) {
        message(mlv_alert, "can't read random numbers\n");
        return 1;
    }

    saved_umask = umask(077);

    if(args->out_file) {
        FILE *f;
        f = fopen(args->out_file, "w");
        if(!f) {
            message_perror(mlv_alert, "FATAL", args->out_file);
            res = 1;
            goto quit;
        }
        get_and_save_keypair(seed, f, f);
        fclose(f);
        res = 0;
    } else {
        FILE *fs, *fp;
        char *fname;
        int bnlen = strlen(args->cmdargs[1]);
        fname = malloc(bnlen + 5);
        strcpy(fname, args->cmdargs[1]);
        strcpy(fname + bnlen, ".key");
        fs = fopen(fname, "w");
        if(!fs) {
            message_perror(mlv_alert, "FATAL", fname);
            free(fname);
            res = 1;
            goto quit;
        }
        strcpy(fname + bnlen, ".pub");
        fp = fopen(fname, "w");
        if(!fs) {
            message_perror(mlv_alert, "FATAL", fname);
            free(fname);
            fclose(fs);
            res = 1;
            goto quit;
        }
        get_and_save_keypair(seed, fs, fp);
        fclose(fs);
        fclose(fp);
    }
    res = 0;
quit:
    umask(saved_umask);
    return res;
}


int main_base32(const struct fedakeys_args *args)
{
    char res[base32_id_max+1];
    int r;
    if(args->cmdargc < 2) {
        message(mlv_alert, "\"fedakeys base32\" needs a node ID, try -h\n");
        return 1;
    }

    r = id_to_base32(res, args->cmdargs[1], node_id_size);
    if(r != 0) {
        message(mlv_alert, "invalid node id\n");
        message(mlv_info, "error position is %d\n", r);
        return 1;
    }
    printf("%s\n", res);
    return 0;
}

