#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <monocypher/monocypher.h>

#include "hexdata.h"
#include "message.h"
#include "txtr_out.h"
#include "textrecx.h"
#include "keyutils.h"
#include "mastrkey.h"
#include "fk_args.h"
#include "fk_data.h"
#include "kfiles.h"
#include "pointcfg.h"
#include "xyespwr.h"
#include "fk_known.h"
#include "knowndb.h"
#include "kfschems.h"
#include "fk_diags.h"
#include "fk_mesg.h"


static int get_config(const char *dir, struct point_config_file *conf)
{
    char *fname;
    int res, errline;

    fname = make_config_fname(dir);
    res = read_config_file(fname, conf, &errline);
    if(res != rcf_res_ok) {
        pointcfg_message(mlv_normal, res, fname, errline);
        message(mlv_alert, "Couldn't get the point configuration\n");
        message(mlv_info, "do you have the workplace deployed here?\n");
        dispose_fname(fname);
        return 0;
    }
    dispose_fname(fname);
    if(conf->point == -1) {
        message(mlv_alert, "can't do this at the master workplace\n");
        return 0;
    }
    return 1;
}

int main_kexgen(const struct fedakeys_args *args)
{
    int res;
    char *main_kex_fn;
    struct kex_keypair_file kex_data;
    FILE *f;
    int saved_umask;
    unsigned char kex_secret[kex_secret_size];
    unsigned char kex_public[kex_public_size];

    main_kex_fn = make_msgkey_fname(args->dir, NULL);
    res = read_kex_keypair_file(main_kex_fn, &kex_data, 1);
    if(res) {   /* this means it's already there so we need to move it */
        char *targ_fn;
        char buf[20];
        hexdata2str(buf, kex_data.kex_public, (sizeof(buf)-1)/2);
        targ_fn = make_msgkey_fname(args->dir, buf);
        res = rename(main_kex_fn, targ_fn);
        if(res == -1) {
            message_perror(mlv_alert, "FATAL: rename", main_kex_fn);
            dispose_fname(targ_fn);
            res = 1;
            goto quit;
        }
        f = fopen(targ_fn, "a");
        if(!f) {
            message_perror(mlv_alert, "WARNING", targ_fn);
            message(mlv_alert, "couln't write ``replaced'' timestamp");
        } else {
            put_int_field(f, "replaced", args->timestamp);
            fclose(f);
        }
        dispose_fname(targ_fn);
    }

    /* the way is clear, we can work */

    res = get_random(kex_secret, sizeof(kex_secret));
    if(!res) {
        message(mlv_alert, "can't read random numbers\n");
        res = 1;
        goto quit;
    }

    crypto_x25519_public_key(kex_public, kex_secret);

    saved_umask = umask(0077);
    f = fopen(main_kex_fn, "w");
    umask(saved_umask);
    if(!f) {
        message_perror(mlv_alert, "FATAL", main_kex_fn);
        return 1;
    }
    put_hex_field(f, "kex_secret", kex_secret, sizeof(kex_secret));
    put_hex_field(f, "kex_public", kex_public, sizeof(kex_public));
    put_int_field(f, "created", args->timestamp);
    fclose(f);
    res = 0;

quit:
    dispose_fname(main_kex_fn);
    return res;
}

int main_nodecert(const struct fedakeys_args *args)
{
    struct point_config_file conf;
    int res;
    char output_fname[32]; /* 20 for node id, 4 for '.pub', 7 resvd */
    const char *ofn;
    FILE *f;

    res = get_config(args->dir, &conf);
    if(!res)    /* everything has already been said by get_config */
        return 1;

    if(args->out_file) {
        ofn = args->out_file;
    } else {
        hexdata2str(output_fname, conf.node_id, sizeof(conf.node_id));
        strcpy(output_fname+2*node_id_size, ".pub");
        ofn = output_fname;
    }

    f = fopen(ofn, "w");
    if(!f) {
        message_perror(mlv_alert, "FATAL", ofn);
        return 1;
    }

    put_hex_field(f, "node_id", conf.node_id, node_id_size);
    put_int_field(f, "rank", conf.rank);
    put_int_field(f, "minpref", conf.minpref);
    put_hex_field(f, "master_pub", conf.master_pub, public_key_size);
    put_hex_field(f, "master_hash", conf.master_hash, yespower_hash_size);
    put_hex_field(f, "master_hsign", conf.master_hsign, signature_size);

    fclose(f);

    message(mlv_normal, "saved to %s\n", ofn);

    return 0;
}

int main_export(const struct fedakeys_args *args)
{
    struct point_config_file conf;
    int res, errline, have_kex;
    char output_fname[32]; /* 20 for node id, 7..9 for '_pNNN.pub', 2 resvd */
    char *fname;
    const char *ofn;
    FILE *f;
    char *main_kex_fn;
    unsigned char secret_key[secret_key_size];
    unsigned char kex_signature[signature_size];
    struct kex_keypair_file kex_data;
    struct feda_kex_info kex_info;
    long pointcert_timestamp;

    res = get_config(args->dir, &conf);
    if(!res)    /* everything has already been said by get_config */
        return 1;

    pointcert_timestamp =
        get_cert_timestamp((const struct feda_cert_info*)&conf.certbody);

    main_kex_fn = make_msgkey_fname(args->dir, NULL);
    res = read_kex_keypair_file(main_kex_fn, &kex_data, 0);
    dispose_fname(main_kex_fn);
    have_kex = res;

    if(!have_kex) {
        message(mlv_normal,
                "NOTE no key-exchg key found, see the ``kexgen'' command\n");
    }

    if(have_kex) {
        compose_kex_info(&kex_info, args->timestamp, kex_data.kex_public);
    
        fname = make_secretkey_fname(args->dir);
        res = read_single_blob(fname, "secret", secret_key, sizeof(secret_key),
                               &errline);
        if(res != kfiles_res_ok) {
            kfiles_message(mlv_alert, res, fname, errline);
            message(mlv_alert, "can't read the point's secret key\n");
            dispose_fname(fname);
            return 1;
        }
        dispose_fname(fname);
    
        crypto_eddsa_sign(kex_signature, secret_key,
                          (unsigned char *)&kex_info, sizeof(kex_info));
        crypto_wipe(secret_key, sizeof(secret_key));
    }

    if(args->out_file) {
        ofn = args->out_file;
    } else {
        hexdata2str(output_fname, conf.node_id, sizeof(conf.node_id));
        sprintf(output_fname+2*node_id_size, "_p%d.pub", conf.point);
        ofn = output_fname;
    }

    f = fopen(ofn, "w");
    if(!f) {
        message_perror(mlv_alert, "FATAL", ofn);
        return 1;
    }

    put_hex_field(f, "node_id", conf.node_id, node_id_size);
    put_int_field(f, "point", conf.point);
    put_int_field(f, "rank", conf.rank);
    put_int_field(f, "minpref", conf.minpref);
    put_int_field(f, "signed_by", conf.signed_by);
    put_hex_field(f, "master_pub", conf.master_pub, public_key_size);
    put_hex_field(f, "master_hash", conf.master_hash, yespower_hash_size);
    if(conf.signed_by == 0) {
        put_hex_field(f, "zp_certbody", conf.zp_certbody,
                                 sizeof(conf.zp_certbody));
        put_hex_field(f, "zp_signature", conf.zp_signature,
                                 sizeof(conf.zp_signature));
    }
    put_int_field(f, "timestamp", pointcert_timestamp);
    put_hex_field(f, "certbody", conf.certbody, sizeof(conf.certbody));
    put_hex_field(f, "signature", conf.signature, sizeof(conf.signature));
    put_hex_field(f, "public", conf.public_key, sizeof(conf.public_key));
    if(have_kex) {
        put_hex_field(f, "kex_public", kex_data.kex_public,
                                       sizeof(kex_data.kex_public));
        put_hex_field(f, "kex_cert", (unsigned char *)&kex_info,
                                     sizeof(kex_info));
        put_hex_field(f, "kex_signature", kex_signature, signature_size);
        put_int_field(f, "kex_timestamp", args->timestamp);
    }

    fclose(f);

    message(mlv_normal, "saved to %s\n", ofn);

    return 0;
}


#if 0
   /* main_{fverify,import} implementation -- old flavor */
static int
verify_import_foreign(int verify_only, const struct fedakeys_args *args)
{
    struct point_cert_file data;
    int res;
    int minrank, rank;

    if(args->cmdargc >= 3 && args->cmdargs[2][0]) {   /* min_rank given */
        char *err;
        minrank = strtol(args->cmdargs[2], &err, 10);
        if(*err != 0 || minrank < minrank_min || minrank > minrank_max) {
            message(mlv_alert, "invalid min. rank [%s]\n", args->cmdargs[2]);
            return 1;
        }
    } else {  /* we need to read the point config in this case only */
        struct point_config_file conf;
        res = get_config(args->dir, &conf);
        if(!res) {
            message(mlv_alert, "can't determine the minimum rank\n");
            return 1;
        }
        minrank = conf.minpref;
    }
    message(mlv_info, "NOTICE: using %d as the minimum rank\n", minrank);

    res = read_pointcert_file(args->cmdargs[1], &data);
    if(!res) {
        message(mlv_alert, "the file isn't found or has invalid format\n");
        return 1;
    }

    rank = check_master_pub(data.node_id, data.master_pub, data.master_hash);
    if(rank < 0) {
        message(mlv_normal, "master pub key verification failed: %s\n",
                            checkmasterpub_message(rank));
        return 1;
    }
    if(rank < minrank) {
        message(mlv_normal, "node rank too low (%d < %d)\n", rank, minrank);
        return 1;
    }

    res = compare_feda_cert((struct feda_cert_info *)data.certbody,
        data.node_id, data.point, data.signed_by, -1,
        data.public_key);
    if(!res) {
        message(mlv_normal, "certbody field malformed\n");
        return 1;
    }

    if(data.signed_by == -1) { /* signed by master, this is simple */
        res = crypto_eddsa_check(data.signature, data.master_pub,
                                 data.certbody, feda_cert_size);
        if(res /* sic! */) {
            message(mlv_normal, "signature check failed\n");
            return 1;
        }
    } else
    if(data.signed_by == 0) { /* signed by Zero Point, check in two steps */
        struct feda_cert_info *zp_cert =
            (struct feda_cert_info *)data.zp_certbody;
        res = crypto_eddsa_check(data.zp_signature, data.master_pub,
                                 data.zp_certbody, feda_cert_size);
        if(res /* sic! */) {
            message(mlv_normal, "zero point cert signature check failed\n");
            return 1;
        }
        res = compare_feda_cert(zp_cert, data.node_id, 0, -1, -1, NULL);
                         /* we ignore both timestamp and pubkey */
        if(!res) {
            message(mlv_normal, "zero point certbody malformed\n");
            return 1;
        }
        res = crypto_eddsa_check(data.signature, zp_cert->key,
                                 data.certbody, feda_cert_size);
        if(res /* sic! */) {
            message(mlv_normal, "certificate signature check failed\n");
            return 1;
        }
    } else {
        message(mlv_normal, "unsupported signed_by value\n");
        return 1;
    }

    res = compare_kex_info((struct feda_kex_info*)data.kex_cert,
                           data.kex_timestamp, data.kex_public);
    if(!res) {
        message(mlv_normal, "malformed key exchange (kex) certificate\n");
        return 1;
    }
    res = crypto_eddsa_check(data.kex_signature, data.public_key,
                             data.kex_cert, feda_kex_info_size);
    if(res /* sic! */) {
        message(mlv_normal, "key exchange cert signature check failed\n");
        return 1;
    }

    /* verification complete */

    message(verify_only ? mlv_normal : mlv_info, "check passed\n");

    if(verify_only)
        return 0;


    message(mlv_alert, "NOT IMPLEMENTED YET\n");
    return 1;
}
#endif

static int arg2_minrank(const struct fedakeys_args *args)
{
    int minrank, res;
    if(args->cmdargc >= 3 && args->cmdargs[2][0]) {   /* min_rank given */
        char *err;
        minrank = strtol(args->cmdargs[2], &err, 10);
        if(*err != 0 || minrank < minrank_min || minrank > minrank_max) {
            message(mlv_alert, "invalid min. rank [%s]\n", args->cmdargs[2]);
            return -1;
        }
    } else {  /* we need to read the point config in this case only */
        struct point_config_file conf;
        res = get_config(args->dir, &conf);
        if(!res) {
            message(mlv_alert, "can't determine the minimum rank\n");
            return 1;
        }
        minrank = conf.minpref;
    }
    message(mlv_info, "NOTICE: using %d as the minimum rank\n", minrank);
    return minrank;
}

static int compare_node_rank(int ckm_res, int minrank, int data_rank)
{
    if(ckm_res < 0) {
        message(mlv_normal, "master pub key verification failed: %s\n",
                            checkmasterpub_message(ckm_res));
        return 0;
    }
    if(ckm_res < minrank) {
        message(mlv_normal, "node rank too low (%d < %d)\n", ckm_res, minrank);
        return 0;
    }
    if(ckm_res != data_rank) {
        message(mlv_normal,
            "node rank %d doesn't match the specified value %d\n",
            ckm_res, data_rank);
        return 0;
    }
    return 1;
}

static int
verify_import_nodecert(int verify_only, const struct fedakeys_args *args)
{
    int res, errline;
    int minrank, rank;
    struct textrecx_data_item *data = NULL;
    const unsigned char *data_node_id;
    const unsigned char *data_master_pub;
    const unsigned char *data_master_hash;
    const unsigned char *data_master_hsign;
    long long data_rank;

    minrank = arg2_minrank(args);
    if(minrank == -1)
        return 1;

    res = read_textrecx_file(args->cmdargs[1], schema_nodecert_public, &data,
                             &errline);
    if(res != kfiles_res_ok) {
        kfiles_message(mlv_info, res, args->cmdargs[1], errline);
        message(mlv_alert, "the file isn't found or has invalid format\n");
        return 1;
    }

    data_node_id      = textrecx_blob_by_name(data, "node_id");
    data_master_pub   = textrecx_blob_by_name(data, "master_pub");
    data_master_hash  = textrecx_blob_by_name(data, "master_hash");
    data_master_hsign = textrecx_blob_by_name(data, "master_hsign");
    res = textrecx_integer_by_name(data, "rank", &data_rank);

    if(!data_node_id || !data_master_pub || !data_master_hash ||
        !data_master_hsign || !res)
    {
            /* actually this must not happen because all the fields are
               checked inside the read_textrecx_file function
             */
        message(mlv_alert, "missing fields (internal error)\n");
        return 1;
    }

    rank = check_master_pub(data_node_id, data_master_pub, data_master_hash,
                            data_master_hsign, args->may_take_hash);
    res = compare_node_rank(rank, minrank, data_rank);
    if(!res)
        return 1;

    /* verification complete */

    message(verify_only ? mlv_normal : mlv_info, "check passed\n");

    if(verify_only)
        return 0;

    res = add_known_nodecert(args->dir, data);
    if(!res) {
        message(mlv_debug, "couldn't save the node as known\n");
        return 1;
    }

    return 0;
}

int main_nverify(const struct fedakeys_args *args)
{
    if(args->cmdargc < 2) {
        message(mlv_alert, "\"fedakeys nverify\" needs a file name, try -h\n");
        return 1;
    }
    return verify_import_nodecert(1, args);
}

int main_nodeimport(const struct fedakeys_args *args)
{
    if(args->cmdargc < 2) {
        message(mlv_alert,
            "\"fedakeys nodeimport\" needs a file name, try -h\n");
        return 1;
    }
    return verify_import_nodecert(0, args);
}



static int
verify_import_foreign(int verify_only, const struct fedakeys_args *args)
{
    int res, res2, res3, errline, have_kex;
    int minrank, rank;
    struct textrecx_data_item *data = NULL;
    const unsigned char *data_node_id;
    const unsigned char *data_master_pub;
    const unsigned char *data_master_hash;
    const unsigned char *data_master_hsign;
    const unsigned char *data_certbody;
    const unsigned char *data_signature;
    const unsigned char *data_public;
    const unsigned char *data_kex_public;
    const unsigned char *data_kex_cert;
    const unsigned char *data_kex_signature;
    long long data_point, data_signed_by, data_kex_timestamp, data_rank;

    minrank = arg2_minrank(args);
    if(minrank == -1)
        return 1;

    res = read_textrecx_file(args->cmdargs[1], schema_point_public, &data,
                             &errline);
    if(res != kfiles_res_ok) {
        kfiles_message(mlv_info, res, args->cmdargs[1], errline);
        message(mlv_alert, "the file isn't found or has invalid format\n");
        return 1;
    }

    data_node_id      = textrecx_blob_by_name(data, "node_id");
    data_master_pub   = textrecx_blob_by_name(data, "master_pub");
    data_master_hash  = textrecx_blob_by_name(data, "master_hash");
    data_master_hsign = textrecx_blob_by_name(data, "master_hsign");
    data_certbody     = textrecx_blob_by_name(data, "certbody");
    data_signature    = textrecx_blob_by_name(data, "signature");
    data_public       = textrecx_blob_by_name(data, "public");
    res  = textrecx_integer_by_name(data, "point", &data_point);
    res2 = textrecx_integer_by_name(data, "signed_by", &data_signed_by);
    res3 = textrecx_integer_by_name(data, "rank", &data_rank);

    if(!data_node_id || !data_master_pub || !data_master_hash ||
        !data_master_hsign ||
        !data_certbody || !data_signature || !data_public ||
        !res || !res2 || !res3)
    {
            /* actually this must not happen because all the fields are
               checked inside the read_textrecx_file function
             */
        message(mlv_alert, "missing fields (internal error)\n");
        return 1;
    }

    rank = check_master_pub(data_node_id, data_master_pub, data_master_hash,
                            data_master_hsign, args->may_take_hash);
    res = compare_node_rank(rank, minrank, data_rank);
    if(!res)
        return 1;

    if(data_point == 0 && data_signed_by != -1) {
        message(mlv_normal, "zero point key must be signed by the master\n");
        return 1;
    }

    res = compare_feda_cert((struct feda_cert_info *)data_certbody,
        data_node_id, data_point, data_signed_by, -1,
        data_public);
    if(!res) {
        message(mlv_normal, "certbody field malformed\n");
        return 1;
    }

    if(data_signed_by == -1) { /* signed by master, this is simple */
        res = crypto_eddsa_check(data_signature, data_master_pub,
                                 data_certbody, feda_cert_size);
        if(res /* sic! */) {
            message(mlv_normal, "signature check failed\n");
            return 1;
        }
    } else
    if(data_signed_by == 0) { /* signed by Zero Point, check in two steps */
        const unsigned char *data_zp_certbody;
        const unsigned char *data_zp_signature;
        struct feda_cert_info *zp_cert;

        data_zp_certbody    = textrecx_blob_by_name(data, "zp_certbody");
        data_zp_signature   = textrecx_blob_by_name(data, "zp_signature");
        if(!data_zp_certbody || !data_zp_signature) {
            message(mlv_normal, "zero point cert or signature missing\n");
            return 1;
        }

        zp_cert = (struct feda_cert_info *)data_zp_certbody;

        res = crypto_eddsa_check(data_zp_signature, data_master_pub,
                                 data_zp_certbody, feda_cert_size);
        if(res /* sic! */) {
            message(mlv_normal, "zero point cert signature check failed\n");
            return 1;
        }
        res = compare_feda_cert(zp_cert, data_node_id, 0, -1, -1, NULL);
                         /* we ignore both timestamp and pubkey */
        if(!res) {
            message(mlv_normal, "zero point certbody malformed\n");
            return 1;
        }
        res = crypto_eddsa_check(data_signature, zp_cert->key,
                                 data_certbody, feda_cert_size);
        if(res /* sic! */) {
            message(mlv_normal, "certificate signature check failed\n");
            return 1;
        }
    } else {
        message(mlv_normal, "unsupported signed_by value\n");
        return 1;
    }

    data_kex_public    = textrecx_blob_by_name(data, "kex_public");
    data_kex_cert      = textrecx_blob_by_name(data, "kex_cert");
    data_kex_signature = textrecx_blob_by_name(data, "kex_signature");
    res = textrecx_integer_by_name(data, "kex_timestamp", &data_kex_timestamp);

    have_kex = data_kex_public && data_kex_cert && data_kex_signature && res;
    if(have_kex) {
        res = compare_kex_info((struct feda_kex_info*)data_kex_cert,
                               data_kex_timestamp, data_kex_public);
        if(!res) {
            message(mlv_normal, "malformed key exchange (kex) certificate\n");
            return 1;
        }
        res = crypto_eddsa_check(data_kex_signature, data_public,
                                 data_kex_cert, feda_kex_info_size);
        if(res /* sic! */) {
            message(mlv_normal, "key exchange cert signature check failed\n");
            return 1;
        }
    } else {
        if(data_kex_public || data_kex_cert || data_kex_signature || res) {
            message(mlv_normal, "some kex_* fields present but not all\n");
            return 1;
        }
        message(mlv_info, "no key exchange information found; it's okay\n");
    }

    /* verification complete */

    message(verify_only ? mlv_normal : mlv_info, "check passed\n");

    if(verify_only)
        return 0;

    res = add_known_cert(args->dir, data, have_kex);
    if(!res) {
        message(mlv_debug, "couldn't save the new cert as known\n");
        return 1;
    }

    return 0;
}

int main_fverify(const struct fedakeys_args *args)
{
    if(args->cmdargc < 2) {
        message(mlv_alert, "\"fedakeys fverify\" needs a file name, try -h\n");
        return 1;
    }
    return verify_import_foreign(1, args);
}

int main_import(const struct fedakeys_args *args)
{
    if(args->cmdargc < 2) {
        message(mlv_alert, "\"fedakeys import\" needs a file name, try -h\n");
        return 1;
    }
    return verify_import_foreign(0, args);
}



static int is_single_dash(const char *s)
{
    return s[0] == '-' && s[1] == 0;
}

    /* NULL for fname means stdin; returns boolean */
static int blake2sum_of_file(unsigned char *hash, int hlen, const char *fname)
{
    crypto_blake2b_ctx ctx;
    static unsigned char buf[4096];
    int fd, rc, need_close;

    if(fname) {                    /* file is specified */
        fd = open(fname, O_RDONLY);
        if(fd == -1) {
            message_perror(mlv_alert, NULL, fname);
            return 0;
        }
        need_close = 1;
    } else {
        fd = 0;
        need_close = 0;
    }

    crypto_blake2b_init(&ctx, hlen);
    while((rc = read(fd, buf, sizeof(buf))) > 0)
        crypto_blake2b_update(&ctx, buf, rc);
    crypto_blake2b_final(&ctx, hash);

    if(need_close)
        close(fd);
    return 1;
}

int main_blake2sum(const struct fedakeys_args *args)
{
    int res, hashlen = 32;
    unsigned char hash[64];
    const char *fname;

    if(args->cmdargc >= 3) { /* hash len given */
        char *err;
        hashlen = strtol(args->cmdargs[2], &err, 10);
        if(*err != 0 || hashlen < 1 || hashlen > 64) {
            message(mlv_alert, "invalid hash length [%s]\n", args->cmdargs[2]);
            return 1;
        }
    }
    if(args->cmdargc < 2 || is_single_dash(args->cmdargs[1])) /* no file */
        fname = NULL;
    else
        fname = args->cmdargs[1];

    res = blake2sum_of_file(hash, hashlen, fname);
    if(!res)
        return 1;

    fputs(hexdata2a(hash, hashlen), stdout);
    fputs("\n", stdout);
    return 0;
}


struct sign_file_names {
    const char *datafname, *sigfname;
    char *fnhold;
};

static void determine_sign_file_names(const struct fedakeys_args *args,
                                      struct sign_file_names *fnames)
{
    fnames->fnhold = NULL;

    if(args->cmdargc < 2 || is_single_dash(args->cmdargs[1]))
        fnames->datafname = NULL;
    else
        fnames->datafname = args->cmdargs[1];

    if(args->cmdargc < 3) {
        if(fnames->datafname) {
            int fnlen = strlen(fnames->datafname);
            fnames->fnhold = malloc(fnlen + 5);
            strcpy(fnames->fnhold, fnames->datafname);
            strcpy(fnames->fnhold + fnlen, ".sig");
            fnames->sigfname = fnames->fnhold;
        } else {
            fnames->sigfname = "__stdout__.sig";
        }
    } else
    if(is_single_dash(args->cmdargs[2])) {  /* stdout */
        fnames->sigfname = NULL;
    } else {
        fnames->sigfname = args->cmdargs[2];
    }
}

static void dispose_sign_file_names(struct sign_file_names *fnames)
{
    if(fnames->fnhold)
        free(fnames->fnhold);
}

int main_sign(const struct fedakeys_args *args)
{
    char *fname;
    unsigned char hash[68];  /* 64 for the hash, 4 for the timestamp */
    unsigned char signature[signature_size];
    unsigned char secret_key[secret_key_size];
    struct point_config_file conf;
    int res, errline;
    struct sign_file_names sfn;
    FILE *f, *fh;

    determine_sign_file_names(args, &sfn);

    res = get_config(args->dir, &conf);
    if(!res) {
        res = 1;
        goto quit;
    }

    fname = make_secretkey_fname(args->dir);
    res = read_single_blob(fname, "secret", secret_key, sizeof(secret_key),
                           &errline);
    if(res != kfiles_res_ok) {
        kfiles_message(mlv_alert, res, fname, errline);
        message(mlv_alert, "can't read the point's secret key\n");
        dispose_fname(fname);
        return 1;
    }
    dispose_fname(fname);

    res = blake2sum_of_file(hash, 64, sfn.datafname);
    if(!res) {
        res = 1;
        goto quit;
    }
    place_timemark(args->timestamp, hash + 64);
    crypto_eddsa_sign(signature, secret_key, hash, sizeof(hash));
    crypto_wipe(secret_key, sizeof(secret_key));

    if(sfn.sigfname) {
        fh = fopen(sfn.sigfname, "w");
        if(!fh) {
            message_perror(mlv_alert, NULL, sfn.sigfname);
            res = 1;
            goto quit;
        }
        f = fh;
    } else {
        f = stdout;
        fh = NULL;
    }

    textrecx_serialize_string(sfn.datafname ? sfn.datafname : "*stdin*",
                             "filename",
                              fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(conf.node_id, node_id_size, "signer_node",
                            fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_integer(conf.point, "signer_point",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(conf.master_pub, public_key_size, "master_pub",
                            fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(conf.master_hash,yespower_hash_size,"master_hash",
                            fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(conf.master_hsign, signature_size, "master_hsign",
                            fieldname_width, putchar_cb_fputc, f);
    if(conf.signed_by == 0) {
        textrecx_serialize_blob(conf.zp_certbody, feda_cert_size, "zp_cert",
                                fieldname_width, putchar_cb_fputc, f);
        textrecx_serialize_blob(conf.zp_signature,signature_size,"zp_certsig",
                                fieldname_width, putchar_cb_fputc, f);
    }
    textrecx_serialize_blob(conf.certbody, feda_cert_size, "point_cert",
                            fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(conf.signature, signature_size, "point_certsig",
                            fieldname_width, putchar_cb_fputc, f);

    textrecx_serialize_blob(hash, sizeof(hash), "hash",
                            fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_integer(args->timestamp, "timestamp",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(signature, signature_size, "signature",
                            fieldname_width, putchar_cb_fputc, f);

    if(fh)
        fclose(fh);
    res = 0;
quit:
    dispose_sign_file_names(&sfn);
    return res;
}


static int check_save_node(struct known_nodes_db *kndb,
                           const unsigned char *node_id,
                           const unsigned char *master_pub,
                           const unsigned char *master_hash,
                           const unsigned char *master_hsign)
{
    int res;
    unsigned char computed_hash[yespower_hash_size];

    res = take_yespower_hash(computed_hash, master_pub);
    if(!res) {
        message(mlv_alert, "can't take yespower hash\n");
        return 0;
    }
    res = kndb_check_save_node(kndb, node_id, master_pub, master_hash,
                               master_hsign, computed_hash);
    if(res != kndb_res_success) {
        message(mlv_alert, "node credentials rejected (%s)\n",
                            kndb_result_message(res));
        return 0;
    }
    return 1;
}

int main_signverify(const struct fedakeys_args *args)
{
#if 0
    message(mlv_alert, "NOT IMPLEMENTED YET\n");
    return 1;
#else
    int res;
    struct known_nodes_db *kndb = NULL;
    struct point_config_file conf;
    int minpref;
    struct sign_file_names sfn;
    struct textrecx_data_item *data = NULL;
    unsigned char hash[68];  /* 64 for the hash, 4 for the timestamp */
    unsigned char pubkey[public_key_size];
    long long data_timestamp;
    const unsigned char *data_hash;
    const unsigned char *data_node_id;
    const unsigned char *data_master_pub;
    const unsigned char *data_master_hash;
    const unsigned char *data_master_hsign;
    long long data_point;
    const unsigned char *data_zp_certbody;
    const unsigned char *data_zp_signature;
    const unsigned char *data_point_certbody;
    const unsigned char *data_point_signature;
    const unsigned char *data_sig;
    

    determine_sign_file_names(args, &sfn);

    res = get_config(args->dir, &conf);
    if(!res) {
        message(mlv_alert,
            "can't determine the minimum rank, do you have a point here?\n");
        res = 1;
        goto quit;
    }
    minpref = conf.minpref;

    res = read_textrecx_file(sfn.sigfname, schema_signature_file, &data, 0);
    if(res != kfiles_res_ok) {
        message(mlv_alert, "problems reading the signature file (%s)\n",
                           kfiles_ctx_error_message(res));
        res = 1;
        goto quit;
    }
    res = blake2sum_of_file(hash, 64, sfn.datafname);
    if(!res) {
        message(mlv_alert, "couldn't read the signed (data) file\n");
        res = 1;
        goto quit;
    }

    res = textrecx_integer_by_name(data, "timestamp", &data_timestamp);
    if(!res) {
        message(mlv_alert, "couldn't get the timestamp from the sign.file\n");
        res = 1;
        goto quit;
    }
    place_timemark(data_timestamp, hash + 64);
    data_hash = textrecx_blob_by_name(data, "hash");
    if(!data_hash || 0 != memcmp(hash, data_hash, sizeof(hash))) {
        message(mlv_alert, "hash doesn't match the file or the timestamp\n");
        res = 1;
        goto quit;
    }

    kndb = make_kndb(args->dir, minpref);

    data_node_id = textrecx_blob_by_name(data, "signer_node");
    data_master_pub = textrecx_blob_by_name(data, "master_pub");
    data_master_hash = textrecx_blob_by_name(data, "master_hash");
    data_master_hsign = textrecx_blob_by_name(data, "master_hsign");
    if(!data_node_id || !data_master_pub ||
        !data_master_hash || !data_master_hsign
    ) {
        message(mlv_alert, "broken format of the sign. file (node creds?)\n");
        res = 1;
        goto quit;
    }

    res = kndb_consider_node(kndb, data_node_id, data_master_pub);
    switch(res) {
    case kndb_res_success:
        break;
    case kndb_res_node_unknown:
        if(args->may_take_hash) {
            res = check_save_node(kndb, data_node_id, data_master_pub,
                                  data_master_hash, data_master_hsign);
            if(res)
                break;
            res = 1;
            goto quit;
        } else {
            message(mlv_alert,
                "node unknown and taking hashes disabled, can't check\n");
            res = 1;
            goto quit;
        }
    case kndb_res_node_key_differs:
        message(mlv_alert, "we know the node and its key is different\n");
        res = 1;
        goto quit;
    default:
        message(mlv_alert, "something wrong happend (bug in the code)\n");
        res = 1;
        goto quit;
    }

    res  = textrecx_integer_by_name(data, "signer_point", &data_point);
    data_zp_certbody = textrecx_blob_by_name(data, "zp_cert");
    data_zp_signature = textrecx_blob_by_name(data, "zp_certsig");
    data_point_certbody = textrecx_blob_by_name(data, "point_cert");
    data_point_signature = textrecx_blob_by_name(data, "point_certsig");
    data_sig = textrecx_blob_by_name(data, "signature");

    if(!res || !data_point_certbody || !data_point_signature || !data_sig) {
        message(mlv_alert, "credentials incomplete\n");
        res = 1;
        goto quit;
    }

    res = kndb_did_we_know(kndb, data_node_id, data_point,
                           data_zp_certbody, data_zp_signature,
                           data_point_certbody, data_point_signature,
                           pubkey);
    if(res != kndb_res_success) {
        message(mlv_alert,
            "point credentials rejected by the database of known certs\n");
        res = 1;
        goto quit;
    }

    res = crypto_eddsa_check(data_sig, pubkey, hash, sizeof(hash)); 
    if(res == 0  /* sic! */) {
        message(mlv_normal, "signature check passed\n");
        res = 0;
    } else {
        message(mlv_normal, "signature check failed\n");
        res = 1;
    }
quit:
        /* no secret keys involved, nothing to wipe */
    if(data)
        textrecx_dispose_data(data);
    if(kndb)
        dispose_kndb(kndb);
    dispose_sign_file_names(&sfn);
    return res;
#endif
}

int main_lock(const struct fedakeys_args *args)
{
    message(mlv_alert, "NOT IMPLEMENTED YET\n");
    return 1;
}

int main_anonlock(const struct fedakeys_args *args)
{
    message(mlv_alert, "NOT IMPLEMENTED YET\n");
    return 1;
}

int main_unlock(const struct fedakeys_args *args)
{
    message(mlv_alert, "NOT IMPLEMENTED YET\n");
    return 1;
}
