#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>

#include <monocypher/monocypher.h>
#include <yespower/yespower.h>

/* this should be the only module of fedakeys to include yespower.h */

#include "xyespwr.h"
#include "keyutils.h"
#include "fedapref.h"
#include "message.h"
#include "txtr_out.h"
#include "hexdata.h"
#include "fileutil.h"
#include "mastrkey.h"
#include "xmonocyp.h"
#include "pointcfg.h"
#include "fk_args.h"
#include "fk_diags.h"
#include "fk_data.h"

#include "fk_mastr.h"


static int
read_verify_master(const char *fname, struct master_file *mf,
                   int may_take_hash, const char **err)
{
    int res, preflen;

    res = read_master_file(fname, mf);
    if(!res) {
        *err = "can't open the file or it is not in proper format";
        return 0;
    }

    res = check_keypair(mf->secret, mf->public_key);
    if(!res) {
        *err = "public and secret keys don't match";
        return 0;
    }

    preflen = check_master_pub(mf->node_id, mf->public_key, mf->yeshash,
                               NULL, may_take_hash);
    if(preflen < 0) {
        *err = checkmasterpub_message(preflen);
        return 0;
    }

    if(preflen != mf->preflen) {
        *err = "0xFEDA prefix length doesn't match\n";
        return 0;
    }

    /* all checks passed, everything's fine */
    return 1;
}

static const char *preflen_comment(int len)
{
    if(len < 12)
        return "way too short";
    if(len < 16)
        return "too short";
    if(len < 19)
        return "pretty short";
    if(len < 21)
        return "not too long";
    if(len == 21)
        return "reasonable";
    if(len == 22)
        return "good";
    if(len == 23)
        return "very good";
    if(len == 24)
        return "perfect";
    if(len == 25)
        return "excellent";
    if(len < 32)
        return "unbelievably excellent";
    return "impossible to believe it really exists";
}


int main_mverify(const struct fedakeys_args *args)
{
    struct master_file mf;
    int res;
    const char *err;

    if(args->cmdargc < 2) {
        message(mlv_alert, "\"fedakeys mverify\" needs a file name, try -h\n");
        return 1;
    }

    if(!args->may_take_hash)
        message(mlv_normal,
            "WARNING not allowed to take hash, will trust it\n");

    res = read_verify_master(args->cmdargs[1], &mf, args->may_take_hash, &err);
    if(!res) {
        message(mlv_normal, "VERIFICATION FAILED: %s\n", err);
        return 1;
    }

    message(mlv_normal, "hash prefix length: %d, which is %s\n",
                        mf.preflen, preflen_comment(mf.preflen));
    message(mlv_normal, "verification passed\n");

    return 0;
}




/* ------------------------------------------------------------------ */



int main_mdeploy(const struct fedakeys_args *args)
{
    struct master_file mf;
    int res, minpreflen;
    const char *err;
    char *masterkey_fname, *config_fname;
    FILE *f;
    int saved_umask;
    unsigned char hash_sign[signature_size];

    if(args->cmdargc < 2) {
        message(mlv_alert, "\"fedakeys mdeploy\" needs a file name, try -h\n");
        return 1;
    }

        /* now we grab all we need, and only quit via 'quit:' from now on */

    masterkey_fname = make_masterkey_fname(args->dir);
    config_fname = make_config_fname(args->dir);
    saved_umask = umask(0077);
    if(!args->may_take_hash)
        message(mlv_normal, "will trust the hash; make sure you checked it\n");
    res = read_verify_master(args->cmdargs[1], &mf, args->may_take_hash, &err);
    if(!res) {
        message(mlv_alert, "master file verification FAILED: %s\n", err);
        res = 1;
        goto quit;
    }

    if(args->cmdargc >= 3 && args->cmdargs[2][0]) {   /* minpreflen given */
        char *err;
        minpreflen = strtol(args->cmdargs[2], &err, 10);
        if(*err != 0 || minpreflen < minrank_min || minpreflen > minrank_max) {
            message(mlv_alert, "invalid minimum prefix len [%s]\n",
                               args->cmdargs[2]);
            res = 1;
            goto quit;
        }
    } else {
        minpreflen = mf.preflen;
    }
    message(mlv_info,"NOTICE: setting min. prefix length to %d\n",minpreflen);

    if(file_is_there(masterkey_fname)) {
        message(mlv_alert,
                "FATAL: %s already exists, move/delete it and retry\n",
                masterkey_fname);
        res = 1;
        goto quit;
    }

    if(file_is_there(config_fname)) {
        message(mlv_alert,
                "FATAL: %s already exists, move/delete it and retry\n",
                config_fname);
        res = 1;
        goto quit;
    }

    res = make_directory_path(args->dir, 0);
    if(res == -1) {
        message_perror(mlv_alert, "FATAL", args->dir);
        res = 1;
        goto quit;
    }

    f = fopen(masterkey_fname, "w");
    if(!f) {
        message_perror(mlv_alert, "FATAL", masterkey_fname);
        res = 1;
        goto quit;
    }
    put_hex_field(f, "secret", mf.secret, sizeof(mf.secret));
    fclose(f);

    crypto_eddsa_sign(hash_sign, mf.secret, mf.yeshash, sizeof(mf.yeshash));

    f = fopen(config_fname, "w");
    if(!f) {
        message_perror(mlv_alert, "FATAL", config_fname);
        res = 1;
        goto quit;
    }
    put_hex_field(f, "node_id", mf.node_id, sizeof(mf.node_id));
    put_int_field(f, "rank", mf.preflen);
    put_int_field(f, "point", -1);
    put_int_field(f, "minpref", minpreflen);
    put_hex_field(f, "master_pub", mf.public_key, sizeof(mf.public_key));
    put_hex_field(f, "master_hash", mf.yeshash, sizeof(mf.yeshash));
    put_hex_field(f, "master_hsign", hash_sign, sizeof(hash_sign));
    fclose(f);
    res = 0;

quit:
    crypto_wipe(&mf, sizeof(mf));
    umask(saved_umask);
    dispose_fname(config_fname);
    dispose_fname(masterkey_fname);
    return res;
}

/* ------------------------------------------------------------------ */


static int
do_mpoint(const char *dir, int point, const char *out_file, long timestamp)
{
    char *masterkey_fname, *config_fname;
    int saved_umask;
    FILE *f;
    struct point_config_file conf;
    unsigned char master_secret[secret_key_size];
    char output_fname[32]; /* 20 for node id, 7..9 for '_pNNN.key', 2 resvd */
    const char *ofn;
    int res, errline;
    struct feda_cert_info certinfo;
    unsigned char seed[seed_size];
    unsigned char secret_key[secret_key_size];
    unsigned char public_key[public_key_size];
    unsigned char signature[signature_size];

    masterkey_fname = make_masterkey_fname(dir);
    config_fname = make_config_fname(dir);
    saved_umask = umask(0077);

    res = read_config_file(config_fname, &conf, &errline);
    if(res != rcf_res_ok) {
        pointcfg_message(mlv_normal, res, config_fname, errline);
        message(mlv_alert, "Couldn't get the point configuration\n");
        message(mlv_info, "do you have the workplace deployed here?\n");
        res = 1;
        goto quit;
    }
    res = read_single_blob(masterkey_fname, "secret",
                           master_secret, sizeof(master_secret), &errline);
    if(res != kfiles_res_ok) {
        kfiles_message(mlv_alert, res, masterkey_fname, errline);
        message(mlv_info, "can't get the master key, do you have it?\n");
        res = 1;
        goto quit;
    }

    res = get_random(seed, sizeof(seed));
    if(!res) {
        message(mlv_alert, "can't read random numbers\n");
        res = 1;
        goto quit;
    }
    crypto_eddsa_key_pair(secret_key, public_key, seed);

    /* we're ready, now fill the cert structure */

    compose_feda_cert(&certinfo,
                      conf.node_id, point, 0xff, timestamp, public_key);

    /* sign it */

    crypto_eddsa_sign(signature, master_secret,
                      (unsigned char *)&certinfo, sizeof(certinfo));

    /* okay, now try to save */
    if(out_file) {
        ofn = out_file;
    } else {
        hexdata2str(output_fname, conf.node_id, sizeof(conf.node_id));
        sprintf(output_fname+2*node_id_size, "_p%d.key", point);
        ofn = output_fname;
    }

    f = fopen(ofn, "w");
    if(!f) {
        message_perror(mlv_alert, "FATAL", ofn);
        res = 1;
        goto quit;
    }

    put_hex_field(f, "node_id", conf.node_id, node_id_size);
    put_int_field(f, "rank", conf.rank);
    put_int_field(f, "point", point);
    put_int_field(f, "minpref", conf.minpref);
    put_hex_field(f, "secret", secret_key, secret_key_size);
    put_hex_field(f, "public", public_key, public_key_size);
    put_int_field(f, "signed_by", -1);
    put_hex_field(f, "master_pub", conf.master_pub, public_key_size);
    put_hex_field(f, "master_hash", conf.master_hash, yespower_hash_size);
    put_hex_field(f, "master_hsign", conf.master_hsign,
                                         signature_size);
    put_int_field(f, "timestamp", timestamp);
    put_hex_field(f, "certbody", &certinfo, sizeof(certinfo));
    put_hex_field(f, "signature", signature, sizeof(signature));
    fclose(f);

    message(mlv_normal, "wrote %s; it contains SECRET information!\n", ofn);

    res = 0;

quit:
    crypto_wipe(secret_key, sizeof(secret_key));
    crypto_wipe(master_secret, sizeof(master_secret));
    umask(saved_umask);
    dispose_fname(config_fname);
    dispose_fname(masterkey_fname);
    return res;
}

int main_mzero(const struct fedakeys_args *args)
{
    return do_mpoint(args->dir, 0, args->out_file, args->timestamp);
}

int main_mpoint(const struct fedakeys_args *args)
{
    long point;
    char *err;

    if(args->cmdargc < 2 || !args->cmdargs[1] || !*args->cmdargs[1]) {
        message(mlv_alert, "\"fedakeys mpoint\" needs the point id, try -h\n");
        return 1;
    }

    point = strtol(args->cmdargs[1], &err, 10);
    if(*err != 0 || point < 0 || point > 254) {
        message(mlv_alert, "invalid point id [%s]\n", args->cmdargs[1]);
        return 1;
    }

    return do_mpoint(args->dir, point, args->out_file, args->timestamp);
}
