#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "cryptodf.h"
#include "hexdata.h"
#include "fileutil.h"
#include "keyutils.h"
#include "kfiles.h"
#include "kfschems.h"
#include "knowndb.h"
#include "message.h"

#include "fk_known.h"


/* *******************************************************************

    * OBSOLETED!!! see the file knowndb.c *

For, e.g., node 72d5c132de46e81193bf:

($WORKDIR stands for $HOME/.fedanet/keys or whatever is set with -c)


$WORKDIR/72d/5c1/72d5c132de46e81193bf/              is the node directory


$WORKDIR/72d/5c1/72d5c132de46e81193bf/node          "global" node information
          * node_id
          * rank
          * minpref
          * master_pub
          * master_hash


$WORKDIR/72d/5c1/72d5c132de46e81193bf/0             actual point information
$WORKDIR/72d/5c1/72d5c132de46e81193bf/1
$WORKDIR/72d/5c1/72d5c132de46e81193bf/47
$WORKDIR/72d/5c1/72d5c132de46e81193bf/253
          * node_id
          * point
          * timestamp
          * signed_by
          * public
          * certbody
          * signature


$WORKDIR/72d/5c1/72d5c132de46e81193bf/47.29115286   old point information


$WORKDIR/72d/5c1/72d5c132de46e81193bf/47.kex        point key exchange pubkey
          * kex_public
          * kex_cert
          * kex_signature
          * kex_timestamp
          * ptk_timestamp


NB: old key exchange keys are not stored in any way, it is senseless as
they are only needed for encrypting messages/files intended for the
particular party, and for this purpose, only the actual (latest known) key
must be used.

   ******************************************************************* */





int get_known_cert(const char *wdir, const unsigned char node_id[], int point,
                   struct textrecx_data_item **the_res)
{
    int res;
    struct kc_path_set ps;
    struct textrecx_data_item *data = NULL, *data2 = NULL;

    make_path_set(wdir, node_id, point, &ps);

#if 0
    res = read_from_point_cert_file(ps.node_file, cert);
#endif

    res = read_textrecx_file(ps.node_file, schema_known_node, &data, 0);
    if(!res)
        goto quit;

    if(point >= 0 && point <= max_point_number) {
        res = read_textrecx_file(ps.point_file, schema_known_point, &data2, 0);
        if(!res)
            goto quit;
        textrecx_merge_data(&data, &data2);
        res = read_textrecx_file(ps.point_kex_file,
                                 schema_known_point_kex, &data2, 0);
        if(res)
            textrecx_merge_data(&data, &data2);
    }

    res = 1;
    *the_res = data;
    data = NULL;
quit:
    if(data)
        textrecx_dispose_data(data);
    if(data2)
        textrecx_dispose_data(data2);
    dispose_path_set(&ps);
    return res;
}

static int
do_write_fields(FILE *f, struct textrecx_data_item *data, const char *fields[])
{
    int res, i;
    res = 1;
    for(i = 0; fields[i]; i++) {
        struct textrecx_data_item *item;
        item = textrecx_data_by_name(data, fields[i]);
        if(!item) {
            message(mlv_alert, "field %s not found (int. err.)\n", fields[i]);
            res = 0;
            continue;
        }
        res = textrecx_serialize_data_item(item, fieldname_width, NULL,
                                           putchar_cb_fputc, f);
        if(res != textrecx_res_ok) {
            message(mlv_alert, "problem writing field %s\n", fields[i]);
            res = 0;
        }
    }
    return res;
}

static int
write_fields_to_file(const char *fname, struct textrecx_data_item *data,
                     const char *fields[])
{
    FILE *f;

    f = fopen(fname, "w");
    if(!f) {
        message_perror(mlv_alert, "write_fields_to_file", fname);
        return 0;
    }

    do_write_fields(f, data, fields);  /* yes, we ignore the result */

    fclose(f);
    message(mlv_debug, "saved in %s\n", fname);
    return 1;
}

static int
write_known_node_file(const char *fname, struct textrecx_data_item *data)
{
    static const char *fields[] = {
        "node_id", "rank", "master_pub", "master_hash", "master_hsign", NULL
    };
    return write_fields_to_file(fname, data, fields);
}


#if 0
int forcibly_save_node(const char *wdir, const unsigned char *node_id,
                                         const unsigned char *master_pub,
                                         int rank,
                                         const unsigned char *master_hash)
{
    int res;
    struct kc_path_set ps;
    FILE *f;

    make_path_set(wdir, node_id, 0, &ps);
    res = make_directory_path(ps.node_dir, 0);
    if(res == -1) {
        res = 0;
        goto quit;
    }
    f = fopen(ps.node_file, "w");
    if(!f) {
        res = 0;
        goto quit;
    }

    textrecx_serialize_blob(node_id, node_id_size, "node_id",
                            fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_integer(rank, "rank",
                            fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(master_pub, public_key_size, "master_pub",
                            fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(master_hash, yespower_hash_size, "master_hash",
                            fieldname_width, putchar_cb_fputc, f);

    /* we don't write minpref here -- it might change, so it doesn't belong
       to the node file (which is, once saved, should never ever change */

    fclose(f);

    res = 1;
quit:
    dispose_path_set(&ps);
    return res;
}
#endif

static int
write_known_point_file(const char *fname, struct textrecx_data_item *data)
{
    static const char *fields[] = {
        "node_id", "point", "timestamp", "signed_by",
        "public", "certbody", "signature", NULL
    };
    return write_fields_to_file(fname, data, fields);
}

static int
check_save_known_node(const char *fname, struct textrecx_data_item *data)
{
    int res, errline;
    struct textrecx_data_item *dnode = NULL;

    res = read_textrecx_file(fname, schema_known_node, &dnode, &errline);
    if(res == kfiles_res_ok) { /* yes, we do know this node */
        const unsigned char *master_pub;
        const unsigned char *existing_master_pub;

        master_pub = textrecx_blob_by_name(data, "master_pub");
        existing_master_pub = textrecx_blob_by_name(dnode, "master_pub");
        if(!existing_master_pub) {
            message(mlv_alert, "the file %s has no master_pub (int. error)\n",
                    fname);
            res = 0;
            goto quit;
        }
        if(0 != memcmp(master_pub, existing_master_pub, public_key_size)) {
            message(mlv_alert,
                    "master public key differs from the known (%s), abort\n",
                    fname);
            res = 0;
            goto quit;
        }
    } else {  /* no, we don't know the node */
        message(mlv_normal, "node previously unknown, saving\n");
        res = write_known_node_file(fname, data);
        if(!res) {
            res = 0;
            goto quit;
        }
    }
    res = 1;
quit:
    if(dnode)
        textrecx_dispose_data(dnode);
    return res;
}


/* this function saves zero point info from non-zeropoint cert data */
static int
write_zeropoint_file(const char *fname, struct textrecx_data_item *data)
{
/*
          * node_id       (same)
          * point         0
          * timestamp     (from zp_certbody)
          * signed_by     -1
          * public        (from zp_certbody)
          * certbody      (zp_certbody)
          * signature     (zp_signature)
 */
    int res;
    FILE *f;

    struct textrecx_data_item *node_id;
    struct textrecx_data_item *zp_certbody;
    struct textrecx_data_item *zp_signature;
    long timestamp;
    const unsigned char *pubkey;

    node_id      = textrecx_data_by_name(data, "node_id");
    zp_certbody  = textrecx_data_by_name(data, "zp_certbody");
    zp_signature = textrecx_data_by_name(data, "zp_signature");

    if(!node_id || !zp_certbody || !zp_signature) {
        message(mlv_alert, "unexpectedly missing fields (int. error)\n");
        return 0;
    }

    f = fopen(fname, "w");
    if(!f) {
        message_perror(mlv_alert, "write_zeropoint_file", fname);
        return 0;
    }

    res = textrecx_serialize_data_item(node_id, fieldname_width, NULL,
                                       putchar_cb_fputc, f);
    if(res != textrecx_res_ok)
        message(mlv_alert, "problem writing node_id\n");

    res = textrecx_serialize_integer(0, "point", fieldname_width,
                                     putchar_cb_fputc, f);
    if(res != textrecx_res_ok)
        message(mlv_alert, "problem writing point number\n");

    timestamp = get_cert_timestamp((struct feda_cert_info *)zp_certbody->buf);
    res = textrecx_serialize_integer(timestamp, "timestamp", fieldname_width,
                                     putchar_cb_fputc, f);
    if(res != textrecx_res_ok)
       message(mlv_alert, "problem writing timestamp\n");

    res = textrecx_serialize_integer(-1, "signed_by", fieldname_width,
                                     putchar_cb_fputc, f);
    if(res != textrecx_res_ok)
        message(mlv_alert, "problem writing signed_by\n");

    pubkey = get_cert_pubkey(zp_certbody->buf);
    res = textrecx_serialize_blob(pubkey, public_key_size, "public",
                                  fieldname_width, putchar_cb_fputc, f);
    if(res != textrecx_res_ok)
        message(mlv_alert, "problem writing public key\n");

    res = textrecx_serialize_data_item(zp_certbody, fieldname_width,
                                       "certbody", putchar_cb_fputc, f);
    if(res != textrecx_res_ok)
        message(mlv_alert, "problem writing certbody\n");

    res = textrecx_serialize_data_item(zp_signature, fieldname_width,
                                       "signature", putchar_cb_fputc, f);
    if(res != textrecx_res_ok)
        message(mlv_alert, "problem writing signature\n");

    fclose(f);
    message(mlv_debug, "saved in %s\n", fname);
    return 1;
}

static int make_point_obsolete(const char *fname)
{
    int res;
    long long timestamp;
    struct textrecx_data_item *data = NULL;
    char *newname;

    res = read_textrecx_file(fname, schema_known_point, &data, 0);
    if(!res)
        return 0;
    res = textrecx_integer_by_name(data, "timestamp", &timestamp);
    if(!res) {
        message(mlv_alert, "missing timestamp in point file\n");
        return 0;
    }

    newname = make_old_point_fname(fname, timestamp);
    res = rename(fname, newname);
    if(res == -1) {
        message_perror(mlv_alert, "make_point_obsolete", fname);
        return 0;
    }
    message(mlv_debug, "renamed %s to %s\n", fname, newname);
    free(newname);
    return 1;
}

static int
check_save_point(const char *fname, struct textrecx_data_item *data)
{
    int res, errline;
    struct textrecx_data_item *dp = NULL;

    res = read_textrecx_file(fname, schema_known_point, &dp, &errline);
    if(res == kfiles_res_ok) { /* we have _some_ info on this point */
        long long known_timestamp, new_timestamp;
        const unsigned char *new_certbody;
        res = textrecx_integer_by_name(dp, "timestamp", &known_timestamp);
        if(!res) {
            message(mlv_alert, "no timestamp for known point (int. error)\n");
            return 0;
        }
        new_certbody = textrecx_blob_by_name(data, "certbody");
        if(!new_certbody) {
            message(mlv_alert, "missing certbody (int. error)\n");
            return 0;
        }
        new_timestamp =
            get_cert_timestamp((struct feda_cert_info *)new_certbody);

        /* now we can decide what to do */
        if(known_timestamp == new_timestamp) {
            message(mlv_info, "point cert's timestamp matches the known\n");
            return 1;
        } else
        if(known_timestamp < new_timestamp) { /* renew the key */
            message(mlv_debug, "known timestamp %lld, new has %lld\n",
                               known_timestamp, new_timestamp);
            message(mlv_normal, "the key is newer than the known, updating\n");
            res = make_point_obsolete(fname);
            if(!res)
                return 0;
            res = write_known_point_file(fname, data);
            return res;
        } else { /* known_timetamp > new_timestamp, simply reject */
            message(mlv_info, "new timestamp: %lld; known timestamp: %lld\n",
                              new_timestamp, known_timestamp);
            message(mlv_alert, "we have newer cert for the point, aborting\n");
            return 0;
        }
    } else { /* we didn't know this zero point */
        message(mlv_normal, "point key previously unknown, saving\n");
        res = write_known_point_file(fname, data);
        return res;
    }
}

static int
check_save_zero_point(const char *fname, struct textrecx_data_item *data)
{
    int res, errline;
    struct textrecx_data_item *dzp = NULL;

    res = read_textrecx_file(fname, schema_known_point, &dzp, &errline);
    if(res) { /* we have _some_ info on this zeropoint */
        long long known_timestamp, new_timestamp;
        const unsigned char *new_certbody;
        res = textrecx_integer_by_name(dzp, "timestamp", &known_timestamp);
        if(!res) {
            message(mlv_alert, "missing known zp timestamp (int. error)\n");
            return 0;
        }
        new_certbody = textrecx_blob_by_name(data, "zp_certbody");
        if(!new_certbody) {
            message(mlv_alert, "missing zp_certbody (int. error)\n");
            return 0;
        }
        new_timestamp =
            get_cert_timestamp((struct feda_cert_info *)new_certbody);

        /* now we can decide what to do */
        if(known_timestamp == new_timestamp) {
            /* just check they're really the same */
            const unsigned char *known_certbody =
                textrecx_blob_by_name(dzp, "certbody");
            if(!known_certbody) {
                message(mlv_alert, "missing certbody (int. error)\n");
                return 0;
            }
            if(0 != memcmp(known_certbody, new_certbody, feda_cert_size)) {
                message(mlv_alert,
                    "ZP key has the same timemark but differs, aborting\n");
                message(mlv_normal, "Remove %s file to force the update\n",
                                    fname);
                return 0;
            }
            /* check passed, no actions needed regarding ZeroPoint */
            message(mlv_debug, "zero point key matches the known one\n");
            return 1;
        } else
        if(known_timestamp < new_timestamp) {
            /* renew the ZeroPoint key */
            message(mlv_debug, "known ZP timestamp %lld, new has %lld\n",
                               known_timestamp, new_timestamp);
            message(mlv_normal, "zero point key is newer, updating\n");
            res = make_point_obsolete(fname);
            if(!res)
                return 0;
            /* check_all_points_for_obsolete_zp(); TO BE IMPLEMENTED! */
            res = write_zeropoint_file(fname, data);
            return res;
        } else { /* known_timetamp > new_timestamp */
            /* simply reject */
            message(mlv_info, "zero point's timestamp from the file: %lld\n",
                              new_timestamp);
            message(mlv_info, "zero point key on record has timestamp %lld\n",
                             known_timestamp);
            message(mlv_alert,
                "the file contains obsolete zero point cert, aborting.\n");
            return 0;
        }
    } else { /* we didn't know this zero point */
        message(mlv_normal, "zero point key previously unknown, saving\n");
        res = write_zeropoint_file(fname, data);
        return res;
    }
}


static int
write_point_kex_file(const char *fname, struct textrecx_data_item *data)
{
    static const char *fields[] =
        { "kex_public", "kex_cert", "kex_signature", "kex_timestamp", NULL };
    int res;
    struct textrecx_data_item *timestamp_rec;
    FILE *f;

    timestamp_rec = textrecx_data_by_name(data, "timestamp");
    if(!timestamp_rec) {
        message(mlv_alert, "missing timestamp field (int. error)\n");
        return 0;
    }

    f = fopen(fname, "w");
    if(!f) {
        message_perror(mlv_alert, "write_known_node_file", fname);
        return 0;
    }

    do_write_fields(f, data, fields);

    res = textrecx_serialize_data_item(timestamp_rec, fieldname_width,
                                       "ptk_timestamp", putchar_cb_fputc, f);
    if(res != textrecx_res_ok)
        message(mlv_alert, "problem writing ptk_timestamp\n");

    fclose(f);
    message(mlv_debug, "saved in %s\n", fname);
    return 1;
}


static int
check_save_point_kex(const char *fname, struct textrecx_data_item *data)
{
    int res, errline;
    struct textrecx_data_item *dpkex = NULL;

    res = read_textrecx_file(fname, schema_known_point_kex, &dpkex, &errline);
    if(res) {
        long long known_ts, new_ts;
        res = textrecx_integer_by_name(dpkex, "kex_timestamp", &known_ts);
        if(res) {  /* otherwise, just let it get overwritten */
            res = textrecx_integer_by_name(data, "kex_timestamp", &new_ts);
            if(!res) {
                message(mlv_normal, "no kex information, skipping\n");
                return 1;
            }
            if(known_ts == new_ts) { /* no need to do anything */
                message(mlv_normal, "we already know the kex key, skipping\n");
                return 1;
            }
            if(known_ts > new_ts) {
                message(mlv_alert, "we know newer kex key\n");
                return 0;
            }
            /* well, new_ts > known_ts, just let it get overwritten */
        }
    }

    message(mlv_normal, "saving new kex information for the point\n");
    return write_point_kex_file(fname, data);
}


static int do_add_nodecert(const char *wdir, const unsigned char *node_id,
                           struct kc_path_set *ps,
                           struct textrecx_data_item *data)
{
    int res;
    long long minpref;

    res = make_directory_path(ps->node_dir, 0);
    if(res == -1) {
        message_perror(mlv_alert, "make_directory_path", ps->node_dir);
        return 0;
    }

    /* first of all, do we know the node? write the info if no */
    res = check_save_known_node(ps->node_file, data);
    if(!res)
        return 0;

    /* do we have the minpref information in the cert? */
    res = textrecx_integer_by_name(data, "minpref", &minpref);
    if(res) {
        struct known_nodes_db *p;
        p = make_kndb(wdir, 0);
        kndb_set_minpref(p, node_id, minpref);
        dispose_kndb(p); 
    }

    return 1;
}

int add_known_nodecert(const char *wdir, struct textrecx_data_item *data)
{
    struct kc_path_set ps;
    const unsigned char *node_id;
    int res;

    node_id = textrecx_blob_by_name(data, "node_id");
    if(!node_id) {
        message(mlv_alert, "incomplete nodecert data (internal error)\n");
        return 0;
    }
    make_path_set(wdir, node_id, -1, &ps);
    res = do_add_nodecert(wdir, node_id, &ps, data);
    dispose_path_set(&ps);
    return res;
}

int add_known_cert(const char *wdir, struct textrecx_data_item *data,
                                     int have_kex_info)
{
    int res, res2;
    struct kc_path_set ps;

    const unsigned char *node_id;
    long long point, signed_by;

    node_id = textrecx_blob_by_name(data, "node_id");
    res = textrecx_integer_by_name(data, "point", &point);
    res2 = textrecx_integer_by_name(data, "signed_by", &signed_by);

    if(!node_id || !res || !res2) {
        message(mlv_alert, "incomplete cert data (internal error)\n");
        return 0;
    }

    make_path_set(wdir, node_id, point, &ps);

    res = do_add_nodecert(wdir, node_id, &ps, data);
    if(!res)
        goto quit;

    /* does the new cert contain zero point key as well? */
    if(point != 0 && signed_by == 0) {   /* yes it does */
        res = check_save_zero_point(ps.zero_point_file, data);
        if(!res) {
            res = 0;
            goto quit;
        }
    }

    /* what about the point itself? */
    res = check_save_point(ps.point_file, data);
    if(!res) {
        res = 0;
        goto quit;
    }

    /* and what about the key exchange public key? */
    if(have_kex_info) {
        res = check_save_point_kex(ps.point_kex_file, data);
        if(!res) {
            res = 0;
            goto quit;
        }
    }

    res = 1;
quit:
    dispose_path_set(&ps);
    return res;
}

int save_known_point(const char *wdir, const struct feda_cert_info *cert,
                     const unsigned char *signature)
{
    int res;
    struct kc_path_set ps;
    FILE *f;

    make_path_set(wdir, cert->node_id, cert->point_id, &ps);

    res = make_directory_path(ps.node_dir, 0);
    if(res == -1) {
        message_perror(mlv_alert, "make_directory_path", ps.node_dir);
        return 0;
    }

/*
          * node_id
          * point
          * timestamp
          * signed_by
          * public
          * certbody
          * signature
 */
    f = fopen(ps.point_file, "w");
    if(!f) {
        message_perror(mlv_alert, "write_zeropoint_file", ps.point_file);
        res = 0;
        goto quit;
    }
    textrecx_serialize_blob(cert->node_id, node_id_size, "node_id",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_integer(cert->point_id, "point",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_integer(get_cert_timestamp(cert), "timestamp",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_integer(cert->signer_id, "signed_by",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(cert->key, public_key_size, "public",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob((const unsigned char *)&cert,
                               feda_cert_size, "certbody",
                               fieldname_width, putchar_cb_fputc, f);
    textrecx_serialize_blob(signature, signature_size, "signature",
                               fieldname_width, putchar_cb_fputc, f);
    fclose(f);
    res = 1;
quit:
    dispose_path_set(&ps);
    return res;
}
