#include <string.h>

#include "_version.h"

#include "fk_help.h"

static const char versiontext[] =
    "fedakeys program: key management for FEDAnet users and node admins\n"
    "vers. " FEDA_VERSION " (compiled " __DATE__ ")\n"
    "Copyright (c) Andrey Vikt. Stolyarov, 2024, 2025\n"
    ;

static const char helptext[] =
    "\n"
    "Usage: fedakeys <global-options> command <command-args>\n"
    "Options are:\n"
    "    -c <directory>      use this dir instead of $HOME/.fedanet/keys\n"
    "    -o <output_file>    override the default output file name\n"
    "    -t <timemark>       use this as the timemark instead of the\n"
    "                        current time; remember it's in MINUTES since\n"
    "                        the Epoch, that is, unixtime/60; '0x' prefix\n"
    "                        means hex, leading '0' means octal\n"
    "    -p                  don't attempt to take yespower hashes\n"
    "                        (useful for low-level machines); existing\n"
    "                        hashes are assumed valid; use with EXTREME\n"
    "                        cautions, first check all the stuff on a\n"
    "                        faster machine; warnings are issued\n"
    "    -v                  be verbose (more flags for more messages)\n"
    "    -q                  be quiet\n"
    "    -h                  print help text and quit; if a command is\n"
    "                        given, the help on that command is printed;\n"
    "                        other options are ignored in this case.\n"
    "\n"
    "Supported commands are briefly explained below; please note that\n"
    "command names starting with ``m'' are intended for a node master\n"
    "and are only useful in presence of your node master secret key,\n"
    "which is not (and MUST NOT be) a very common situation.  Keep your\n"
    "node master secret key away from computers connected to networks!\n"
    "Use ``fedakeys -h <command>'' for command-specific help\n"
    "\n"
    "    status              inspect what's deployed here\n"
    "\n"
    "    mverify             verify a file generated by feda-ng\n"
    "    mdeploy             deploy nodemaster workplace\n"
    "    mzero               make zero (0th) point key\n"
    "    mpoint              make a key for the given point number\n"
    "\n"
    "    pverify             verify a point key file\n"
    "    deploy              deploy point using a point key file\n"
    "    zcrpoint            make a key for another point using Zero Point\n"
    "    zsign               sign an existing public key as belonging to\n"
    "                        the given point using Zero Point\n"
    "    key2pub             create a public info file (.pub) according\n"
    "                        to the given point key file (.key)\n"
    "    keypair             make a pair of secret and public keys\n"
    "    kexgen              (re)generate the key exchange keypair\n"
    "    nodecert            compose a file with the node public info\n"
    "    nverify             verify a node file (created by nodecert)\n"
    "    nodeimport          store the public node information for a node,\n"
    "                        taken from the given file (made by nodecert)\n"
    "    export              compose a file with public keys and certs\n"
    "                        for the locally configured point\n"
    "    fverify             verify a foreign point public keys file\n"
    "    import              store the key/cert information for a foreign\n"
    "                        point, taken from the given file\n"
    "    sign                sign a given file with the point's signature\n"
    "    signverify          check the signature for the file\n"
    "    lock                enclose the given file in a ciphered container\n"
    "    anonlock            use a one-time keypair to enclose the given\n"
    "                        file in a ciphered container\n"
    "    unlock              check and open a container\n"
    "\n"
    "    base32              convert a node ID to base32 domain name token;\n"
    "                        only the first 20 chars of the argument are\n"
    "                        analysed, so any node/point/etc filename is\n"
    "                        accepted\n"
    "    blake2sum           computes blake2b hash of a given file\n"
    ;

static const char status_helptext[] =
    "fedakeys <options> status\n"
    "    Display some general information on what's deployed here.  You\n"
    "    might want to use -v (or -v -v) to get more information.\n"
    ;

static const char mverify_helptext[] =
    "fedakeys <options> mverify <master_key_file>\n"
    "    Verify a file generated by the feda-ng program for integrity.\n"
    ;

static const char mdeploy_helptext[] =
    "fedakeys <options> mdeploy <master_key_file> [<min_pref_len>]\n"
    "    Deploy nodemaster workplace using the file generated by the\n"
    "    feda-ng program, setting the given minimal prefix length for\n"
    "    foreign node keys to be trusted.\n"
    "    If the min_pref_len arg is omitted, the prefix length of the\n"
    "    key being deployed is used as the minimum (may be changed later)\n"
    ;

static const char mzero_helptext[] =
    "fedakeys <options> mzero\n"
    "    Generate a key pair and create a certificate for the current\n"
    "    node's Point Zero (0th point), which may be used to sign other\n"
    "    points' certificates and other things on behalf of the node.\n"
    "    Requires deployed nodemaster workplace.\n"
    "    All the data is saved in the file named NODEID_p0.key, where\n"
    "    NODEID is the 20-hex-digits node ID.\n"
    ;

static const char mpoint_helptext[] =
    "fedakeys <options> mpoint <point>\n"
    "    Generate a key pair and create a certificate for the given\n"
    "    point using the master key to sign it.  <point> must be a decimal\n"
    "    number from 1 to 253.  Deployed nodemaster workplace is required.\n"
    "    All the data is saved in the file named NODEID_pNUM.key, where\n"
    "    NODEID is the 20-hex-digits node ID and NUM is the point number.\n"
    "    Generally this is discouraged practice; Zero Point workplace\n"
    "    and keys should be used to sign all point certificates, so the\n"
    "    master key is accessed less often.  However, replacing your Zero\n"
    "    Point key with a newer one invalidates all certs signed with\n"
    "    the old key, so sometimes it might make sense to create some keys\n"
    "    for your points using the master key while you're at it anyway.\n"
    ;

static const char pverify_helptext[] =
    "fedakeys <options> pverify <point_key_file>\n"
    "    Verify a key file created by ``mzero'', or ``mpoint'',\n"
    "    or ``zcrpoint'', or ``zsign'' command.\n"
    ;

static const char deploy_helptext[] =
    "fedakeys <options> deploy <point_key_file> [<separate_secret_file>]\n"
    "    Deploy a point's key and configuration using a key file created\n"
    "    by either ``mzero'', or ``mpoint'', or ``zcrpoint'', or ``zsign''\n"
    "    command.\n"
    ;

static const char zcrpoint_helptext[] =
    "fedakeys <options> zcrpoint <point>\n"
    "    Generate a key pair and create a certificate for the given\n"
    "    point using the Zero Point key to sign it.  <point> must be a\n"
    "    decimal number from 1 to 253.  Deployed Zero Point is required.\n"
    "    All the data is saved in the file named NODEID_pNUM.key, where\n"
    "    NODEID is the 20-hex-digits node ID and NUM is the point number.\n"
    ;

static const char zsign_helptext[] =
    "fedakeys <options> zsign <file> <point>\n"
    "    Take the given public key and sign it with the Zero Point key;\n"
    "    append all the necessary fields to the given file so it becomes\n"
    "    suitable for point deployment.\n"
    "    The file must initially contain the \"public\" field and SHOULD NOT\n"
    "    contain anything else.  The command modifies the file so you might\n"
    "    want to make a copy of it, just in case.\n"
    ;

static const char key2pub_helptext[] =
    "fedakeys <options> key2pub <key_file>\n"
    "    Create a public cert file, which corresponds to the given point\n"
    "    key file (one made by ``mpoint'' or ``zcrpoint'' commands).\n"
    "    Effectively, this command simply copies to the new file all\n"
    "    fields from the old file except the \"secret\" field.\n"
    "    If the output file name is not set explicitly, it is composed\n"
    "    by replacing the suffix \".key\" with the suffix \".pub\", in\n"
    "    case the original file has that suffix, otherwise the suffix\n"
    "    \".pub\" is simply appended to the original file name.\n"
    ;

static const char keypair_helptext[] =
    "fedakeys <options> keypair <basename>\n"
    "    Generate a keypair (suitable, e.g., for the zsign command).\n"
    "    Two files, <basename>.key and <basename>.pub, will be created.\n"
    "    No error checking is performed, so be careful not to write over\n"
    "    any files you still need.\n"
    "    In case the output file name is specified with -o, the <basename>\n"
    "    is ignored (and may be omitted), and both secret and public keys\n"
    "    are written to that file.\n"
    ;

static const char kexgen_helptext[] =
    "fedakeys <options> kexgen\n"
    "    (Re)generate the keypair intended for key exchange.\n"
    "    The key pair is always saved in the $dir/kex.key file, where $dir\n"
    "    is either specified with -c, or (by default) is the\n"
    "    $HOME/.fedanet/keys directory.\n"
    "    If the file already exists, it is renamed to kex_IDIDIDIDID.key,\n"
    "    where IDIDIDIDID is the first 10 hex digits of the public key.\n"
    ;

static const char nodecert_helptext[] =
    "fedakeys <options> nodecert\n"
    "    Compose a file with the node's public key, hash and the hash\n"
    "    signature, using the information of the locally deployed point.\n"
    "    The file is appropriate to be published openly, and/or sent\n"
    "    directly to people you're going to communicate with.  If there's\n"
    "    no -o option, the file is named NODEID.pub, where NODEID is your\n"
    "    node ID.\n"
    ;

static const char nverify_helptext[] =
    "fedakeys <options> nverify <filename> [<min_rank>]\n"
    "    Check a file, presumably generated somewhere else by the nodecert\n"
    "    command.  Use the additional min_rank parameter to override\n"
    "    the configured minimum for this single operation.\n"
    ;

static const char nodeimport_helptext[] =
    "fedakeys <options> nodeimport <filename> [<min_rank>]\n"
    "    Check a file, presumably generated somewhere else by the nodecert\n"
    "    command, and if the hash and the signature are okay AND the node\n"
    "    hash rank is above or equal to the configured minimum, store all\n"
    "    the information for future use.  Use the additional min_rank\n"
    "    parameter to override the configured minimum for this single\n"
    "    operation.\n"
    ;

static const char export_helptext[] =
    "fedakeys <options> export\n"
    "    Compose a file with public keys and appropriate certs for the\n"
    "    locally deployed point.  The file is appropriate to be published\n"
    "    openly, and/or sent directly to people you're going to communicate\n"
    "    with.  If there's no -o option, the file is named NODEID_pNUM.pub,\n"
    "    where NODEID is your node ID and NUM is your point number.\n"
    "    If you don't yet have the keypair for message encryption (as\n"
    "    distinct from signatures), use the kexgen command first.\n"
    ;

static const char fverify_helptext[] =
    "fedakeys <options> fverify <filename> [<min_rank>]\n"
    "    Check a file, presumably generated somewhere else by the export\n"
    "    command.  Use the additional min_rank parameter to override\n"
    "    the configured minimum for this single operation.\n"
    ;

static const char import_helptext[] =
    "fedakeys <options> import <filename> [<min_rank>]\n"
    "    Check a file, presumably generated somewhere else by the export\n"
    "    command, and if all signatures are okay AND the node hash rank is\n"
    "    above or equal to the configured minimum, store all the information\n"
    "    for future use.  Use the additional min_rank parameter to override\n"
    "    the configured minimum for this single operation.\n"
    ;

static const char sign_helptext[] =
    "fedakeys <options> sign [<filename> [<signature_file_name>]]\n"
    "    Make a signature for the given file.  Use ``-'' instead of the name\n"
    "    (or omit all names altogether) to sign the data received from\n"
    "    stdin. If the signature file name is omitted, the name is derived\n"
    "    by adding the suffix ``.sign'' to the original file name; for\n"
    "    data received from stdout, ``__stdout__.sig'' is used by default.\n"
    ;

static const char signverify_helptext[] =
    "fedakeys <options> signverify [<filename> [<signature_file_name>]]\n"
    "    Check the signature made with the sign command.  Use ``-'' instead\n"
    "    of the file name to check the data received from stdin.  If the\n"
    "    signature file name is omitted, the name is derived by adding the\n"
    "    suffix ``.sign'' to the original file name; for data received from\n"
    "    stdout, ``__stdout__.sig'' is used by default.\n"
    ;

static const char lock_helptext[] =
    "fedakeys <options> lock\n"
    "    NOT IMPLEMENTED YET\n"
    ;

static const char anonlock_helptext[] =
    "fedakeys <options> anonlock\n"
    "    NOT IMPLEMENTED YET\n"
    ;

static const char unlock_helptext[] =
    "fedakeys <options> unlock\n"
    "    NOT IMPLEMENTED YET\n"
    ;

static const char base32_helptext[] =
    "fedakeys <options> base32 <nodeID or filename>\n"
    "    Convert a node ID to base32 domain name token.  Only the first\n"
    "    20 chars of the argument are analysed, so any node/point/etc\n"
    "    filename is accepted, but the file is never accessed.\n"
    ;

static const char blake2sum_helptext[] =
    "fedakeys <options> blake2sum [<filename> [<hash_size>]]\n"
    "    Takes blake2b hash of the given size (1..64 bytes, default is 32)\n"
    "    of a file or the data taken from stdin.  Use ``-'' in place of the\n"
    "    filename to take the data from stdin yet specifying a non-default\n"
    "    hash size.  Please note hash sizes below 32 are ``discouraged''\n"
    "    according to the official monocypher documentation.\n"
    ;

#define HELPFOR(name)                     \
    if(0 == strcmp(cmd, #name)) {         \
        fputs(name##_helptext, stream);   \
        return;                           \
    }

void help(FILE *stream, const char *cmd)
{
    if(!cmd) {
        fputs(versiontext, stream);
        fputs(helptext, stream);
        return;
    }
    if(!*cmd) {
        fputs(helptext, stream);
        return;
    }
    HELPFOR(status)
    HELPFOR(mverify)
    HELPFOR(mdeploy)
    HELPFOR(mzero)
    HELPFOR(mpoint)
    HELPFOR(pverify)
    HELPFOR(deploy)
    HELPFOR(zcrpoint)
    HELPFOR(zsign)
    HELPFOR(key2pub)
    HELPFOR(keypair)
    HELPFOR(kexgen)
    HELPFOR(nodecert)
    HELPFOR(nverify)
    HELPFOR(nodeimport)
    HELPFOR(export)
    HELPFOR(fverify)
    HELPFOR(import)
    HELPFOR(sign)
    HELPFOR(signverify)
    HELPFOR(lock)
    HELPFOR(anonlock)
    HELPFOR(unlock)
    HELPFOR(base32)
    HELPFOR(blake2sum)
    fprintf(stream, "Command %s unknown, try -h\n", cmd);
}
