#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "message.h"
#include "textrec.h"
#include "txtr_out.h"
#include "hexdata.h"
#include "keyutils.h"
#include "fk_data.h"





static void message_parser_abort(const char *fname, int line, int code)
{
    message(mlv_normal, "%s:%d: parse error [%s]\n", fname, line,
                        textrec_error_message(code));
}

static int run_parser_on_file(struct textrec_parser *parser, const char *name,
                              int suppress_messages)
{
    FILE *f;
    int c, res, ok;

    f = fopen(name, "r");
    if(!f) {
        if(!suppress_messages)
            message_perror(mlv_normal, NULL, name);
        return 0;
    }

    while((c = fgetc(f)) != EOF) {
        res = textrec_feedchar(parser, c);
        if(res != textrec_res_ok)
            break;
    }
    if(c == EOF)
        res = textrec_feedchar(parser, EOF);
    fclose(f);

    ok = res == textrec_res_ok || res == textrec_res_eof;
    if(!ok && !suppress_messages)
        message_parser_abort(name, parser->line, res);

    return ok;
}

/* ------- read_master_file --------------------------------------- */

static int masterfile_type_cb(const char *field_name, void *user_data)
{
    if(0 == strcmp(field_name, "preflen"))
        return ftype_integer;

    if( 0 == strcmp(field_name, "id") ||
        0 == strcmp(field_name, "secret") ||
        0 == strcmp(field_name, "public") ||
        0 == strcmp(field_name, "yeshash")
    )
        return ftype_blob;

    return ftype_unknown;
}

#define BLOB_FIELD_IF(Name, Target)                            \
    if(0 == strcmp(fnm, Name)) {                               \
        lencp = len > sizeof(Target) ? sizeof(Target) : len;   \
        memcpy(Target, buf, lencp);                            \
        return 0;                                              \
    } else


static int
masterfile_blob_cb(const char *fnm, const uchar *buf, int len, void *ud)
{
    struct master_file *mfr = ud;
    int lencp;

    BLOB_FIELD_IF("id", mfr->node_id)
    BLOB_FIELD_IF("secret", mfr->secret)
    BLOB_FIELD_IF("public", mfr->public_key)
    BLOB_FIELD_IF("yeshash", mfr->yeshash)
    {
        message(mlv_normal, "unknown field [%s]\n", fnm);
        return 1;
    }
}

static int masterfile_integer_cb(const char *fnm, long long num, void *ud)
{
    struct master_file *mfr = ud;

    if(0 == strcmp(fnm, "preflen")) {
        mfr->preflen = num;
        return 0;
    } else {
        message(mlv_normal, "unknown field [%s]\n", fnm);
        return 1;
    }
}

int read_master_file(const char *name, struct master_file *data)
{
    struct textrec_parser parser;
    int res;

    textrec_init(&parser);
    textrec_set_type_cb(&parser, masterfile_type_cb);
    textrec_set_blob_cb(&parser, masterfile_blob_cb);
    textrec_set_integer_cb(&parser, masterfile_integer_cb);
    textrec_set_user_data(&parser, data);
    memset(data, 0, sizeof(*data));

    res = run_parser_on_file(&parser, name, 0);

    textrec_cleanup(&parser);
    return res;
}


/* ------- read_config_file --------------------------------------- */

#if 0  /* moved to pointcfg.[ch] to get clear of issuing messages */

static int configfile_type_cb(const char *field_name, void *user_data)
{
    if( 0 == strcmp(field_name, "point") ||
        0 == strcmp(field_name, "signed_by") ||
        0 == strcmp(field_name, "rank") ||
        0 == strcmp(field_name, "minpref")
    )
        return ftype_integer;

    if( 0 == strcmp(field_name, "node_id") ||
        0 == strcmp(field_name, "master_pub") ||
        0 == strcmp(field_name, "master_hash") ||
        0 == strcmp(field_name, "master_hsign") ||
        0 == strcmp(field_name, "zp_certbody") ||
        0 == strcmp(field_name, "zp_signature") ||
        0 == strcmp(field_name, "certbody") ||
        0 == strcmp(field_name, "signature") ||
        0 == strcmp(field_name, "public") ||
        0 == strcmp(field_name, "msg_public") ||
        0 == strcmp(field_name, "msgpub_sign")
    )
        return ftype_blob;

    return ftype_ignored;
}

static int
configfile_blob_cb(const char *fnm, const uchar *buf, int len, void *ud)
{
    struct point_config_file *cfr = ud;
    int lencp;

    BLOB_FIELD_IF("node_id", cfr->node_id)
    BLOB_FIELD_IF("master_pub", cfr->master_pub)
    BLOB_FIELD_IF("master_hash", cfr->master_hash)
    BLOB_FIELD_IF("master_hsign", cfr->master_hsign)
    BLOB_FIELD_IF("zp_certbody", cfr->zp_certbody)
    BLOB_FIELD_IF("zp_signature", cfr->zp_signature)
    BLOB_FIELD_IF("certbody", cfr->certbody)
    BLOB_FIELD_IF("signature", cfr->signature)
    BLOB_FIELD_IF("public", cfr->public_key)
    {
        message(mlv_normal, "unknown field [%s]\n", fnm);
        return 1;
    }
}

static int configfile_integer_cb(const char *fnm, long long num, void *ud)
{
    struct point_config_file *cfr = ud;

    if(0 == strcmp(fnm, "point")) {
        cfr->point = num;
        return 0;
    } else
    if(0 == strcmp(fnm, "signed_by")) {
        cfr->signed_by = num;
        return 0;
    } else
    if(0 == strcmp(fnm, "rank")) {
        cfr->rank = num;
        return 0;
    } else
    if(0 == strcmp(fnm, "minpref")) {
        cfr->minpref = num;
        return 0;
    } else
    {
        message(mlv_normal, "unknown field [%s]\n", fnm);
        return 1;
    }
}

int read_config_file(const char *name, struct point_config_file *data)
{
    struct textrec_parser parser;
    int res;

    textrec_init(&parser);

    textrec_set_type_cb(&parser, configfile_type_cb);
    textrec_set_blob_cb(&parser, configfile_blob_cb);
    textrec_set_integer_cb(&parser, configfile_integer_cb);

    textrec_set_user_data(&parser, data);
    memset(data, 0, sizeof(*data));

    res = run_parser_on_file(&parser, name, 0);

    textrec_cleanup(&parser);
    return res;
}

int write_config_file(const char *name, struct point_config_file *data)
{
    FILE *f;
    f = fopen(name, "w");
    if(!f) {
        message_perror(mlv_alert, "FATAL", name);
        return 0;
    }
    put_hex_field(f, "node_id", data->node_id, sizeof(data->node_id));
    put_int_field(f, "point", data->point);
    put_int_field(f, "rank", data->rank);
    put_int_field(f, "minpref", data->minpref);
    put_hex_field(f, "master_pub", data->master_pub, sizeof(data->master_pub));
    put_hex_field(f, "master_hash", data->master_hash,
                                    sizeof(data->master_hash));
    put_hex_field(f, "master_hsign", data->master_hsign,
                                    sizeof(data->master_hsign));
    if(data->point != -1) {
        put_hex_field(f, "public", data->public_key, sizeof(data->public_key));
        put_int_field(f, "signed_by", data->signed_by);
        put_hex_field(f, "certbody", data->certbody, sizeof(data->certbody));
        put_hex_field(f, "signature", data->signature,
                                      sizeof(data->signature));
        if(!all_zeroes(data->zp_certbody, sizeof(data->zp_certbody))) {
            put_hex_field(f, "zp_certbody", data->zp_certbody,
                                            sizeof(data->zp_certbody));
            put_hex_field(f, "zp_signature", data->zp_signature,
                                             sizeof(data->zp_signature));
        }
    }
    fclose(f);
    return 1;
}

#endif

/* ------- read_single_blob --------------------------------------- */

#if 0     /* this version should never become needed again */

    /* the version implemented via textrecx is in kfiles.h */

struct rsb_userdata {
    const char *blobname;
    unsigned char *buf;
    int buflen, datalen;
    int gotit;
};

static int rsb_type_cb(const char *field_name, void *ud)
{
    struct rsb_userdata *rsb_ud = ud;
    if(0 == strcmp(field_name, rsb_ud->blobname))
        return ftype_blob;
    return ftype_ignored;
}

static int rsb_blob_cb(const char *fnm, const uchar *buf, int len, void *ud)
{
        /* it's the only field to read, no need to check the fnm */
    struct rsb_userdata *rsb_ud = ud;
    rsb_ud->datalen = len > rsb_ud->buflen ? rsb_ud->buflen : len;
    memcpy(rsb_ud->buf, buf, rsb_ud->datalen);
    rsb_ud->gotit = 1;
    return 0;
}

int read_single_blob(const char *filename, const char *blobname,
                     unsigned char *buf, int buflen)
{
    struct textrec_parser parser;
    struct rsb_userdata ud;

    ud.blobname = blobname;
    ud.buf = buf;
    ud.buflen = buflen;
    ud.datalen = -1;
    ud.gotit = 0;

    textrec_init(&parser);
    textrec_set_type_cb(&parser, rsb_type_cb);
    textrec_set_blob_cb(&parser, rsb_blob_cb);
    textrec_set_user_data(&parser, &ud);

    run_parser_on_file(&parser, filename, 0);

    textrec_cleanup(&parser);

    return ud.gotit == 1;
}

#endif


/* ------- read_point_file --------------------------------------- */


static int pointfile_type_cb(const char *field_name, void *user_data)
{
    if( 0 == strcmp(field_name, "rank") ||
        0 == strcmp(field_name, "minpref") ||
        0 == strcmp(field_name, "point") ||
        0 == strcmp(field_name, "signed_by") ||
        0 == strcmp(field_name, "timestamp")
    )
        return ftype_integer;

    if( 0 == strcmp(field_name, "node_id") ||
        0 == strcmp(field_name, "secret") ||
        0 == strcmp(field_name, "public") ||
        0 == strcmp(field_name, "master_pub") ||
        0 == strcmp(field_name, "master_hash") ||
        0 == strcmp(field_name, "master_hsign") ||
        0 == strcmp(field_name, "certbody") ||
        0 == strcmp(field_name, "signature") ||
        0 == strcmp(field_name, "zp_pub") ||
        0 == strcmp(field_name, "zp_certbody") ||
        0 == strcmp(field_name, "zp_signature")
    )
        return ftype_blob;

    return ftype_unknown;
}


static int
pointfile_blob_cb(const char *fnm, const uchar *buf, int len, void *ud)
{
    struct point_file *pfr = ud;
    int lencp;

    BLOB_FIELD_IF("node_id", pfr->node_id)
    BLOB_FIELD_IF("secret", pfr->secret)
    BLOB_FIELD_IF("public", pfr->public_key)
    BLOB_FIELD_IF("master_pub", pfr->master_pub)
    BLOB_FIELD_IF("master_hash", pfr->master_hash)
    BLOB_FIELD_IF("master_hsign", pfr->master_hsign)
    BLOB_FIELD_IF("certbody", pfr->certbody)
    BLOB_FIELD_IF("signature", pfr->signature)
    BLOB_FIELD_IF("zp_pub", pfr->zp_pub)
    BLOB_FIELD_IF("zp_certbody", pfr->zp_certbody)
    BLOB_FIELD_IF("zp_signature", pfr->zp_signature)
    {
        message(mlv_normal, "unknown field [%s]\n", fnm);
        return 1;
    }
}

static int pointfile_integer_cb(const char *fnm, long long num, void *ud)
{
    struct point_file *mfr = ud;

    if(0 == strcmp(fnm, "rank")) {
        mfr->rank = num;
        return 0;
    } else
    if(0 == strcmp(fnm, "minpref")) {
        mfr->minpref = num;
        return 0;
    } else
    if(0 == strcmp(fnm, "point")) {
        mfr->point = num;
        return 0;
    } else
    if(0 == strcmp(fnm, "signed_by")) {
        mfr->signed_by = num;
        return 0;
    } else
    if(0 == strcmp(fnm, "timestamp")) {
        mfr->timestamp = num;
        return 0;
    } else
    {
        message(mlv_normal, "unknown field [%s]\n", fnm);
        return 1;
    }
}

int read_point_file(const char *name, struct point_file *data)
{
    struct textrec_parser parser;
    int res;

    textrec_init(&parser);

    textrec_set_type_cb(&parser, pointfile_type_cb);
    textrec_set_blob_cb(&parser, pointfile_blob_cb);
    textrec_set_integer_cb(&parser, pointfile_integer_cb);

    textrec_set_user_data(&parser, data);
    memset(data, 0, sizeof(*data));

    res = run_parser_on_file(&parser, name, 0);

    textrec_cleanup(&parser);
    return res;
}



/* ------- read_kex_keypair_file --------------------------------------- */





static int kexkeypairfile_type_cb(const char *field_name, void *user_data)
{
    if( 0 == strcmp(field_name, "created") ||
        0 == strcmp(field_name, "replaced")
    )
        return ftype_integer;

    if( 0 == strcmp(field_name, "kex_secret") ||
        0 == strcmp(field_name, "kex_public")
    )
        return ftype_blob;

    return ftype_unknown;
}

static int
kexkeypairfile_blob_cb(const char *fnm, const uchar *buf, int len, void *ud)
{
    struct kex_keypair_file *kxk = ud;
    int lencp;

    BLOB_FIELD_IF("kex_secret", kxk->kex_secret)
    BLOB_FIELD_IF("kex_public", kxk->kex_public)
    {
        message(mlv_normal, "unknown field [%s]\n", fnm);
        return 1;
    }
}

static int kexkeypairfile_integer_cb(const char *fnm, long long num, void *ud)
{
    struct kex_keypair_file *kxk = ud;

    if(0 == strcmp(fnm, "created")) {
        kxk->created = num;
        return 0;
    } else
    if(0 == strcmp(fnm, "replaced")) {
        kxk->replaced = num;
        return 0;
    } else {
        message(mlv_normal, "unknown field [%s]\n", fnm);
        return 1;
    }
}

int read_kex_keypair_file(const char *name, struct kex_keypair_file *data,
                          int suppress_messages)
{
    struct textrec_parser parser;
    int res;

    textrec_init(&parser);

    textrec_set_type_cb(&parser, kexkeypairfile_type_cb);
    textrec_set_blob_cb(&parser, kexkeypairfile_blob_cb);
    textrec_set_integer_cb(&parser, kexkeypairfile_integer_cb);

    textrec_set_user_data(&parser, data);
    memset(data, 0, sizeof(*data));
    data->created = -1;
    data->replaced = -1;

    res = run_parser_on_file(&parser, name, suppress_messages);

    textrec_cleanup(&parser);
    return res;
}

/* ------- read_pointcert_file --------------------------------------- */


static int pointcertfile_type_cb(const char *field_name, void *user_data)
{
    if( 0 == strcmp(field_name, "descr") ||
        0 == strcmp(field_name, "remark") ||
        0 == strcmp(field_name, "minpref")
    )
        return ftype_ignored;
            /* descriptive text is allowed to be there as the string
               parameter(s) ``descr'' and ``remark'', but the reader
               ignores them; we also ignore the minpref parameter of the
               remote node -- generally we don't care
             */

    if( 0 == strcmp(field_name, "point") ||
        0 == strcmp(field_name, "signed_by") ||
        0 == strcmp(field_name, "rank") ||
        0 == strcmp(field_name, "kex_timestamp")
    )
        return ftype_integer;

    if( 0 == strcmp(field_name, "node_id") ||
        0 == strcmp(field_name, "public") ||
        0 == strcmp(field_name, "master_pub") ||
        0 == strcmp(field_name, "master_hash") ||
        0 == strcmp(field_name, "master_hsign") ||
        0 == strcmp(field_name, "zp_certbody") ||
        0 == strcmp(field_name, "zp_signature") ||
        0 == strcmp(field_name, "certbody") ||
        0 == strcmp(field_name, "signature") ||
        0 == strcmp(field_name, "kex_public") ||
        0 == strcmp(field_name, "kex_cert") ||
        0 == strcmp(field_name, "kex_signature")
    )
        return ftype_blob;

    return ftype_unknown;
}


static int
pointcertfile_blob_cb(const char *fnm, const uchar *buf, int len, void *ud)
{
    struct point_cert_file *pfr = ud;
    int lencp;

    BLOB_FIELD_IF("node_id", pfr->node_id)
    BLOB_FIELD_IF("public", pfr->public_key)
    BLOB_FIELD_IF("master_pub", pfr->master_pub)
    BLOB_FIELD_IF("master_hash", pfr->master_hash)
    BLOB_FIELD_IF("master_hsign", pfr->master_hsign)
    BLOB_FIELD_IF("zp_certbody", pfr->zp_certbody)
    BLOB_FIELD_IF("zp_signature", pfr->zp_signature)
    BLOB_FIELD_IF("certbody", pfr->certbody)
    BLOB_FIELD_IF("signature", pfr->signature)
    BLOB_FIELD_IF("kex_public", pfr->kex_public)
    BLOB_FIELD_IF("kex_cert", pfr->kex_cert)
    BLOB_FIELD_IF("kex_signature", pfr->kex_signature)
    {
        message(mlv_normal, "unknown field [%s]\n", fnm);
        return 1;
    }
}

static int pointcertfile_integer_cb(const char *fnm, long long num, void *ud)
{
    struct point_cert_file *pfr = ud;

    if(0 == strcmp(fnm, "rank")) {
        pfr->rank = num;
        return 0;
    } else
    if(0 == strcmp(fnm, "point")) {
        pfr->point = num;
        return 0;
    } else
    if(0 == strcmp(fnm, "signed_by")) {
        pfr->signed_by = num;
        return 0;
    } else
    if(0 == strcmp(fnm, "kex_timestamp")) {
        pfr->kex_timestamp = num;
        return 0;
    } else
    {
        message(mlv_normal, "unknown field [%s]\n", fnm);
        return 1;
    }
}

int read_from_pointcert_file(const char *name, struct point_cert_file *data)
{
    struct textrec_parser parser;
    int res;

    textrec_init(&parser);

    textrec_set_type_cb(&parser, pointcertfile_type_cb);
    textrec_set_blob_cb(&parser, pointcertfile_blob_cb);
    textrec_set_integer_cb(&parser, pointcertfile_integer_cb);

    textrec_set_user_data(&parser, data);

    res = run_parser_on_file(&parser, name, 0);

    textrec_cleanup(&parser);
    return res;
}

int read_pointcert_file(const char *name, struct point_cert_file *data)
{
    memset(data, 0, sizeof(*data));
    return read_from_pointcert_file(name, data);
}

