#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <arpa/inet.h>  /* for ntohs */


#include <sue/sue_base.h>

#include "servlog.h"
#include "hexdata.h"
#include "addrport.h"
#include "keyutils.h"
#include "fsrv_cfg.h"
#include "fsrv_cfd.h"
#include "fsrv_rx.h"
#include "fsrv_dst.h"
#include "fsrv_tun.h"
#include "fsrv_pxy.h"
#include "fsrv_sig.h"
#include "fsrv_con.h"
#include "_version.h"




static void do_help()
{
    fputs("\n"
        "fedaserv: the server daemon for FEDAnet\n"
        "vers. " FEDA_VERSION " (compiled " __DATE__ ")\n"
        "Copyright (c) Andrey Vikt. Stolyarov, 2025, 2026\n"
        "\n"
        "Launch: \"fedaserv <options>\"  where options are as follows:\n"
        "\n"
        "    -c <config_file>   get the configuration from this file\n"
        "                       instead of $HOME/.fedanet/serv.conf\n"
        "    -e                 send the log messages to stderr as well\n"
        "    -E                 send the log messages to stderr _only_,\n"
        "                       ignoring the logging configuration\n"
        "                    BOTH -e and -E assume stderr is private enough\n"
        "    -q                 (with -e or -E) be quiet, only log errors\n"
        "    -v                 (with -e or -E) be verbose (mention twice\n"
        "                       to enable debugging messages)\n"
        "    -f                 don't fork, stay foreground (-e is implied,\n"
        "                       but you might want to use -E to have more)\n"
        "    -h                 print this text and exit\n"
        "\n",
        stdout);
}



struct cmdline_opts {
    char *config_file;
    int verbosity;
    int log_stderr;
    int log_stderr_only;
    int dont_fork;
    int help;
    char errmsg[80];
};

static void set_defaults(struct cmdline_opts *opts)
{
    opts->config_file = NULL;
    opts->verbosity = 1;
    opts->log_stderr = 0;
    opts->log_stderr_only = 0;
    opts->dont_fork = 0;
    opts->help = 0;
    opts->errmsg[0] = 0;
}

static void print_need_param(char c, struct cmdline_opts *opts)
{
    sprintf(opts->errmsg, "-%c needs a parameter\n", c);
}

static int
parse_cmdline(int argc, const char *const *argv, struct cmdline_opts *opts)
{
    int idx = 1;
    while(idx < argc) {
        if(argv[idx][0] == '-') {
            switch(argv[idx][1]) {
            case 'c':
                if(idx+1 >= argc || argv[idx+1][0] == '-') {
                    print_need_param('c', opts);
                    return 0;
                }
                opts->config_file = (char*)argv[idx+1];
                idx += 2;
                break;
            case 'v':
                opts->verbosity++;
                idx++;
                break;
            case 'q':
                opts->verbosity = 0;
                idx++;
                break;
            case 'e':
                opts->log_stderr = 1;
                idx++;
                break;
            case 'E':
                opts->log_stderr_only = 1;
                idx++;
                break;
            case 'f':
                opts->dont_fork = 1;
                idx++;
                break;
            case 'h':
                opts->help = 1;
                idx++;
                break;
            default:
                sprintf(opts->errmsg, "unknown option '-%c'\n", argv[idx][1]);
                return 0;
            }
        } else {   /* we don't accept freestanding parameters */
            sprintf(opts->errmsg,
                    "option '%.20s' not recognized\n", argv[idx]);
            return 0;
        }
    }
    return 1;
}

static void settle_config_path(struct cmdline_opts *opts)
{
    settle_localpath(&opts->config_file, ".fedanet/serv.conf");
}

static void
process_cmdline(int argc, const char *const *argv, struct cmdline_opts *cmdl)
{
    int res;

    set_defaults(cmdl);
    res = parse_cmdline(argc, argv, cmdl);
    if(!res) {
        servlog_message(srvl_alert, "FATAL: wrong args: %s", cmdl->errmsg);
        exit(1);
    }
    if(cmdl->help) {
        do_help();
        exit(0);
    }

    if(cmdl->dont_fork && !cmdl->log_stderr_only)
        cmdl->log_stderr = 1;

    if(cmdl->log_stderr || cmdl->log_stderr_only) {
        static const char verbosity[] =
            { srvl_alert, srvl_normal, srvl_info, srvl_debug, srvl_debug2 };
        if(cmdl->verbosity < 0)
            cmdl->verbosity = 0;
        else
        if(cmdl->verbosity >= sizeof(verbosity))
            cmdl->verbosity = sizeof(verbosity) - 1;
        setup_stderr_log(verbosity[cmdl->verbosity] | srvl_private);
        if(cmdl->log_stderr_only)
            setup_syslog(srvl_disable, NULL, 0);
    }

    settle_config_path(cmdl);
}

static void logging_setup(struct server_conf_info *conf, int skip_stderr)
{
    int lev;

    lev = conf->log_syslog_level | (conf->log_syslog_priv ? srvl_private : 0);
    setup_syslog(lev, conf->log_syslog_ident, conf->log_syslog_facility);

    if(conf->log_file_level != srvl_disable) {
        lev = conf->log_file_level | (conf->log_file_priv ? srvl_private : 0);
        setup_file_log(lev, conf->log_file_name);
    }

    if(!skip_stderr) {
        lev = conf->log_stderr_level | (conf->log_stderr_priv?srvl_private:0);
        setup_stderr_log(lev);
    }
}

static void daemonize()
{
    int nf, pid;
    chdir("/");
    nf = open("/dev/null", O_RDWR);
    if(nf != -1) {
        dup2(nf, 0);
        dup2(nf, 1);
        if(isatty(2))
            dup2(nf, 2);
        close(nf);
    } else {
        servlog_perror(srvl_alert, "daemonize", "/dev/null");
    }
    pid = fork();
    if(pid > 0)
        exit(0);
    setsid();
    pid = fork();
    if(pid > 0)
        exit(0);
}

static void
make_compiler_happy(struct feda_signal_target *s)
{
    /* nothing to do here, this is just to suppress these warnings */
}

int main(int argc, const char * const *argv)
{
    struct cmdline_opts cmdl;
    struct sue_event_selector selector;
    struct server_conf_info *conf_info = NULL;
    struct feda_udp_receiver *feda_rx = NULL;
    struct feda_tunnel_gate *feda_gate = NULL;
    struct feda_destination_map *destmap = NULL;
    struct feda_proxy_set *proxy_set = NULL;
    struct feda_signal_target *sigtarget = NULL;
    struct feda_console *ctlsock = NULL;
    int res;

        /* we set up default logging, just in case the failure occurs
           before we can figure out the desired logging configuration
         */
    setup_stderr_log(srvl_debug | srvl_private);
    setup_syslog(srvl_debug | srvl_private, argv[0], srvl_facil_user);

    process_cmdline(argc, argv, &cmdl);

    conf_info = make_servconf();
    res = read_config(cmdl.config_file, conf_info);
    if(!res) {
        servlog_message(srvl_alert, "problems with config file [%s], exiting",
                                    cmdl.config_file);
        return 1;
    }

    if(!cmdl.log_stderr_only)
        logging_setup(conf_info, cmdl.log_stderr);

    /* if log_stderr_only is set, logging is already fully configured;
       if log_stderr is set, but log_stderr_only isn't, the stderr
       channel is already set as it should be, but the other two
       aren't yet, so we tell the function to skip the stderr channel
     */

    servlog_message(srvl_alert, "Starting FEDAnet server vers. "
                    FEDA_VERSION " (compiled " __DATE__ ")");
    servlog_message(srvl_debug, "processed config file %s", cmdl.config_file);

    settle_keydir(conf_info);

    dump_configuration_to_log(conf_info, srvl_debug);

    if(!cmdl.dont_fork) {
        daemonize();
        servlog_message(srvl_debug, "daemonized, pid %d", getpid());
    }

    sue_alloc_init_default();
    sue_sel_init(&selector);

    feda_rx = make_udp_receiver(&selector, conf_info);
    if(!feda_rx) {
        servlog_message(srvl_alert, "udp receiver construction failed");
        return 1;
    }
 
    res = launch_udp_receiver(feda_rx);
    if(!res) {
        servlog_message(srvl_alert,
                        "failed to bring up the udp receiver, exiting");
        return 1;
    }

    if(conf_info->forwarding) {
        destmap = make_destination_map();
        if(conf_info->tun_iface) {
            feda_gate = make_tunnel_gate(&selector, feda_rx, destmap,
                                         conf_info->tun_iface);
            if(!feda_gate)
                servlog_message(srvl_alert, "failed to set up the tun iface");
        }
        udp_receiver_set_tun(feda_rx, feda_gate, destmap);
    }

    sigtarget = make_signal_target(&selector, feda_rx);

    if(conf_info->control_socket)
        ctlsock = launch_control_console(&selector, conf_info, feda_rx);

    proxy_set = make_proxy_set(&selector, feda_rx, conf_info);
    if(proxy_set)
        udp_receiver_set_pxyset(feda_rx, proxy_set);


    make_compiler_happy(sigtarget);

    servlog_message(srvl_debug, "entering the main loop...");
    res = sue_sel_go(&selector);
    if(res == -1)
        servlog_message(srvl_normal, "the main loop reported problems");
    servlog_message(srvl_debug, "main loop exited");

        /* this is mainly to remove the stale socket file */ 
    if(ctlsock)
        destroy_control_console(ctlsock);

    return 0;
}

