#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>


#include "cryptodf.h"
#include "keyutils.h"
#include "message.h"
#include "fk_help.h"
#include "fk_args.h"
#include "fk_mastr.h"
#include "fk_point.h"
#include "fk_mesg.h"



struct cmdline_opts {
    int verbosity;
    int help;
    struct fedakeys_args args;
};

static void set_defaults(struct cmdline_opts *opts)
{
    opts->verbosity = 0;
    opts->help = 0;
    opts->args.dir = NULL;
    opts->args.out_file = NULL;
    opts->args.timestamp = -1;
    opts->args.may_take_hash = 1;
    opts->args.cmdargc = 0;
    opts->args.cmdargs = NULL;
}

static void print_need_param(char c)
{
    fprintf(stderr, "-%c needs a parameter; try -h for help\n", c);
}

static int
parse_cmdline(int argc, const char *const *argv, struct cmdline_opts *opts)
{
    int idx = 1;
    char *err;
    while(idx < argc) {
        if(argv[idx][0] == '-') {
            switch(argv[idx][1]) {
            case 'c':
                if(idx+1 >= argc || argv[idx+1][0] == '-') {
                    print_need_param('c');
                    return 0;
                }
                opts->args.dir = argv[idx+1];
                idx += 2;
                break;
            case 'o':
                if(idx+1 >= argc || argv[idx+1][0] == '-') {
                    print_need_param('o');
                    return 0;
                }
                opts->args.out_file = argv[idx+1];
                idx += 2;
                break;
            case 't':
                if(idx+1 >= argc || argv[idx+1][0] == '-') {
                    print_need_param('t');
                    return 0;
                }
                opts->args.timestamp = strtol(argv[idx+1], &err, 0);
                if(*err) {
                    fprintf(stderr, "invalid timemark %s\n", argv[idx+1]);
                    return 0;
                }
                idx += 2;
                break;
            case 'p':
                opts->args.may_take_hash = 0;
                idx++;
                break;
            case 'v':
                opts->verbosity++;
                idx++;
                break;
            case 'q':
                opts->verbosity = -1;
                idx++;
                break;
            case 'h':
                opts->help = 1;
                idx++;
                break;
            default:
                fprintf(stderr, "unknown option ``-%c''\n", argv[idx][1]);
                return 0;
            }
        } else {  /* no ``-''; must be the command, time to bail out */
            opts->args.cmdargs = argv + idx;
            opts->args.cmdargc = argc - idx;
            return 1;
        }
    }
    /* no command; this is okay if either there are no args at all,
       or -h is given
     */
    if(argc == 1 || opts->help) {
        opts->help = 1;
        return 1;
    }
    fprintf(stderr, "no command specified, aborting; try -h for help\n");
    return 0;
}

static void settle_workdir(struct cmdline_opts *opts)
{
    static const char localpath[] = ".fedanet/keys";
    const char *home;
    char *dir;
    int homelen;

    if(opts->args.dir)
        return;

    home = getenv("HOME");
    if(!home || !*home) {  /* rare case; fallback to the current directory */
        opts->args.dir = localpath;
        return;
    }
    homelen = strlen(home);
    dir = malloc(homelen + 1 + sizeof(localpath));
        /* please note this will never be free'd; but this is only done
           once per run, and the memory remains needed until the end of
           the program, so we don't care
         */
    strcpy(dir, home);
    dir[homelen] = '/';
    strcpy(dir + homelen + 1, localpath);
    opts->args.dir = dir;
}

int main(int argc, const char * const *argv)
{
    struct cmdline_opts opts;
    const char *cmd;
    int r;

    set_defaults(&opts);
    r = parse_cmdline(argc, argv, &opts);
    if(!r) {
        help(stderr, "");
        return 1;
    }
    if(opts.help) {
        const char * const *ca = opts.args.cmdargs;
        help(stdout, ca && *ca ? *ca : NULL);
        return 0;
    }
        /* the ``no command'' situation is already handled */
    cmd = *opts.args.cmdargs;

    message_set_verbosity(opts.verbosity);
    if(opts.args.timestamp == -1) {
        opts.args.timestamp = time(NULL) / 60;
    } else
    if(opts.args.timestamp < 20000000) {
        message(mlv_normal, "WARNING: time too far in the past\n");
    } else
    if(opts.args.timestamp > 100000000) {
        message(mlv_normal, "WARNING: time too far in the future\n");
    }
    settle_workdir(&opts);
    message(mlv_info, "NOTICE: workdir [%s]\n", opts.args.dir);


#define RUNCMD(name)                      \
    if(0 == strcmp(cmd, #name))           \
        return main_##name(&opts.args);

    RUNCMD(status)

    RUNCMD(mverify)
    RUNCMD(mdeploy)
    RUNCMD(mzero)
    RUNCMD(mpoint)

    RUNCMD(pverify)
    RUNCMD(deploy)
    RUNCMD(zcrpoint)
    RUNCMD(zsign)
    RUNCMD(key2pub)
    RUNCMD(keypair)
    RUNCMD(kexgen)

    RUNCMD(nodecert)
    RUNCMD(nverify)
    RUNCMD(nodeimport)
    RUNCMD(export)
    RUNCMD(fverify)
    RUNCMD(import)
    RUNCMD(sign)
    RUNCMD(signverify)
    RUNCMD(lock)
    RUNCMD(anonlock)
    RUNCMD(unlock)

    RUNCMD(base32)
    RUNCMD(blake2sum)

    message(mlv_alert, "command \"%s\" unknown, try -h\n", cmd);
    return 0;
}
