#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#ifndef FEDA_NG_SCHED
#define FEDA_NG_SCHED 1
#endif

#if FEDA_NG_SCHED == 1
#define __USE_GNU   /* for SCHED_IDLE to be available */
#include <sched.h>
#endif

#include <monocypher/monocypher.h>

#include "cryptodf.h"
#include "keyutils.h"
#include "xyespwr.h"
#include "fedapref.h"
#include "txtr_out.h"
#include "_version.h"


enum {
    dfl_min_len_to_save = 12,
    dot_display_freq = 32,
    challenge_table_recsize = public_key_size + yespower_hash_size,
    dfl_ctable_recnum = 1048576
};


static void save_the_found(int preflen,
    unsigned char *secretkey, unsigned char *pubkey, unsigned char *hash)
{
    int r, idx, id_idx, i;
    char pt[64];
    FILE *f;

    sprintf(pt, "%d", preflen);
    r = mkdir(pt, 0700);
    if(r == -1 && errno != EEXIST) {
        perror(pt);
        return;
    }
    idx = strlen(pt);
    pt[idx] = '/';
    idx++;
    pt[idx] = 0;
    id_idx = idx;
    for(i = public_key_size - node_id_size; i < public_key_size; i++) {
        sprintf(pt+idx, "%02x", pubkey[i]);
        idx += 2;
    }
    f = fopen(pt, "w");
    if(!f) {
        perror(pt);
        return;
    }
    put_string_field(f, "id", pt+id_idx);
    put_int_field(f, "preflen", preflen);
    put_hex_field(f, "secret", secretkey, secret_key_size);
    put_hex_field(f, "public", pubkey, public_key_size);
    put_hex_field(f, "yeshash", hash, yespower_hash_size);
    fclose(f);
}

#if FEDA_NG_SCHED == 1
static void set_nice()
{
    int r;
    struct sched_param sp;
    sp.sched_priority = 0;
    r = sched_setscheduler(0, SCHED_IDLE, &sp);
    if(r == -1)
        perror("warning: sched_setscheduler");
}
#endif

static int keychar(int pln)
{
    return pln < 10 ? '0' + pln : 'A' - 10 + pln;
}


struct challenge_table {
    int recnum;
    unsigned char *mapping;
};


static unsigned long long pack_index(const unsigned char *data)
{
    int i;
    unsigned long long res = 0;
    for(i = 0; i < 8; i++) {
        res <<= 8;
        res |= data[7 - i];
    }
    return res;
}

static void save_challenge(struct challenge_table *tbl,
                           unsigned char *challenge,
                           unsigned char *hash)
{
    unsigned long long index;
    unsigned char *start;

    index = pack_index(challenge);
    index %= tbl->recnum;
    start = tbl->mapping + index * challenge_table_recsize;

    if(!all_zeroes(start, challenge_table_recsize))
        return;   /* it's filled already */

    memcpy(start + public_key_size, hash, yespower_hash_size);
    memcpy(start + 8, challenge + 8, public_key_size - 8);
    memcpy(start, challenge, 8);
        /* we copy the index part last to minimize the chance that
           the program gets interrupted during the copying so the
           table will contain an invalid record; even if the interruption
           happens, we'll be able to detect the incomplete record with
           the ``feda-ct poscheck'' command
         */
}

static int setup_challenge_table(struct challenge_table *tbl,
                                 const char *fname, int recs)
{
    int fd;
    long long size;

    tbl->recnum = -1;
    tbl->mapping = NULL;

    fd = open(fname, O_RDWR | O_CREAT, 0600);
    if(fd == -1) {
        perror(fname);
        return 0;
    }

    size = lseek(fd, 0, SEEK_END);
    if(size == -1) {
        perror("lseek");
        close(fd);
        return 0;
    }

    if(size < challenge_table_recsize) {
        tbl->recnum = recs > 0 ? recs : dfl_ctable_recnum;
        size = (long long)tbl->recnum * challenge_table_recsize;
        int res = ftruncate(fd, size);
        if(res == -1) {
            perror("ftruncate");
            close(fd);
            return 0;
        }
    } else {
        tbl->recnum = size / challenge_table_recsize;
    }

    tbl->mapping = mmap(NULL, size, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
    if(tbl->mapping == MAP_FAILED) {
        perror("mmap");
        close(fd);
        return 0;
    }

    return 1;
}

static int perform_generation(int min_len, struct challenge_table *tbl)
{
    long c;
    int res, pln;
    unsigned char buf[seed_size];
    umask(077);
#if FEDA_NG_SCHED == 1
    set_nice();
#endif
    res = get_random(buf, sizeof(buf));
    if(!res) {
        fprintf(stderr, "couldn't generate random, aborting\n");
        return 1;
    }
    for(c = 1;; c++) {
        unsigned char seed[seed_size];
        unsigned char secret_key[secret_key_size];
        unsigned char public_key[public_key_size];
        unsigned char yespower_hash[yespower_hash_size];
        memcpy(seed, buf, sizeof(buf));
        crypto_eddsa_key_pair(secret_key, public_key, seed);
        res = take_yespower_hash(yespower_hash, public_key);
        if(!res) {
            fprintf(stderr, "couldn't take yespower hash, aborting\n");
            return 1;
        }
        pln = feda_prefix_len(yespower_hash, sizeof(yespower_hash));
        if(pln >= min_len) {
            save_the_found(pln, secret_key, public_key, yespower_hash);
            crypto_wipe(secret_key, sizeof(secret_key));
            crypto_wipe(buf, sizeof(buf));
            putchar(keychar(pln));
            fflush(stdout);
            res = get_random(buf, sizeof(buf));
            if(!res) {
                fprintf(stderr, "couldn't generate random, aborting\n");
                return 1;
            }
        } else {
            if(tbl)
                save_challenge(tbl, public_key, yespower_hash);
            if(!(c % dot_display_freq)) {
                putchar('.');
                fflush(stdout);
            }
            increment_buf(buf, sizeof(buf));
        }
    }
    return 0;  /* never reached, actually */
}

static void help()
{
    fputs(
        "feda-ng program: node master key generation\n"
        "vers. " FEDA_VERSION " (compiled " __DATE__ ")\n"
        "Copyright (c) Andrey Vikt. Stolyarov, 2024\n"
        "\n"
        "Usage: feda-ng [<options>]\n"
        "Options are:\n"
        "    -c <directory>      change to this dir before start\n"
        "    -m <len>            save keys of this rank and higher\n"
        "                        (default is 12)\n"
        "    -t <table_file>     save challenges to this challenge table\n"
        "    -r <records>        let the challenge table contain this many\n"
        "                        records; each record is 64 bytes long, the\n"
        "                        default (for new files) is 1048576 records,\n"
        "                        for existing files this option is ignored\n"
        "    -h                  display this help and exit\n",
        stdout);
}

struct cmdline_opts {
    int min_len;
    const char *dir;
    const char *ct_fname;
    int ct_recs;
};

static void init_cmdline_opts(struct cmdline_opts *p)
{
    memset(p, 0, sizeof(*p));
    p->min_len = dfl_min_len_to_save;
}

static int opt_needs_arg(int optc)
{
    return optc == 'c' || optc == 'm' || optc == 't' || optc == 'r';
}

static int
parse_cmdline(int argc, const char *const *argv, struct cmdline_opts *opts)
{
    int idx = 1;
    while(idx < argc) {
        if(argv[idx][0] == '-') {
            char optc = argv[idx][1];
            if(opt_needs_arg(optc) && (idx+1>=argc || argv[idx+1][0]=='-')) {
                fprintf(stderr, "-%c needs argument; try -h for help\n", optc);
                return 0;
            }
            switch(optc) {
            case 'c':
                opts->dir = argv[idx+1];
                idx += 2;
                break;
            case 'm': {
                char *err;
                int n = strtol(argv[idx+1], &err, 10);
                if(*argv[idx+1] != '\0' && *err == '\0' && n > 1 && n < 25) {
                    opts->min_len = n;
                } else {
                    fprintf(stderr, "Invalid minimum rank value\n");
                    return 0;
                }
                idx += 2;
                break;
            }
            case 't':
                opts->ct_fname = argv[idx+1];
                idx += 2;
                break;
            case 'r': {
                char *err;
                int n = strtol(argv[idx+1], &err, 10);
                if(*argv[idx+1] != 0 && *err == 0 && n > 1 && n < 1<<30) {
                    opts->ct_recs = n;
                } else {
                    fprintf(stderr, "Invalid challenge table record number\n");
                    return 0;
                }
                idx += 2;
                break;
            }
            case 'h':
                help();
                exit(0);
                break;
            default:
                fprintf(stderr, "unknown option ``-%c''\n", argv[idx][1]);
                return 0;
            }
        } else {  /* no ``-''; we don't know how to handle this */
            fprintf(stderr, "stray parameter ``%s''\n", argv[idx]);
            return 0;
        }
    }
    return 1;
}

int main(int argc, const char * const *argv)
{
    struct cmdline_opts opts;
    struct challenge_table ct;
    int res;

    init_cmdline_opts(&opts);
    res = parse_cmdline(argc, argv, &opts);
    if(!res)
        return 1;

    if(opts.dir) {
        res = chdir(opts.dir);
        if(res == -1) {
            perror(opts.dir);
            return 1;
        }
    }

    if(!opts.ct_fname)  /* simpler case */
        return perform_generation(opts.min_len, NULL);

    res = setup_challenge_table(&ct, opts.ct_fname, opts.ct_recs);
    if(!res)
        return 1;

    return perform_generation(opts.min_len, &ct);
}
