#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <linux/if.h>
#include <linux/if_tun.h>
#include <arpa/inet.h>

#include "ip6struc.h"

static int get_interface(char ifname[IFNAMSIZ], const char *clonedev)
{
    struct ifreq ifr;
    int fd, err;

    memset(&ifr, 0, sizeof(ifr));
    ifr.ifr_flags = IFF_TUN /* | IFF_NO_PI */;
    if(ifname && *ifname)
        strcpy(ifr.ifr_name, ifname);
    else
        ifr.ifr_name[0] = 0;

    fd = open(clonedev, O_RDWR);
    if(fd == -1) {
        perror(clonedev);
        return -1;
    }

    err = ioctl(fd, TUNSETIFF, &ifr);
    if(err == -1) {
        perror("ioctl: TUNSETIFF");
        close(fd);
        return -1;
    }

    strcpy(ifname, ifr.ifr_name);

    return fd;
}

static int set_persistent(int tun_fd, int pers)
{
    int r = ioctl(tun_fd, TUNSETPERSIST, pers);
    if(r == -1)
        perror("ioctl: TUNSETPERSIST");
    return r;
}

static int set_uid_gid(int tun_fd, int uid, int gid)
{
    int r;
    if(uid >= 0) {
        r = ioctl(tun_fd, TUNSETOWNER, uid);
        if(r == -1) {
            perror("ioctl: TUNSETOWNER");
            return -1;
        }
    }
    if(gid >= 0) {
        r = ioctl(tun_fd, TUNSETGROUP, gid);
        if(r == -1) {
            perror("ioctl: TUNSETGROUP");
            return -1;
        }
    }
    return 0;
}

static void dump_hex(const char *p, int size)
{
    int i;

    for(i = 0; i < size; i++) {
        printf("%02x ", (unsigned char) (p[i]));
        if(i % 16 == 15)
            printf("\n");
        else
        if(i % 8 == 7)
            printf("  ");
    }
    if(i % 16 != 15)
        printf("\n");
    printf("\n");
}

static void parse_packet(const char *p, int size)
{
    struct ipv6_data data;
    int code;
    char addrbuf[48];

    code = parse_ipv6_packet(&data, p, size, 1);
    if(code != 0) {
        const char * const * diags;
        int i;

        diags = parse6_get_diags(code);
        for(i = 0; diags[i]; i++)
            printf("  %% %s\n", diags[i]);

        if(code & parse6_no_info) {
            printf("couldn't dissect the data\n");
            return;
        }
    }

    printf("[PI: flags %04x proto %04x]\n", data.pi_flags, data.pi_proto);
    printf("version %d   traffic class %d   next_header %d   hop_limit %d\n",
           data.version, data.traffic_class, data.next_header, data.hop_limit);
    printf("flow_label %05x   payload_length %d\n", 
           data.flow_label, data.payload_length);
    printf("SRC: %s\n",
        inet_ntop(AF_INET6, data.source_addr, addrbuf, sizeof(addrbuf)));
    printf("DST: %s\n",
        inet_ntop(AF_INET6, data.dest_addr, addrbuf, sizeof(addrbuf)));
}


static void dump_traffic(int tun_fd)
{
    char buf[2048];
    int res;
    for(;;) {
        res = read(tun_fd, buf, sizeof(buf));
        if(res <= 0)
            return;
        printf("* %d bytes\n", res);
        dump_hex(buf, res);
        parse_packet(buf, res);
    }
}

/* ----------------------------------------------------------------- */
/* command line parsing; please note getopt.h is not used here
 * deliberately, and that's for a reason
 */

static void help(FILE *s)
{
    fprintf(s,
        "feda-if <options> [<iface_name>]\n"
        "  where options are:\n"
        "    -p            make the interface persistent\n"
        "    -r            remove the interface and exit\n"
        "    -u <uid>      set uid for the interface\n"
        "    -g <gid>      set gid for the interface\n"
        "    -m            monitor the interface (the default in case\n"
        "                  neither -p nor -r given; incompatible with -r)\n"
        "    -l <path>     the cLone dev path (default: /dev/net/tun)\n"
        "\n"
        "    -h            print this help and exit\n"
    );
}

struct cmdline_opts {
    int remove, mkpers, monitor;
    int uid, gid;
    char ifname[IFNAMSIZ];
    const char *clonedev;
};

static void set_defaults(struct cmdline_opts *co)
{
    co->remove = 0;
    co->mkpers = 0;
    co->monitor = 0;
    co->uid = -1;
    co->gid = -1;
    co->ifname[0] = 0;
    co->clonedev = NULL;
}

static void pcmd_setif(struct cmdline_opts *co, const char *arg)
{
    if(co->ifname[0]) {  /* already set */
        fprintf(stderr, "duplicate interface name is not allowed\n");
        exit(1);
    }
    if(strlen(arg) > IFNAMSIZ-1) {
        fprintf(stderr, "interface name [%s] too long\n", arg);
        exit(1);
    }
    strcpy(co->ifname, arg);
}

static void pcmd_getint(int *dest, const char *arg, const char *parname)
{
    long n;
    char *err;
    n = strtol(arg, &err, 10);
    if(!*arg || *err) {
        fprintf(stderr, "invalid %s [%s]\n", parname, arg);
        exit(1);
    }
    *dest = n;
}

static int opt_needs_arg(int c)
{
    return c == 'u' || c == 'g' || c == 'l';
}

static void
parse_cmdline(int argc, const char *const *argv, struct cmdline_opts *co)
{
    int cur = 1;
    while(cur < argc) {
        if(!argv[cur])   /* a bit of paranoia */
            break;
        if(argv[cur][0] != '-') {   /* freestanding parameter */
            pcmd_setif(co, argv[cur]);
            cur++;
            continue;
        }
        /* okay, it starts with the '-', so it's a flag */
        if(!argv[cur][1] || argv[cur][2]) {
            fprintf(stderr, "invalid option [%s]\n", argv[cur]);
            exit(1);
        }
        /* exactly one char after the dash, right as desired */
        if(opt_needs_arg(argv[cur][1]) &&
            (cur+1 >= argc || !argv[cur+1] || argv[cur+1][0] == '-'))
        {
            fprintf(stderr, "option [%s] needs argument\n", argv[cur]);
            exit(1);
        }
        switch(argv[cur][1]) {
        case 'h':
            help(stdout);
            exit(0);
        case 'r':
            co->remove = 1;
            cur++;
            break;
        case 'p':
            co->mkpers = 1;
            cur++;
            break;
        case 'm':
            co->monitor = 1;
            cur++;
            break;
        case 'u':
            pcmd_getint(&co->uid, argv[cur+1], "uid");
            cur += 2;
            break;
        case 'g':
            pcmd_getint(&co->gid, argv[cur+1], "gid");
            cur += 2;
            break;
        case 'l':
            if(co->clonedev) {
                fprintf(stderr, "duplicate [-l] is not allowed\n");
                exit(1);
            }
            co->clonedev = argv[cur+1];
            cur += 2;
            break;
        default:
            fprintf(stderr, "option unknown [%s]\n\n", argv[cur]);
            help(stderr);
            exit(1);
        }
    }
}

/* ----------------------------------------------------------------- */

int main(int argc, const char *const *argv)
{
    int tun_fd;
    struct cmdline_opts co;

    set_defaults(&co);
    parse_cmdline(argc, argv, &co);

    if(!co.clonedev)
        co.clonedev = "/dev/net/tun";

    tun_fd = get_interface(co.ifname, co.clonedev);
    if(tun_fd == -1)
        return 1;

    printf("bound to [%s]\n", co.ifname);

    if(co.remove) {
        int res = set_persistent(tun_fd, 0);
        if(res == -1) {
            fprintf(stderr, "failed to remove the interface\n");
            return 1;
        }
        printf("removed\n");
        return 0;
    }

    if(co.mkpers) {
        int res = set_persistent(tun_fd, 1);
        if(res == -1) {
            fprintf(stderr, "failed to make the interface persistent\n");
            return 1;
        }
    }

    if(co.uid >= 0 || co.gid >= 0) {
        int res = set_uid_gid(tun_fd, co.uid, co.gid);
        if(res == -1) {
            fprintf(stderr, "failed to set ownership\n");
            return 1;
        }
    }

    if(co.monitor || (!co.remove && !co.mkpers))
        dump_traffic(tun_fd);

    return 0;
}
